#include "request.h"

#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/adaptor/transformed.hpp>

#include <format>

namespace NNwSmtp::NSO {

namespace {

std::string MakeHeader(const std::string& header, const std::string& value) {
    return std::format("{}: {}\r\n", header, value);
}

} // namespace anonymous

TRequest::TRequest(TEnvelope envelope, std::string rawEmail, bool dryRun)
    : Envelope(std::move(envelope))
    , RawEmail(std::move(rawEmail))
    , DryRun(dryRun)
{}

std::string BuildSoHeaders(const TBuildSoHttpRequestParams& args, const std::string& queueId) {
    std::string soHeaders;

    soHeaders = MakeHeader("X-Yandex-QueueID", queueId);
    soHeaders += MakeHeader("X-Yandex-So-Front", args.SoClusterName);
    if (args.Spf.has_value()) {
        soHeaders += MakeHeader("Received-SPF", args.Spf.value());
    }
    if (args.Dmarc.has_value()) {
        soHeaders += MakeHeader("X-DMARC", args.Dmarc.value());
    }
    if (args.AuthResults.has_value()) {
        soHeaders += MakeHeader("X-Yandex-Authentication-Results", args.AuthResults.value());
    }
    if (args.MessageId.has_value()) {
        soHeaders += MakeHeader("Message-Id", args.MessageId.value());
    }
    if (args.Mailish.has_value()) {
        soHeaders += MakeHeader("X-Yandex-Mailish", args.Mailish.value());
    }

    auto filteredHeaders = args.Headers.FilterHeaders([&internalHeaders = args.InternalHeaders](const auto& header) {
        return !internalHeaders.contains(header);
    });

    soHeaders += boost::join(
        filteredHeaders | boost::adaptors::transformed([](const auto& value) {
            auto header = value.GetWhole();
            return std::string {header.begin(), header.end()};
        }),
        "\r\n"
    );
    soHeaders += "\r\n";
    return soHeaders;
}

TRequestPtr BuildSoHttpRequest(TBuildSoHttpRequestParams args) {
    const auto queueId = std::format("{}-{}", args.SessionId, args.EnvelopeId);

    TEnvelope soEnvelope = {
        .ConnectInfo = {
            .Host = args.RemoteHost,
            .Ip = IpToBytes(args.RemoteIp),
            .Domain = args.HeloHost.has_value() && IsASCII(args.HeloHost.value()) ? args.HeloHost.value() : args.RemoteHost,
            .SessionId = queueId,
        },
        .MailFrom = std::move(args.MailFrom),
        .Rcpts = std::move(args.Rcpts),
        .Timestamp = std::move(args.Timestamp),
    };

    if (args.EmailType.has_value()) {
        soEnvelope.EmailType = std::move(args.EmailType).value();
    }

    auto content = BuildSoHeaders(args, queueId)
                 + "\r\n"
                 + std::string {args.MessageBody.begin(), args.MessageBody.end()};

    return std::make_shared<TRequest>(
        std::move(soEnvelope),
        std::move(content),
        args.IsDryRun
    );
}

} // namespace NNwSmtp::NSO
