#include "aspf.h"

#include <boost/thread.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/format.hpp>
#include <boost/unordered_map.hpp>
#include <boost/bind/protect.hpp>
#include <boost/optional.hpp>
#include <nwsmtp/resolver.h>
#include "utils.h"


namespace impl
{

extern "C"
{
#include <contrib/libs/libspf2/src/include/spf.h>
#include <contrib/libs/libspf2/src/include/spf_utils.h>
}

using namespace std;
using NNwSmtp::resolver_t;

typedef pair<string, ns_type> lookup_key;
struct dns_data
{
    typedef boost::unordered_map<lookup_key, boost::shared_ptr<SPF_dns_rr_t>
    > hash_t;
    hash_t hash;
    spf_logger_t logger;
    boost::mutex mutex;
};

inline SPF_dns_rr_t* clone_SPF_dns_rr(boost::shared_ptr<SPF_dns_rr_t> rr)
{
    SPF_dns_rr_t* new_rr = 0;
    SPF_dns_rr_dup(&new_rr, rr.get());
    return new_rr;
}

inline dns_data* get_dns_data(SPF_dns_server_t* d)
{   return static_cast<dns_data*>(d->hook);    }

std::string canonical_host(const std::string& host)
{
    std::size_t sz = host.size();
    if (sz > 0 && host[sz-1] == '.')
        return std::string(host.begin(), host.end()-1);
    return host;
}

inline void insert_dns_data(SPF_dns_server_t* d,
    boost::shared_ptr<SPF_dns_rr_t> rr)
{
    dns_data* dd =  get_dns_data(d);
    boost::mutex::scoped_lock lock(dd->mutex);
    dd->hash.insert(dns_data::hash_t::value_type(
        lookup_key(canonical_host(rr->domain), rr->rr_type), rr));
}

inline bool has_dns_data(SPF_dns_server_t* d, const string& domain, ns_type t)
{
    dns_data* dd =  get_dns_data(d);
    boost::mutex::scoped_lock lock(dd->mutex);
    return dd->hash.find(lookup_key(canonical_host(domain), t))
        != dd->hash.end();
}

extern "C" SPF_dns_rr_t* ydns_resolv_lookup(SPF_dns_server_t* d,
    const char *domain, ns_type type, int)
{
    dns_data* dd = get_dns_data(d);
    boost::mutex::scoped_lock lock(dd->mutex);
    dns_data::hash_t::iterator it = dd->hash.find(
        lookup_key(canonical_host(domain), type));

#ifdef ASPF_IMPL_DEBUG
    if (it != dd->hash.end())
    {
        std::ostringstream os;
        os << "successfully queried " << domain << ", type=" << type;
        dd->logger(os.str());
    }
#endif

    if (it != dd->hash.end())
        return clone_SPF_dns_rr(it->second);

    if (dd->logger)
    {
        std::ostringstream os;
        os << "failed to resolve " << domain << ", type=" << type;
        dd->logger(os.str());
    }

    return SPF_dns_rr_new_init(d, domain, type, 0, 0);
}

extern "C" void free_spf_dns_resolver(SPF_dns_server_t* d)
{
    delete static_cast<dns_data*>(d->hook);
    delete d;
}

SPF_dns_server_t* create_spf_dns_resolver(SPF_dns_server_t* layer_below,
    dns_data* hook, int debug)
{
    SPF_dns_server_t* spf_dns = new SPF_dns_server_t;
    if (!spf_dns)
        return spf_dns;

    spf_dns->destroy     = NULL;
    spf_dns->lookup      = ydns_resolv_lookup;
    spf_dns->get_spf     = NULL;
    spf_dns->get_exp     = NULL;
    spf_dns->add_cache   = NULL;
    spf_dns->layer_below = layer_below;
    spf_dns->name        = "yresolv";
    spf_dns->debug       = debug;
    spf_dns->hook        = hook;

    return spf_dns;
}

boost::shared_ptr<SPF_dns_rr_t> create_spf_dns_txt_rr(SPF_dns_server_t* d,
    const string& domain, const string& text)
{
    SPF_dns_rr_t* rr = SPF_dns_rr_new_init(d, domain.c_str(), ns_t_txt, 0, 0);
    if (!rr)
        return boost::shared_ptr<SPF_dns_rr_t>();
    SPF_dns_rr_buf_realloc(rr, 0, text.size()+1);
    strcpy(rr->rr[0]->txt, text.c_str());
    rr->num_rr++;
    return boost::shared_ptr<SPF_dns_rr_t>(rr, SPF_dns_rr_free);
}

boost::shared_ptr<SPF_dns_rr_t>
create_spf_dns_ptr_rr(SPF_dns_server_t* d, const string& domain,
    resolver_t::iterator_ptr it)
{
    SPF_dns_rr_t* rr = SPF_dns_rr_new_init(d, domain.c_str(), ns_t_ptr, 0, 0);
    if (!rr)
        return boost::shared_ptr<SPF_dns_rr_t>();
    int i=0;
    for ( ; it != resolver_t::iterator_ptr() ; ++it, ++i)
    {
        std::string ptr(*it);
        SPF_dns_rr_buf_realloc(rr, i, ptr.size()+1);
        strcpy(rr->rr[i]->txt, ptr.c_str());
        rr->num_rr++;
    }
    return boost::shared_ptr<SPF_dns_rr_t>(rr, SPF_dns_rr_free);
}

boost::shared_ptr<SPF_dns_rr_t>
create_spf_dns_mx_rr(SPF_dns_server_t* d, const string& domain,
    resolver_t::iterator_mx it)
{
    SPF_dns_rr_t* rr = SPF_dns_rr_new_init(d, domain.c_str(), ns_t_mx, 0, 0);
    if (!rr)
        return boost::shared_ptr<SPF_dns_rr_t>();
    int i=0;
    for ( ; it != resolver_t::iterator_mx() ; ++it, ++i)
    {
        std::string mx(it->second);
        SPF_dns_rr_buf_realloc(rr, i, mx.size()+1);
        strcpy(rr->rr[i]->mx, mx.c_str());
        rr->num_rr++;
    }
    return boost::shared_ptr<SPF_dns_rr_t>(rr, SPF_dns_rr_free);
}

boost::shared_ptr<SPF_dns_rr_t>
create_spf_dns_a_rr(SPF_dns_server_t* d, const string& domain,
    resolver_t::iterator_a it)
{
    SPF_dns_rr_t* rr = SPF_dns_rr_new_init(d, domain.c_str(), ns_t_a, 0, 0);
    if (!rr)
        return boost::shared_ptr<SPF_dns_rr_t>();
    int i=0;
    for ( ; it != resolver_t::iterator_a() ; ++it, ++i)
    {
        SPF_dns_rr_buf_realloc(rr, i, it->size()+1);
        inet_pton( AF_INET, it->c_str(), &rr->rr[i]->a);
        rr->num_rr++;
    }
    return boost::shared_ptr<SPF_dns_rr_t>(rr, SPF_dns_rr_free);
}

boost::shared_ptr<SPF_dns_rr_t>
create_spf_dns_aaaa_rr(SPF_dns_server_t* d, const string& domain,
    resolver_t::iterator_aaaa it)
{
    SPF_dns_rr_t* rr = SPF_dns_rr_new_init(d, domain.c_str(), ns_t_aaaa, 0, 0);
    if (!rr)
        return boost::shared_ptr<SPF_dns_rr_t>();
    int i=0;
    for ( ; it != resolver_t::iterator_aaaa() ; ++it, ++i)
    {
        SPF_dns_rr_buf_realloc(rr, i, it->size()+1);
        inet_pton( AF_INET6, it->c_str(), &rr->rr[i]->aaaa);
        rr->num_rr++;
    }
    return boost::shared_ptr<SPF_dns_rr_t>(rr, SPF_dns_rr_free);
}

template <class T>
class yscoped_ptr_helper_base
{
public:
    virtual ~yscoped_ptr_helper_base() {}
    virtual void dispose(T* t) = 0;
};

template <class T, class D>
class yscoped_ptr_helper_td : public yscoped_ptr_helper_base<T>
{
    D del;
public:
    yscoped_ptr_helper_td(D d)
        : del(d)
    {}

    void dispose(T* t) override
    {
        if (t)
            del(t);
    }
};

template <class T>
class yscoped_ptr_helper_t : public yscoped_ptr_helper_base<T>
{
public:
    void dispose(T* t)
    {
        delete t;
    }
};

template<class T>
class yscoped_ptr : private boost::noncopyable
{
    T* px;
    yscoped_ptr_helper_base<T>* ph;
    typedef yscoped_ptr<T> this_type;

public:
    template <class D>
    yscoped_ptr(T* t, D d)
        : px(t), ph(new yscoped_ptr_helper_td<T, D>(d))
    {
    }

    yscoped_ptr(T* t = 0)
        : px(t), ph(new yscoped_ptr_helper_t<T>)
    {
    }

    ~yscoped_ptr()
    {
        if (px)
        {
            ph->dispose(px);
            px = 0;
        }
        delete ph;
        ph = 0;
    }

    operator bool()
    {
        return px != 0;
    }

    void reset(T* p = 0)
    {
        if (px)
            ph->dispose(px);
        px = p;
    }

    T* release()
    {
        T* tpx = px;
        px = 0;
        return tpx;
    }

    T& operator*() const
    {
        return *px;
    }

    T* operator->() const
    {
        return px;
    }

    T* get() const
    {
        return px;
    }
};

struct collect_shared_state : private boost::noncopyable
{
    collect_shared_state(boost::asio::io_service& ios, const resolver_options& resolver_options, SPF_server_t* srv_ = 0,
        SPF_dns_server_t* dns_ = 0, SPF_request_t* req_ = 0)
        : srv(srv_), dns(dns_), req(req_),
          r(ios, resolver_options),
          inprogress(0),
          done(false)
    {
    }

    ~collect_shared_state()
    {
        if (req)
            SPF_request_free(req);
        if (dns)
            free_spf_dns_resolver(dns);
        if (srv)
            SPF_server_free(srv);
    }

    SPF_server_t* srv;
    SPF_dns_server_t* dns;
    SPF_request_t* req;
    resolver_t r;
    int inprogress;
    bool done;
    boost::mutex mux;
};

struct collect_state
{
    collect_state(boost::shared_ptr<collect_shared_state> st, string dom,
        bool e=false)
        : shared_state(st), cur_dom(dom), err(e)
    {}

    typedef boost::shared_ptr<collect_shared_state> shared_state_t;
    shared_state_t shared_state;
    string cur_dom;
    bool err;
};

template <class Handle>
inline void handle_partial_collect_spf_dns_data(collect_state st, Handle handle)
{
    if (! --st.shared_state->inprogress)
        handle(st);
}

template <class Handle>
void handle_resolve_txt_exp(const boost::system::error_code& ec,
    resolver_t::iterator_txt it, collect_state st, Handle handle)
{
    if (ec == boost::asio::error::operation_aborted || st.shared_state->done)
        return;

    if (!ec)
    {
        boost::shared_ptr<SPF_dns_rr_t> spf_rr =
            create_spf_dns_txt_rr(st.shared_state->dns, st.cur_dom, *it);
        insert_dns_data(st.shared_state->dns, spf_rr);
        handle_partial_collect_spf_dns_data(st, handle);
        return;
    }
    handle_partial_collect_spf_dns_data(collect_state(st.shared_state,
        st.cur_dom, true), handle);
}

template <class Handle>
void handle_resolve_txt(const boost::system::error_code& ec,
    resolver_t::iterator_txt it, collect_state st, Handle handle)
{
    if (ec == boost::asio::error::operation_aborted || st.shared_state->done)
        return;

    while (!ec || st.shared_state->done)
    {
        SPF_errcode_t err = SPF_E_SUCCESS;
        SPF_record_t* r = 0;
        yscoped_ptr<SPF_record_t> r_guard(0, SPF_record_free);
        yscoped_ptr<SPF_response_t> spf_res(0, SPF_response_free);

        for (; it != resolver_t::iterator_txt(); ++it)
        {
            r_guard.reset(0);
            r = 0;
            spf_res.reset(SPF_response_new(st.shared_state->req));
            err = SPF_record_compile(st.shared_state->srv, spf_res.get(), &r,
                it->c_str());
            r_guard.reset(r);
            if (err == SPF_E_SUCCESS)
            {
                boost::shared_ptr<SPF_dns_rr_t> spf_rr =
                    create_spf_dns_txt_rr(st.shared_state->dns, st.cur_dom, *it);
                insert_dns_data(st.shared_state->dns, spf_rr);

                char* buf = NULL;
                size_t buf_len = 0;
                int err = SPF_record_find_mod_value(st.shared_state->srv,
                    st.shared_state->req, spf_res.get(), r, SPF_EXP_MOD_NAME,
                    &buf, &buf_len);
                if (err)
                    err = SPF_record_find_mod_value(st.shared_state->srv,
                        st.shared_state->req, spf_res.get(), r, "exp", &buf,
                        &buf_len);
                yscoped_ptr<char> buf_guard(buf, free);
                if (err == SPF_E_SUCCESS)
                {
                    collect_state nst(st.shared_state, string(buf));
                    collect_spf_dns_data_exp(nst, handle);
                }
                break; // we need exactly 1 compilable spf record
            }
        }

        if (err != SPF_E_SUCCESS)
            break;

        SPF_mech_t* mech = r->mech_first;
        SPF_data_t* data = 0;
        SPF_data_t* data_end = 0;
        char* buf = NULL;
        size_t buf_len = 0;
        yscoped_ptr<char> buf_guard(0, free);
        const char* lookup = 0;

        for (int m = 0; m < r->num_mech; m++, mech=SPF_mech_next(mech))
        {
            if (spf_res->num_dns_mech > st.shared_state->srv->max_dns_mech)
                break;
            data = SPF_mech_data(mech);
            data_end = SPF_mech_end_data(mech);
            switch (mech->mech_type)
            {
                case MECH_A:
                    if (data < data_end && data->dc.parm_type == PARM_CIDR)
                        data = SPF_data_next(data);
                    if (data == data_end)
                        lookup = st.cur_dom.c_str();
                    else
                    {
                        buf_guard.release();
                        err = SPF_record_expand_data(st.shared_state->srv,
                            st.shared_state->req, spf_res.get(),
                            data, ((char*)data_end - (char*)data),
                            &buf, &buf_len);
                        buf_guard.reset(buf);
                        lookup = buf;
                    }
                    if (err)
                        break;

                    collect_spf_dns_data_a(collect_state(st.shared_state,
                        lookup), handle);
                    break;

                case MECH_MX:
                    if (data < data_end && data->dc.parm_type == PARM_CIDR)
                        data = SPF_data_next(data);
                    if (data == data_end)
                        lookup = st.cur_dom.c_str();
                    else
                    {
                        buf_guard.release();
                        err = SPF_record_expand_data(st.shared_state->srv,
                            st.shared_state->req, spf_res.get(),
                            data, ((char*)data_end - (char*)data),
                            &buf, &buf_len);
                        buf_guard.reset(buf);
                        lookup = buf;
                    }
                    if (err)
                        break;

                    collect_spf_dns_data_mx(collect_state(st.shared_state,
                        lookup), handle);
                    break;

                case MECH_PTR:
                {
                    SPF_request_t* spf_req = st.shared_state->req;
                    if (spf_req->client_ver == AF_INET)
                    {
                        typedef boost::asio::ip::address_v4::bytes_type
                            bytes_type;
                        bytes_type ipv4;
                        memcpy(ipv4.data(), &spf_req->ipv4.s_addr, 4);
                        collect_spf_dns_data_ptr(collect_state(
                            st.shared_state,
                            rev_order_av4_str(
                                boost::asio::ip::address_v4(ipv4))),
                            handle);
                    }
                    else
                    {
                        typedef boost::asio::ip::address_v6::bytes_type
                            bytes_type;
                        bytes_type ipv6;
                        memcpy(ipv6.data(), &spf_req->ipv6.s6_addr, 16);
                        collect_spf_dns_data_ptr(collect_state(
                            st.shared_state,
                            rev_order_av6_str(
                                boost::asio::ip::address_v6(ipv6))),
                            handle);
                    }
                }
                    break;

                case MECH_INCLUDE:
                case MECH_REDIRECT:
                    buf_guard.release();
                    err = SPF_record_expand_data(st.shared_state->srv,
                        st.shared_state->req, spf_res.get(),
                        SPF_mech_data(mech), SPF_mech_data_len(mech),
                        &buf, &buf_len );
                    buf_guard.reset(buf);
                    if (err)
                        break;

                    // see if we go in circles
                    if (has_dns_data(st.shared_state->dns, buf, ns_t_txt))
                    {
                        err = SPF_E_RECURSIVE;
                        break;
                    }

                    collect_spf_dns_data_redirect(collect_state(st.shared_state,
                        buf), handle);
                    break;

                case MECH_EXISTS:
                    buf_guard.release();
                    err = SPF_record_expand_data(st.shared_state->srv,
                        st.shared_state->req, spf_res.get(),
                        SPF_mech_data(mech),SPF_mech_data_len(mech),
                        &buf, &buf_len);
                    buf_guard.reset(buf);
                    if (err)
                        break;

                    collect_spf_dns_data_a(collect_state(st.shared_state, buf),
                        handle);
                    break;

                default:
                    break;
            }
            if (err)
                break;
        }

        if (err)
            break;

        handle_partial_collect_spf_dns_data(st, handle);
        return;
    }

    handle_partial_collect_spf_dns_data(collect_state(st.shared_state,
        st.cur_dom, true), handle);
}

template <class Handle>
void handle_resolve_a(const boost::system::error_code& ec,
    resolver_t::iterator_a it, collect_state st, Handle handle)
{
    if (ec == boost::asio::error::operation_aborted || st.shared_state->done)
        return;

    if (!ec)
    {
        boost::shared_ptr<SPF_dns_rr_t> spf_rr =
            create_spf_dns_a_rr(st.shared_state->dns, st.cur_dom, it);
        insert_dns_data(st.shared_state->dns, spf_rr);
        handle_partial_collect_spf_dns_data(st, handle);
        return;
    }

    handle_partial_collect_spf_dns_data(collect_state(st.shared_state,
        st.cur_dom, true), handle);
}

template <class Handle>
void handle_resolve_aaaa(const boost::system::error_code& ec,
    resolver_t::iterator_aaaa it, collect_state st, Handle handle)
{
    if (ec == boost::asio::error::operation_aborted || st.shared_state->done)
        return;

    if (!ec)
    {
        boost::shared_ptr<SPF_dns_rr_t> spf_rr =
            create_spf_dns_aaaa_rr(st.shared_state->dns, st.cur_dom, it);
        insert_dns_data(st.shared_state->dns, spf_rr);
        handle_partial_collect_spf_dns_data(st, handle);
        return;
    }

    handle_partial_collect_spf_dns_data(collect_state(st.shared_state,
        st.cur_dom, true), handle);
}

template <class Handle>
void collect_spf_dns_data_a(collect_state st, Handle handle)
{
    st.shared_state->inprogress++;
    boost::mutex::scoped_lock lock(st.shared_state->mux);
    if (st.shared_state->req->client_ver == AF_INET)
    {
        st.shared_state->r.async_resolve_a(st.cur_dom,
            boost::bind(handle_resolve_a<Handle>,
                boost::asio::placeholders::error,
                boost::asio::placeholders::iterator,
                st,
                handle)
                                          );
    }
    else
    {
        st.shared_state->r.async_resolve_aaaa(st.cur_dom,
            boost::bind(handle_resolve_aaaa<Handle>,
                boost::asio::placeholders::error,
                boost::asio::placeholders::iterator,
                st,
                handle)
                                             );
    }
}

template <class Handle>
void handle_resolve_mx(const boost::system::error_code& ec,
    resolver_t::iterator_mx it, collect_state st, Handle handle)
{
    if (ec == boost::asio::error::operation_aborted || st.shared_state->done)
        return;

    boost::shared_ptr<SPF_dns_rr_t> spf_rr =
        create_spf_dns_mx_rr(st.shared_state->dns, st.cur_dom, it);
    insert_dns_data(st.shared_state->dns, spf_rr);

    if (!ec)
    {
        for( ; it != resolver_t::iterator_mx(); ++it)
            collect_spf_dns_data_a(collect_state(st.shared_state, it->second),
                handle);

        handle_partial_collect_spf_dns_data(st, handle);
        return;
    }

    handle_partial_collect_spf_dns_data(collect_state(st.shared_state,
        st.cur_dom, true), handle);
}


template <class Handle>
void collect_spf_dns_data_mx(collect_state st, Handle handle)
{
    st.shared_state->inprogress++;
    boost::mutex::scoped_lock lock(st.shared_state->mux);
    st.shared_state->r.async_resolve_mx(st.cur_dom,
        boost::bind(handle_resolve_mx<Handle>,
            boost::asio::placeholders::error,
            boost::asio::placeholders::iterator,
            st,
            handle)
                                       );
}

template <class Handle>
void handle_resolve_ptr(const boost::system::error_code& ec,
    resolver_t::iterator_ptr it, collect_state st, Handle handle)
{
    if (ec == boost::asio::error::operation_aborted || st.shared_state->done)
        return;

    boost::shared_ptr<SPF_dns_rr_t> spf_rr =
        create_spf_dns_ptr_rr(st.shared_state->dns, st.cur_dom, it);
    insert_dns_data(st.shared_state->dns, spf_rr);

    if (!ec)
    {
        for( ; it != resolver_t::iterator_ptr(); ++it)
            collect_spf_dns_data_a(collect_state(st.shared_state, *it),
                handle);

        handle_partial_collect_spf_dns_data(st, handle);
        return;
    }

    handle_partial_collect_spf_dns_data(collect_state(st.shared_state,
        st.cur_dom, true), handle);
}

template <class Handle>
void collect_spf_dns_data_ptr(collect_state st, Handle handle)
{
    st.shared_state->inprogress++;
    boost::mutex::scoped_lock lock(st.shared_state->mux);
    st.shared_state->r.async_resolve_ptr(st.cur_dom,
        boost::bind(handle_resolve_ptr<Handle>,
            boost::asio::placeholders::error,
            boost::asio::placeholders::iterator,
            st,
            handle)
                                        );
}

template <class Handle>
void collect_spf_dns_data_exp(collect_state st, Handle handle)
{
    st.shared_state->inprogress++;
    if (!st.cur_dom.empty())
    {
        boost::mutex::scoped_lock lock(st.shared_state->mux);
        st.shared_state->r.async_resolve_txt(
            st.cur_dom,
            boost::bind(handle_resolve_txt_exp<Handle>,
                boost::asio::placeholders::error,
                boost::asio::placeholders::iterator,
                st,
                handle)
                                            );
    }
    else
    {
        handle_partial_collect_spf_dns_data(
            collect_state(st.shared_state, st.cur_dom, true), handle);
    }
}

template <class Handle>
void collect_spf_dns_data_redirect(collect_state st, Handle handle)
{
    st.shared_state->inprogress++;
    if (!st.cur_dom.empty())
    {
        boost::mutex::scoped_lock lock(st.shared_state->mux);
        st.shared_state->r.async_resolve_txt(
            st.cur_dom,
            boost::bind(handle_resolve_txt<Handle>,
                boost::asio::placeholders::error,
                boost::asio::placeholders::iterator,
                st,
                handle)
                                            );
    }
    else
    {
        handle_partial_collect_spf_dns_data(
            collect_state(st.shared_state, st.cur_dom, true), handle);
    }
}

template <class Handle>
void continue_spf_check(collect_state st, Handle handle)
{
    SPF_response_t* res = 0;
    SPF_request_query_mailfrom(st.shared_state->req, &res);

    yscoped_ptr<SPF_response_t> spf_res(res, SPF_response_free);

    std::optional<std::string> result;
    std::optional<std::string> expl;

    if (SPF_RESULT_NONE != SPF_response_result(res))
    {
        const char* str = SPF_response_get_header_comment(res);
        if (str)
            expl = std::optional<std::string>(str);
        SPF_result_t spf = SPF_response_result(res);
        str = SPF_strresult(spf);
        if (str)
            result = std::optional<std::string>(str);
    }

    st.shared_state->done = true;
    handle( result, expl );
}

collect_state create_init_collect_state(
    boost::asio::io_service& ios, const spf_parameters& p, int debug, const resolver_options& resolver_options)
{
    collect_state::shared_state_t sst(new collect_shared_state(ios, resolver_options));
    if (!sst)
        return collect_state(sst, p.domain, true);

    dns_data* dd = new dns_data;
    dd->logger = p.logger;

    sst->dns = create_spf_dns_resolver(NULL, dd, debug);
    sst->srv = SPF_server_new_dns(sst->dns, 0);
    sst->req = SPF_request_new(sst->srv);

    return collect_state(sst, p.domain, false);
}

inline bool sanitize_spf_parameters(spf_parameters& p)
{
    string name, domain;
    if (!p.from.empty() && parse_email(p.from, name, domain))
        p.domain = domain;
    else if (p.from.empty() && p.domain.empty())
        return false;
    if (p.ip.is_unspecified())
        return false;
    return true;
}

} // namespace impl

template <class Handle>
void async_check_SPF(boost::asio::io_service& ios, const spf_parameters& pp,
    const resolver_options& resolver_options, Handle handle, int debug)
{
    spf_parameters p = pp;
    if ( !impl::sanitize_spf_parameters(p) ) // cosher input?
    {
        handle(boost::optional<std::string>(), boost::optional<std::string>());
        return;
    }

    impl::collect_state st = impl::create_init_collect_state(ios, p, debug, resolver_options);
    if (st.err)
        return;

    if (!p.from.empty())
        SPF_request_set_env_from(st.shared_state->req, p.from.c_str());
    else if (!p.domain.empty())
        SPF_request_set_helo_dom(st.shared_state->req, p.domain.c_str());

    int err = impl::SPF_E_SUCCESS;
    if (p.ip.is_v4())
    {
        typedef boost::asio::ip::address_v4::bytes_type bytes_type;
        bytes_type ipv4 = p.ip.to_v4().to_bytes();
        in_addr addr;
        memcpy(&addr.s_addr, ipv4.data(), 4);
        err = SPF_request_set_ipv4(st.shared_state->req, addr);
    }
    else
    {
        typedef boost::asio::ip::address_v6::bytes_type bytes_type;
        bytes_type ipv6 = p.ip.to_v6().to_bytes();
        in6_addr addr;
        memcpy(addr.s6_addr, ipv6.data(), 16);
        err = SPF_request_set_ipv6(st.shared_state->req, addr);
    }

    if (err != impl::SPF_E_SUCCESS)
        return;

    collect_spf_dns_data_redirect(
        st,
        boost::protect(
            boost::bind(impl::continue_spf_check<Handle>, _1, handle)
                      )
                                 );
}

struct spf_check_impl
{
    boost::shared_ptr<impl::collect_shared_state> shared_state;
};

spf_check::spf_check()
{
}

void spf_check::start(boost::asio::io_service& ios, const spf_parameters& pp,
    const resolver_options& resolver_options, callback_t handler, int debug)
{
    impl_.reset(new spf_check_impl);
    spf_parameters p = pp;
    if (!impl::sanitize_spf_parameters(p))
    {
        handler(std::optional<std::string>(), std::optional<std::string>());
        return;
    }

    impl::collect_state st = impl::create_init_collect_state(ios, p, debug, resolver_options);
    if (st.err)
    {
        handler(std::optional<std::string>(), std::optional<std::string>());
        return;
    }

    if (!p.from.empty())
        SPF_request_set_env_from(st.shared_state->req, p.from.c_str());
    else if (!p.domain.empty())
        SPF_request_set_helo_dom(st.shared_state->req, p.domain.c_str());

    int err = impl::SPF_E_SUCCESS;
    if (p.ip.is_v4())
    {
        typedef boost::asio::ip::address_v4::bytes_type bytes_type;
        bytes_type ipv4 = p.ip.to_v4().to_bytes();
        in_addr addr;
        memcpy(&addr.s_addr, ipv4.data(), 4);
        err = SPF_request_set_ipv4(st.shared_state->req, addr);
    }
    else
    {
        typedef boost::asio::ip::address_v6::bytes_type bytes_type;
        bytes_type ipv6 = p.ip.to_v6().to_bytes();
        in6_addr addr;
        memcpy(addr.s6_addr, ipv6.data(), 16);
        err = SPF_request_set_ipv6(st.shared_state->req, addr);
    }

    if ( err != impl::SPF_E_SUCCESS)
    {
        handler(std::optional<std::string>(), std::optional<std::string>());
        return;
    }

    impl_->shared_state = st.shared_state;

    collect_spf_dns_data_redirect(st,
        boost::protect(
            boost::bind(impl::continue_spf_check<callback_t>, _1, handler)));
}

void spf_check::stop()
{
    if (impl_ && impl_->shared_state)
    {
        impl_->shared_state->done = true;
        boost::mutex::scoped_lock lock(impl_->shared_state->mux);
        impl_->shared_state->r.cancel();
    }
    impl_.reset();
}

bool spf_check::is_inprogress() const
{
    return impl_ && impl_->shared_state && !impl_->shared_state->done;
}
