#pragma once

#include "config.h"
#include "spf_check.h"

#include <boost/asio.hpp>
#include <boost/asio/deadline_timer.hpp>
#include <boost/optional.hpp>

#include <memory>
#include <mutex>

namespace NNwSmtp::NSPF {

template<class TSPFClient>
class TSPFCheckClient : public ISPFCheckClient {
public:
    TSPFCheckClient(TConfigPtr config, boost::asio::io_context& io)
        : Config(std::move(config))
        , Io(io) { }

    void Check(
        std::string from,
        std::string domain,
        boost::asio::ip::address ip,
        spf_logger_t logger,
        TCallback callback
    ) const {
        auto context = std::make_shared<TContext>(Io);
        auto handler =
            [callback = std::move(callback), ex = Io.get_executor()]
            (auto ec, auto response) mutable {
                boost::asio::post(ex, std::bind(std::move(callback), std::move(ec), std::move(response)));
            };

        spf_parameters params;
        params.domain = std::move(domain);
        params.from = std::move(from);
        params.ip = std::move(ip);
        params.logger = std::move(logger);

        context->SPFClient.start(
            Io,
            std::move(params),
            Config->ResolverOptions,
            [handler = handler, context = context]
            (auto result, auto expl) mutable {
                SPFCheckHandler(
                    {std::move(result), std::move(expl)},
                    std::move(context),
                    std::move(handler)
                );
            }
        );

        context->Timer.expires_from_now(boost::posix_time::seconds(Config->SPFOpts.timeout));
        context->Timer.async_wait(
            [handler = std::move(handler), context = context]
            (auto ec) mutable {
                TimeOutHandler(
                    std::move(ec),
                    std::move(context),
                    std::move(handler)
                );
            }
        );
    }

private:
    struct TContext {
        TContext(boost::asio::io_context& io) : Timer(io) { }
        std::mutex HandlerMutex;
        bool IsHandlerCalled = false;
        boost::asio::deadline_timer Timer;
        TSPFClient SPFClient;
    };

    using TContextPtr = std::shared_ptr<TContext>;


    static void SPFCheckHandler(TResponse response, TContextPtr context, TCallback callback) {
        if (std::lock_guard(context->HandlerMutex); context->IsHandlerCalled) {
            return;
        } else {
            context->Timer.cancel();
            context->IsHandlerCalled = true;
        }
        callback({}, std::move(response));
    }

    static void TimeOutHandler(TErrorCode ec, TContextPtr context, TCallback callback) {
        if (std::lock_guard(context->HandlerMutex); context->IsHandlerCalled) {
            return;
        } else {
            context->SPFClient.stop();
            context->IsHandlerCalled = true;
        }
        callback(std::move(ec), {});
    }

    TConfigPtr Config;
    boost::asio::io_context& Io;
};

} // namespace NNwSmtp::NSPF
