#include "options.h"

#include <mail/nwsmtp/src/utils.h>
#include <mail/nwsmtp/src/xyandexhint/parser.h>
#include <mail/library/dsn/options.hpp>
#include <mail/sova/include/nwsmtp/decoder/mimedecoder.h>

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>
#include <util/generic/algorithm.h>
#include <yplatform/find.h>

#include <algorithm>
#include <iomanip>
#include <iostream>
#include <iterator>
#include <sstream>

namespace NNwSmtp {

namespace {

const std::locale locale("ru_RU.UTF-8");

} // namespace anonymous

namespace detail {

ymod_smtpclient::SmtpPoint remote_point_to_smtp(const remote_point& rp) {
    ymod_smtpclient::SmtpPoint result;
    result.host = rp.host_name_;
    if (rp.port_) {
        result.port = rp.port_;
    }
    if (boost::iequals("lmtp", rp.proto_)) {
        result.proto = ymod_smtpclient::SmtpPoint::lmtp;
    }
    return result;
}

} // namespace detail

std::size_t TCaseInsensitiveHash::operator()(const std::string& email) const noexcept {
    const auto lower = boost::to_lower_copy(email, locale);
    return std::hash<std::string>{}(lower);
}

bool TCaseInsensitiveCompare::operator()(const std::string& lhs, const std::string& rhs) const {
    return boost::iequals(lhs, rhs, locale);
}

std::vector<NHint::TParamsContainer> GetNonpersonalHintsParameters(THeaderStorageImpl::THeaderStorageRange headers) {
    std::vector<NHint::TParamsContainer> parameters;
    parameters.reserve(boost::size(headers));
    for (const auto& [_, value, __] : headers) {
        NHint::TParamsContainer params;
        NHint::ParseBase64(std::string{value.begin(), value.end()}, params);
        if (params.find("email") == params.end()) {
            parameters.push_back(params);
        }
    }
    return parameters;
}

bool Contains(const std::vector<NHint::TParamsContainer>& hints, const std::string& parameter) {
    return AnyOf(hints, [&](const auto& hint) { return hint.find(parameter) != hint.end(); });
}

// Headers are added to the top of a message. Messages are parsed from beginning to end.
// So, the first header will be added last into THeaderStorage.
std::optional<std::string> GetFirstValueInOrderOfAddition(
    const std::vector<NHint::TParamsContainer>& hints,
    const std::string& parameter)
{
    for (auto hint = hints.rbegin(); hint != hints.rend(); ++hint) {
        const auto it = hint->find(parameter);
        if (it != hint->end()) {
            return it->second;
        }
    }
    return std::nullopt;
}

} // namespace NNwSmtp

namespace NNwSmtp::NUtil {

TSystemTimePoint GetNow() {
    return TSystemClock::now();
}

int64_t GetNowMs() {
    auto duration = GetNow().time_since_epoch();
    return std::chrono::duration_cast<TMilliSeconds>(duration).count();
}

std::string ToString(const TBufferRange& range) {
    return {range.begin(), range.end()};
}

std::string GetRfc822Date(std::time_t date) {
    struct tm ltm{};
    localtime_r(&date, &ltm);

    std::ostringstream os;
    os.imbue(std::locale::classic());
    os << std::put_time(&ltm, "%a, %d %b %Y %T %z");
    return os.str();
}

std::string GetRfc822DateNow() {
    return GetRfc822Date(TSystemClock::to_time_t(GetNow()));
}

TGetClusterClient MakeGetClusterClient(const std::string& moduleName) {
    return [=]{return yplatform::find<yhttp::cluster_client, std::shared_ptr>(moduleName);};
}

TGetTvm2Module MakeGetTvm2Module(const std::string& moduleName) {
    return [=]{return yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>(moduleName);};
}

boost::asio::ip::address ToIpV4IfPossible(boost::asio::ip::address addr) {
    if (addr.is_v6()) {
        auto v6 = addr.to_v6();
        if (v6.is_v4_mapped() || v6.is_v4_compatible()) {
            addr = v6.to_v4();
        }
    }
    return addr;
}

bool IsCheckSenderInRcpts(bool useDeliveryToSenderControl, const std::string& senderUid) {
    return useDeliveryToSenderControl && !senderUid.empty();
}

bool IsRouting() {
    return gconfig->delivery.routing.primary != RoutingSettings::NONE;
}

} // namespace NNwSmtp::NUtil


std::string rev_order_av4_str(const boost::asio::ip::address_v4& a,
    const std::string& d)
{
  boost::asio::ip::address_v4::bytes_type bytes = a.to_bytes();
  return str(boost::format("%1%.%2%.%3%.%4%.%5%")
      % static_cast<int>(bytes[3])
      % static_cast<int>(bytes[2])
      % static_cast<int>(bytes[1])
      % static_cast<int>(bytes[0])
      % d
    );
}

std::string rev_order_av6_str(const boost::asio::ip::address_v6& a,
    const std::string& d)
{
  std::ostringstream os;
  boost::asio::ip::address_v6::bytes_type bytes = a.to_bytes();
  const char a16[] = "0123456789abcdef";
  for (auto it = bytes.rbegin(); it != bytes.rend(); ++it)
    os << a16[*it % 16] << "." << a16[*it / 16] << ".";
  os << d;
  return os.str();
}

std::string rev_order_str(boost::asio::ip::address address) {
    if (address.is_v6()) {
        auto ipv6 = address.to_v6();
        if (ipv6.is_v4_mapped() || ipv6.is_v4_compatible()) {
            address = ipv6.to_v4();
        }
    }
    if (address.is_v4()) {
        return rev_order_av4_str(address.to_v4());
    }
    return rev_order_av6_str(address.to_v6());
}

bool parse_email(const std::string& email,
        std::string& local, std::string& domain)
{
    if (const char* const at = std::strchr(email.c_str(), '@'))
    {
        local.assign(email.c_str(), at);
        domain.assign(at + 1, email.c_str() + email.size());
        std::transform(domain.begin(), domain.end(), domain.begin(), ::tolower);
        return true;
    }
    return false;
}

std::string GetDomainFromEmail(const std::string& email) {
    if (email.empty()) {
        return {};
    }
    std::string local;
    std::string domain;
    if (!parse_email(email, local, domain)) {
        return {};
    }
    return domain;
}

namespace
{

std::string url_enc2(char c)
{
    if ( (c >= '0' && c <= '9')
            || (c >= 'a' && c <= 'z')
            || (c >= 'A' && c <= 'Z')
            || c == '-' || c == '_' || c == '.')
    {
        return std::string(&c,1);
    }
    else
    {
        char buffer[20];

        snprintf(buffer, sizeof(buffer), "%%%02x", c);

        return buffer;
    }
}
}

std::string url_encode2(const std::string &_buffer)
{
    std::ostringstream remote_filt;

    std::transform(_buffer.begin(), _buffer.end(), std::ostream_iterator<std::string>(remote_filt), url_enc2);

    return remote_filt.str();
}

std::string removePlusFromEmail(const std::string& email) {
    auto posPlus = email.find("+");
    if (posPlus != std::string::npos && (posPlus != 0)) {
        auto posDog = email.find("@");
        if (posDog == std::string::npos) {
            return email.substr(0, posPlus);
        } else {
            return email.substr(0, posPlus) + email.substr(posDog, email.length() - posDog);
        }
    }
    return email;
}

std::string format_notify_mode(int mode)
{
    if (mode == dsn::Options::NONE)
        return std::string();
    if (mode & dsn::Options::NEVER)
        return "NEVER";

    std::string result;
    if (mode & dsn::Options::SUCCESS)
        result += "SUCCESS";
    if (mode & dsn::Options::FAILURE)
        result += (result.empty() ? "FAILURE" : ",FAILURE");
    if (mode & dsn::Options::DELAY)
        result += (result.empty() ? "DELAY" : ",DELAY");
    return result;
}

bool Contains(std::string text, std::string pattern) {
    auto iter = std::search(text.begin(), text.end(), pattern.begin(), pattern.end());
    return iter != text.end();
}

void encode_to_base64(const std::string& to_encode, std::string& dst) {
    std::stringstream ss;
    ss << to_encode;
    ss.seekg(0);

    std::stringstream encoded_string;
    CMimeDecoder encoder;
    encoder.encode("base64", ss, encoded_string);

    encoded_string.str().swap(dst);
}

std::string encode_to_base64(const std::string& to_encode) {
    std::string dst;
    encode_to_base64(to_encode, dst);
    return dst;
}

bool is_invalid(char _elem) {
    return !((_elem >= 'a' && _elem <='z') || (_elem >= 'A' && _elem <='Z') ||
        (_elem >= '0' && _elem <='9') || _elem == '-' || _elem =='.' ||
        _elem == '_' || _elem == '@' || _elem == '%' || _elem == '+' ||
        _elem == '=' || _elem == '!' || _elem == '#' ||   _elem == '$' ||
        _elem == '"' ||   _elem == '*' ||   _elem == '-' || _elem == '/' ||
        _elem == '?' ||   _elem == '^' ||   _elem == '`' || _elem == '{' ||
        _elem == '}' ||   _elem == '|' ||   _elem == '~' || _elem == '&' ||
        _elem == '\''
    ) ;
}

bool address_valid(const std::string& addr, bool checkSyntax) {
    if (addr.empty())
        return true;
    if (std::count_if(addr.begin(), addr.end(), is_invalid) > 0)
        return false;
    if (checkSyntax) {
        std::string::size_type at = addr.find('@');
        return !(at == std::string::npos || at == 0 || at == addr.size() - 1);
    }
    return true;
}

void squash_consecutive_spaces(std::string& str)
{
    str.erase(
        std::unique(str.begin(), str.end(),
            [] (char l, char r) { return ::isspace(l) && ::isspace(r); }),
        str.end());
    std::replace_if(str.begin(), str.end(), ::isspace, ' ');
}

bool request_id_valid(const std::string& r) {
    static const auto is_invalid = [] (char c)
    {
        return !(std::isalnum(c)
            || c == '+' || c == '/' || c == '=' || c == '-');
    };

    const std::size_t size = r.size();
    if (size > 200)
        return false;
    if (std::find_if(r.begin(), r.end(), is_invalid) != r.end())
        return false;
    return true;
}

bool is_corp(const std::string& cluster) {
    return cluster.find("corp") != std::string::npos;
}

std::string GetTimestamp() {
    std::time_t rawtime = std::chrono::system_clock::to_time_t(std::chrono::system_clock::now());
    std::string time(15, '\0');
    strftime(time.data(), time.size(), "%Y%m%d%H%M%S", localtime(&rawtime));
    time.resize(14);
    return time;
}

std::string GetUnixTimestamp() {
    auto  timestamp = std::chrono::duration_cast<std::chrono::seconds>(
        std::chrono::system_clock::now().time_since_epoch()
    ).count();
    return std::to_string(timestamp);
}

std::string IpToBytes(const std::string& ip) {
    boost::system::error_code ec;
    if (auto address = boost::asio::ip::make_address(ip, ec); ec) {
        return {};
    } else if (address.is_v4()) {
        auto bytes = address.to_v4().to_bytes();
        return {bytes.begin(), bytes.end()};
    } else {
        auto bytes = address.to_v6().to_bytes();
        return {bytes.begin(), bytes.end()};
    }
}

bool IsASCII(const std::string& value) {
    return std::all_of(value.begin(), value.end(),
        [] (std::uint8_t symbol) { return std::isprint(symbol); });
}
