#pragma once

#include "response.h"
#include "tvm_guard.h"
#include "types.h"

#include <mail/nwsmtp/src/web/utils/exception.h>
#include <mail/nwsmtp/src/web/utils/get_header.h>

#include <functional>

namespace NNwSmtp::NWeb {

template <class THandle>
struct TBaseHandle {
    TBaseHandle(
        TTvmGuardPtr tvmGuard,
        std::string uri,
        EHttpMethod method
    )
        : TvmGuard(std::move(tvmGuard))
        , Uri(std::move(uri))
        , Method(method) {}

    std::string GetUri() const {
        return Uri;
    }

    template <class ...Ts>
    void operator() (THttpStream stream, TRequest request, Ts&&... args) const {
        if (request->method != EHttpMethod::mth_post) {
            return RespondError(stream, EHttpCode::method_not_allowed,
                "method_not_allowed", "Method is not allowed");
        }

        const auto tvm = TvmGuard->check(
            Uri,
            {},
            GetOptionalHeader(request, tvm_guard::header::serviceTicket()),
            GetOptionalHeader(request, tvm_guard::header::userTicket())
        );

        if (tvm.action == tvm_guard::Action::reject) {
            return RespondError(stream, EHttpCode::unauthorized,
                "unauthorized", "Unauthorized");
        }

        try {
            Handle(stream, std::forward<Ts>(args)...);
        } catch(const TArgumentParseException& exc) {
            return RespondError(stream, EHttpCode::bad_request,
                "bad_request", exc.what());
        } catch(const TMissingArgumentException& exc) {
            return Respond(stream, EHttpCode::bad_request, exc.what());
        } catch(const std::exception& exc) {
            return RespondError(stream, EHttpCode::internal_server_error,
                "internal_server_error", exc.what());
        } catch(...) {
            return RespondError(stream, EHttpCode::internal_server_error,
                "internal_server_error", "Request result is unknown error");
        }
    }

private:
    TTvmGuardPtr TvmGuard;
    THandle Handle;
    std::string Uri;
    EHttpMethod Method;
};

} // namespace NNwSmtp::NWeb
