#include "handle_store.h"

#include "sync_request.h"
#include "sync_response.h"

#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/context.h>
#include <mail/nwsmtp/src/delivery/sync/errors.h>
#include <mail/nwsmtp/src/web/utils/exception.h>
#include <mail/nwsmtp/src/web/utils/get_param.h>
#include <mail/nwsmtp/src/web/utils/message_transforms.h>
#include <mail/nwsmtp/src/web/utils/set_return_code.h>

#include <yplatform/util.h>

namespace NNwSmtp::NWeb {

void TStore::operator() (
    THttpStream stream,
    std::string uid,
    std::string service
) const {
    NSync::TRequest request{};
    try {
        request = NSync::ParseRequest<TRegularTag>(stream);
    } catch (const std::exception& exc) {
        throw TArgumentParseException{"Request parsing error: " + std::string{exc.what()}};
    }
    auto [envInfo, mailInfo, userInfo, options, message] = std::move(request);

    mailInfo.Fid = GetParam(stream->request()->url.params, "fid", "");
    if ((!mailInfo.FolderPath || mailInfo.FolderPath->empty()) && mailInfo.Fid.empty()) {
        throw TMissingArgumentException{"missing arguments fid and folder_path"};
    }

    userInfo.Uid = uid;
    if (yplatform::iequals(service, "imap")) {
        options.DlvType = NDlv::EDlvType::Imap;
    } else {
        options.DlvType = NDlv::EDlvType::Collectors;
    }

    TBuffer transformedMessage;
    if (options.UnDotStuffing && options.UnDotStuffing.value()) {
        auto result = StoreMessageTransforms(message);
        transformedMessage.swap(result);
    } else {
        transformedMessage.swap(message);
    }

    auto context = boost::make_shared<TContext>(envInfo.SessionId, envInfo.EnvId, gconfig->clusterName, gconfig->hostName);

    try {
        return NDlv::RunSyncDelivery(context, envInfo, mailInfo, userInfo, options, transformedMessage,
            [stream = stream] (auto ec, auto result) {
                if (ec && ec != NDlv::EError::DuplicateFound) {
                    SetReturnCode(stream, NSync::ToErrorCode(ec));
                    return NSync::RespondError(stream, ec);
                }
                return NSync::RespondSuccess<TRegularTag>(stream, result);
            });
    } catch (const std::exception& exc) {
        NWLOG_CTX(error, context, "handle_store", exc.what());
        throw;
    }
}

}  // namespace NNwSmtp::NWeb
