#pragma once

#include <mail/nwsmtp/src/big_ml/errors.h>
#include <mail/nwsmtp/src/blackbox/error_code.h>
#include <mail/nwsmtp/src/delivery/async/error_code.h>
#include <mail/nwsmtp/src/ml/client_impl.h>
#include <mail/nwsmtp/src/smtp_client/error_code.h>

#include <boost/system/error_code.hpp>

namespace NNwSmtp::NWeb::NSendMail {

enum class EError {
    Ok,
    PathToLong,
    BadAddressSyntax,
    BadSender,
    BadRecipient,
    Exception,
    ReadOnlyMailList,
    MailListWriteProhibited,
    ToManyRecipients,
    BadKarma,
    TempBanUser,
    Virus,
    Spam,
    StrongSpam,
    SendMessageFailed,
    SendMessageTempFail,
    FailedToAuthSender,
    SendMessageDiscarded,
    SizeLimitExceeded,
    MLNotFound,
    MLPermissionDenied
};

} // namespace NNwSmtp::NWeb::NSendMail

namespace boost::system {

template <>
struct is_error_code_enum<NNwSmtp::NWeb::NSendMail::EError> : std::true_type {};

} // namespace boost::system

namespace NNwSmtp::NWeb::NSendMail {

class TErrorCategory final : public boost::system::error_category {

public:
    const char* name() const noexcept override {
        return "nwsmtp_send_mail";
    }

    std::string message(int value) const override {
        switch (static_cast<EError>(value)) {
            case EError::Ok:
                return "Ok";
            case EError::PathToLong:
                return "Path too long";
            case EError::BadAddressSyntax:
                return "Bad address mailbox syntax";
            case EError::BadRecipient:
                return "Bad recipient";
            case EError::BadSender:
                return "Bad sender";
            case EError::Exception:
                return "Exception";
            case EError::ReadOnlyMailList:
                return "Maillist is readonly";
            case EError::MailListWriteProhibited:
                return "External user can't write to internal list";
            case EError::ToManyRecipients:
                return "Too many recipients";
            case EError::BadKarma:
                return "User has bad karma";
            case EError::TempBanUser:
                return "Temporary ban user";
            case EError::Virus:
                return "Trying to send a virus email";
            case EError::Spam:
                return "Trying to send a spam email";
            case EError::StrongSpam:
                return "Trying to send a strong spam email";
            case EError::SendMessageFailed:
                return "Send message failed";
            case EError::SendMessageTempFail:
                return "Send message temporarily failed";
            case EError::FailedToAuthSender:
                return "Failed to authorize the sender";
            case EError::SendMessageDiscarded:
                return "Send message discarded";
            case EError::SizeLimitExceeded:
                return "Mail size limit exceeded";
            case EError::MLNotFound:
                return "Mail list not found";
            case EError::MLPermissionDenied:
                return "Mail list permission denied";
        }
        return "Unknown error code: " + std::to_string(value);
    }

    bool equivalent(const boost::system::error_code& code, int condition) const noexcept override {
        using namespace NNwSmtp;
        switch (EError(condition)) {
            case EError::BadKarma: {
                return code == NBlackBox::EError::BadKarma;
            }
            case EError::TempBanUser: {
                return code == NBlackBox::EError::TempBanUser;
            }
            case EError::BadRecipient: {
                return code == NBlackBox::EError::UserNotFound
                    || code == NBlackBox::EError::UserBlocked;
            }
            case EError::BadSender: {
                return code == NBlackBox::EError::UserNotFound
                    || code == NBlackBox::EError::EmptySender
                    || code == NBlackBox::EError::UserBlocked
                    || code == NBlackBox::EError::EmptyAuthData
                    || code == NBlackBox::EError::ForbiddenForAssessors
                    || code == NBlackBox::EError::Mdbreg
                    || code == NBlackBox::EError::NoSuid
                    || code == NBlackBox::EError::NoPddeula
                    || code == NBlackBox::EError::NotFoundSmtpScope
                    || code == NBlackBox::EError::NoAccessRights
                    || code == NBlackBox::EError::NoAuthSuccess;
            }
            case EError::FailedToAuthSender: {
                return code == NAsyncDlv::EError::RpopAuthTempFail
                    || code == NAsyncDlv::EError::RpopAuthRejected;
            }
            case EError::Virus: {
                return code == NAsyncDlv::EError::AvirRejected;
            }
            case EError::Spam: {
                return code == NAsyncDlv::EError::SpamRejected;
            }
            case EError::StrongSpam: {
                return code == NAsyncDlv::EError::RfcFailRejected
                    || code == NAsyncDlv::EError::UrlRblRejected
                    || code == NAsyncDlv::EError::BadKarmaRejected
                    || code == NAsyncDlv::EError::MailLimitsRejected
                    || code == NAsyncDlv::EError::PddAdminKarmaRejected
                    || code == NAsyncDlv::EError::BouncesRejected
                    || code == NAsyncDlv::EError::SpamComplRejected
                    || code == NAsyncDlv::EError::MaliciousRejected;
            }
            case EError::SendMessageDiscarded: {
                return code == NAsyncDlv::EError::MaliciousDiscarded
                    || code == NAsyncDlv::EError::AvirDiscarded
                    || code == NAsyncDlv::EError::SpamDiscarded;
            }
            case EError::SendMessageFailed: {
                return code == NAsyncDlv::EError::TooManyHeadersReject
                    || code == NAsyncDlv::EError::DeliveryRejected
                    || code == NAsyncDlv::EError::SenderIdRejected
                    || code == NAsyncDlv::EError::RelayNotSet
                    || code == NAsyncDlv::EError::CycleDetected
                    || code == SmtpClient::EError::Rejected;
            }
            case EError::SendMessageTempFail: {
                return code == NAsyncDlv::EError::DkimSignTempFail
                    || code == NAsyncDlv::EError::DeliveryTempFail
                    || code == NAsyncDlv::EError::RouteTempFail
                    || code == SmtpClient::EError::TempFail
                    || code == NBlackBox::EError::BbTempUserError
                    || code == NAsyncDlv::EError::SoTempFail;

            }
            case EError::MLNotFound: {
                return code == NBigML::EErrorCode::EC_NOT_FOUND
                    || code == list_client_ec::NOT_FOUND;
            }
            case EError::MLPermissionDenied: {
                return code == NBigML::EErrorCode::EC_PERMISSION_DENIED;
            }
            default:
                return boost::system::error_category::equivalent(code, condition);
        }
    }
};

inline const TErrorCategory& getErrorCategory() {
    static TErrorCategory errorCategory;
    return errorCategory;
}

inline auto make_error_code(EError ec) {
    return boost::system::error_code(static_cast<int>(ec), getErrorCategory());
}

inline auto make_error_condition(EError ec) {
    return boost::system::error_condition(static_cast<int>(ec), getErrorCategory());
}

} // namespace NNwSmtp::NWeb::NSendMail
