#include "make_args.h"

#include <mail/nwsmtp/src/web/utils/get_header.h>
#include <mail/nwsmtp/src/web/utils/get_param.h>
#include <mail/nwsmtp/src/web/utils/message_transforms.h>
#include <mail/nwsmtp/src/utils.h>
#include <mail/nwsmtp/src/web/request.h>

#include <boost/asio.hpp>

#include <boost/range/algorithm.hpp>
#include <boost/range/adaptor/filtered.hpp>

namespace NNwSmtp::NWeb::NSendMail {

TXYandexHintFull MakeXYandexCommonHint(const TArgsPtr& args) {
    TXYandexHintFull common;
    if (args->Ip.has_value()) {
        common.Set(NHint::TIpFrom{args->Ip.value()});
    }
    common.Set(NHint::TSource{args->Service});
    if (args->CommonLabels.has_value()) {
        common.Set(NHint::TLabels{{args->CommonLabels->begin(), args->CommonLabels->end()}});
    }
    return common;
}

TXYandexHintFull MakeXYandexSenderHint(const TArgsPtr& args) {
    TXYandexHintFull sender;
    sender.Set(NHint::TEmail{args->FromEmail});
    sender.Set(NHint::TReceivedDate{std::time(nullptr)});
    sender.Set(NHint::TNotify{args->Notify.value_or("0") == "1" ? true : false});
    sender.Set(NHint::TFilters{false});
    sender.Set(NHint::TSkipLoopPrevention{true});
    sender.Set(NHint::TSaveToSent{args->SaveToSent.value_or("0") == "1" ? true : false});
    if (args->Host.has_value()) {
        sender.Set(NHint::THost{args->Host.value()});
    }
    if (args->RecipientsHiddenCopy.has_value()) {
        sender.Set(NHint::TBcc{{args->RecipientsHiddenCopy->begin(), args->RecipientsHiddenCopy->end()}});
    }
    if (args->Lids.has_value()) {
        sender.Set(NHint::TLids{{args->Lids->begin(), args->Lids->end()}});
    }
    if (args->SenderLabels.has_value()) {
        sender.Set(NHint::TLabels{{args->SenderLabels->begin(), args->SenderLabels->end()}});
    }
    return sender;
}

TXYandexHintFull MakeXYandexPersonalHint(const std::string& email, const std::vector<std::string>& labels) {
    TXYandexHintFull personal;
    personal.Set(NHint::TEmail{email});
    personal.Set(NHint::TLabels{{labels.begin(), labels.end()}});
    return personal;
}


yplatform::zerocopy::segment MakeSendMailXYandexHint(const TArgsPtr& args) {
    auto headers = MakeSegment(MakeXYandexCommonHint(args));
    headers.append(MakeSegment(MakeXYandexSenderHint(args)));

    if (args->PersonalLabels.has_value()) {
        boost::range::for_each(
            args->PersonalLabels.value(), [&](const auto& labels) {
                headers.append(MakeSegment(MakeXYandexPersonalHint(labels.first, labels.second)));
            }
        );
    }

    return headers;
}

std::string MakeAdditionalHeaders(const TArgsPtr& args) {
    std::string headers;
    if (args->SpamCheck == "0") {
        headers += "X-Yandex-Spam: 1\r\n";
    }
    if (args->AvirCheck == "0") {
        headers += "X-Yandex-Avir: 1\r\n";
    }
    return headers;
}

yplatform::zerocopy::segment MakeSendMailHeaders(const TArgsPtr& args) {
    auto hint = MakeSendMailXYandexHint(args);
    auto headers = MakeSegment(MakeAdditionalHeaders(args));
    hint.append(headers);
    return hint;
}

TXYandexHint<NHint::TLabels, NHint::TLids> MakeSendSystemMailXYandexHint(const TArgsPtr& args) {
    TXYandexHint<NHint::TLabels, NHint::TLids> hint;
    if (args->Lids.has_value()) {
        hint.Set(NHint::TLids{{args->Lids->begin(), args->Lids->end()}});
    }
    if (args->SenderLabels.has_value()) {
        hint.Set(NHint::TLabels{{args->SenderLabels->begin(), args->SenderLabels->end()}});
    }
    return hint;
}

yplatform::zerocopy::segment MakeSendSystemMailHeaders(const TArgsPtr& args) {
    auto hint = MakeSegment(MakeSendSystemMailXYandexHint(args));
    auto headers = MakeSegment(MakeAdditionalHeaders(args));
    hint.append(headers);
    return hint;
}

void MakeRecipients(const TArgsPtr& args) {
    auto recipientsCopySize = args->RecipientsCopy.has_value() ? args->RecipientsCopy->size() : 0;
    auto recipientsHiddenCopySize = args->RecipientsHiddenCopy.has_value() ? args->RecipientsHiddenCopy->size() : 0;
    args->Recipients.reserve(recipientsCopySize + recipientsHiddenCopySize + args->Recipients.size());

    if (args->RecipientsCopy.has_value()) {
        boost::range::copy(args->RecipientsCopy.value(), std::back_inserter(args->Recipients));
    }
    if (args->RecipientsHiddenCopy.has_value()) {
        boost::range::copy(args->RecipientsHiddenCopy.value(), std::back_inserter(args->Recipients));
    }
}

void MakeHostArgs(const TArgsPtr& args, const std::string& remoteIp) {
    args->RemoteIp = remoteIp;
    auto rawRemoteIp = boost::asio::ip::make_address(remoteIp);
    if (!args->Host || args->Host->empty() || !args->Ip || args->Ip->empty()) {
        args->Host = remoteIp;
        args->Ip = remoteIp;
        args->RawIp = rawRemoteIp;
    } else {
        boost::system::error_code ec;
        if (auto rawIp = boost::asio::ip::make_address(args->Ip.value(), ec); ec) {
            args->Host = remoteIp;
            args->Ip = remoteIp;
            args->RawIp = rawRemoteIp;
        } else {
            args->RawIp = rawIp;
        }
    }
}

TArgsPtr MakeArgs(
    const TRequest& request,
    std::string uid,
    std::string fromEmail,
    std::string service,
    std::string detectSpam,
    std::string detectVirus
) {
    auto&& [json, message] = ParseBodyTo<NSendMail::TArgs>(request);

    TArgsPtr args = std::make_shared<TArgs>(std::move(json));

    if (message.size() == 0) {
        throw std::invalid_argument("Empty message in request body");
    }

    args->Uid = std::move(uid);
    args->Service = std::move(service);
    args->FromEmail = std::move(fromEmail);
    args->AvirCheck = std::move(detectVirus);
    args->SpamCheck = std::move(detectSpam);
    args->ContextId = request->context->uniq_id();

    MakeHostArgs(args, request->context->remote_address);
    auto headers = MakeSendMailHeaders(args);
    std::string headersStr{headers.cbegin(), headers.cend()};
    std::string msgStr{message.cbegin(), message.cend()};
    AddEndingCRLF(msgStr);

    args->Message = NUtil::MakeSegment(NUtil::JoinStrings(headersStr, msgStr));

    MakeRecipients(args);
    return args;
}

TArgsPtr MakeArgs(
    const TRequest& request,
    std::string uid,
    std::string fromEmail,
    std::string service
) {
    if (request->raw_body.size() == 0) {
        throw std::invalid_argument("Empty message in request body");
    }

    auto args = std::make_shared<TArgs>();
    args->Uid = std::move(uid);
    args->Service = std::move(service);
    args->FromEmail = std::move(fromEmail);
    args->Recipients = GetVectorParam(request->url.params, "to");
    args->Lids = GetVectorParam(request->url.params, "lid", {});
    args->SenderLabels = GetVectorParam(request->url.params, "label", {});
    args->ContextId = request->context->uniq_id();

    MakeHostArgs(args, request->context->remote_address);
    auto headers = MakeSendSystemMailHeaders(args);
    std::string headersStr{headers.cbegin(), headers.cend()};
    std::string rawBodyStr{request->raw_body.cbegin(), request->raw_body.cend()};
    AddEndingCRLF(rawBodyStr);

    args->Message = NUtil::MakeSegment(NUtil::JoinStrings(headersStr, rawBodyStr));

    args->CheckMailFrom = false;
    MakeRecipients(args);
    return args;
}

} // namespace NNwSmtp::NWeb::NSendMail
