#include "send_mail.h"
#include "utils.h"

#include <mail/nwsmtp/src/aliases.h>
#include <mail/nwsmtp/src/avir/client_impl.h>
#include <mail/nwsmtp/src/bb_result.h>
#include <mail/nwsmtp/src/big_ml/errors.h>
#include <mail/nwsmtp/src/blackbox/config.h>
#include <mail/nwsmtp/src/delivery/async/async.h>
#include <mail/nwsmtp/src/rcpt_to/expand_alias.h>
#include <mail/nwsmtp/src/options.h>
#include <mail/nwsmtp/src/so/client_impl.h>
#include <mail/nwsmtp/src/smtp_client/smtp_client.h>
#include <mail/nwsmtp/src/recognizer/recognizer_impl.h>
#include <mail/nwsmtp/src/yarm/client_impl.h>
#include <mail/nwsmtp/src/utils.h>
#include <mail/nwsmtp/src/types.h>

#include <mail/library/received_header/received_header.h>

#include <boost/range/algorithm.hpp>

#include <format>

namespace NNwSmtp::NWeb::NSendMail {

namespace ph = std::placeholders;

const std::string Where {"SEND-MAIL"};

#include <yplatform/yield.h>

void TSendMail::operator()(
    TYieldCtx yieldCtx,
    TErrorCode ec
) {
    try {
        reenter (yieldCtx) {
            SessionStartTime = time(NULL);

            NWLOG_L(notice, Where, "from=" + Args->FromEmail);

            if (Args->FromEmail.size() > MAX_LENGTH_ADDRESS) {
                ec = EError::PathToLong;
                yield break;
            }

            if (Args->Recipients.size() >= Config->RcptCountLimit) {
                ec = EError::ToManyRecipients;
                yield break;
            }

            if (!ValidateAddress(Args->FromEmail, Config)) {
                ec = EError::BadAddressSyntax;
                yield break;
            }

            MailEnvelope->m_sender = Args->FromEmail;
            MailEnvelope->orig_message_ = Args->Message;
            MailEnvelope->orig_message_size_ = Args->Message.size();

            NWLOG_L(notice, Where, "message size=" + std::to_string(MailEnvelope->orig_message_size_));

            if (MailEnvelope->orig_message_size_ > Config->MessageSizeLimit) {
                NWLOG_L(notice, Where, "queue file size limit exceeded");
                ec = EError::SizeLimitExceeded;
                yield break;
            }

            yield BlackBoxCheckMailFrom(yieldCtx);
            if (ec) {
                yield break;
            }

            for (
                RecipientIt = Args->Recipients.cbegin();
                RecipientIt != Args->Recipients.cend();
                ++RecipientIt
            ) {
                NWLOG_L(notice, Where, "rcpt=" + *RecipientIt);
                RcptContext = std::make_shared<TRcptContext>();

                if (
                    !CheckAt(*RecipientIt) ||
                    !CheckStrictAscii(*RecipientIt, Config) ||
                    !CheckPercent(*RecipientIt, Config)
                ) {
                    ec = EError::BadRecipient;
                    yield break;
                }

                RcptContext->Email = TransformPercentRecipient(*RecipientIt, Config);
                RcptContext->IsAlias = gAliases.Contains(RcptContext->Email);

                AddNotifyMode();

                yield CallRouter(yieldCtx);
                if (ec) {
                    yield break;
                }

                yield BlackBoxCheckRecipient(yieldCtx);
                if (ec) {
                    yield break;
                }

                CheckSenderInRcpts();

                yield ExpandMaillist(yieldCtx);
                if (ec) {
                    yield break;
                }
            }

            AddSenderToRcpts();

            if (MailEnvelope->m_rcpts.Empty()) {
                yield break;
            }

            AddReceivedHeader();

            yield Delivery(yieldCtx);
        }
    } catch(const std::exception& exc) {
        NWLOG_L_EXC(notice, Where, "exception occurred", exc);
        return Handler(make_error_code(EError::Exception));
    }

    if (yieldCtx.is_complete()) {
        if (ec) {
            NWLOG_L_EC(error, Where, "error occurred", ec);
            return Handler(ec);
        }
        Handler(TErrorCode {});
    }
}

#include <yplatform/unyield.h>

void TSendMail::CallRouter(TYieldCtx yieldCtx) {
    if (IsRouting(Config, RcptContext)) {
        RouterClient->asyncRoute(Context, GetDomainFromEmail(RcptContext->Email),
            [rc = RcptContext, yieldCtx = std::move(yieldCtx)](auto ec, auto response) {
                if (!ec) {
                    rc->DomainType = response.domainType;
                }
                yieldCtx(std::move(ec));
            });
    } else {
        yieldCtx(TErrorCode {});
    }
}

void TSendMail::BlackBoxCheckRecipient(TYieldCtx yieldCtx) {
    if (!IsBlackBoxRecipientCheck(Config, RcptContext)) {
        return yieldCtx(TErrorCode {});
    }
    BlackBoxChecks->CheckRecipient(
        Context,
        removePlusFromEmail(RcptContext->Email),
        Args->Ip.value_or(""),
        Config->CorpList.use,
        true,
        std::bind(CheckRecipientHandler, RcptContext, Context, std::move(yieldCtx), ph::_1, ph::_2)
    );
}

void TSendMail::BlackBoxCheckMailFrom(TYieldCtx yieldCtx) {
    if (!IsBlackBoxMailFromCheck(Config, Args->CheckMailFrom)) {
        return yieldCtx(TErrorCode {});
    }

    BlackBoxChecks->CheckMailFrom(
        Context,
        MailEnvelope->m_sender,
        Args->Ip.value_or(""),
        std::bind(CheckMailFromHandler, MailEnvelope, Context, std::move(yieldCtx), ph::_1, ph::_2)
    );
}

void TSendMail::ExpandMaillist(TYieldCtx yieldCtx) {
    if (Config->CorpList.use) {
        MLClient->Run(Context, {RcptContext->Email},
            std::bind(ResolveMLHandler, MailEnvelope, RcptContext, Context,
                Config, std::move(yieldCtx), ph::_1, ph::_2)
        );
    } else if (Config->BigMLUse && RcptContext->BBData.IsMaillist) {
        BigMLClient->Run(Context,
            {MailEnvelope->m_sender, removePlusFromEmail(RcptContext->Email)},
            std::bind(ResolveBigMLHandler, MailEnvelope, RcptContext,
                Context, Config, std::move(yieldCtx), ph::_1, ph::_2)
        );
    } else {
        ExpandAlias(*RcptContext, MailEnvelope, Config->DeliveryToSenderControl.use);
        yieldCtx(TErrorCode {});
    }
}

void TSendMail::AddNotifyMode() {
    if (Args->Notify.has_value() && Args->Notify.value() == "1") {
        RcptContext->NotifyMode |= dsn::Options::SUCCESS;
        RcptContext->NotifyMode |= dsn::Options::FAILURE;
    }
}

void TSendMail::CheckSenderInRcpts() {
    if (NUtil::IsCheckSenderInRcpts(Config->DeliveryToSenderControl.use, MailEnvelope->m_sender_uid)
        && MailEnvelope->m_sender_uid == RcptContext->BBData.Uid
    ) {
        MailEnvelope->is_sender_in_rcpts_ = true;
    }
}

void TSendMail::CheckRecipientHandler(
    TRcptContextPtr rcptContext,
    TContextPtr context,
    TYieldCtx yieldCtx,
    TErrorCode ec,
    NBlackBox::TResponse response
) {
    if (ec) {
        return yieldCtx(std::move(ec));
    }

    rcptContext->BBData.Suid = response.Suid;
    rcptContext->BBData.Country = response.Country;
    rcptContext->BBData.PhoneConfirmed = response.PhoneConfirmed;
    rcptContext->BBData.Uid = response.Uid;
    rcptContext->BBData.DefaultEmail = response.DefaultEmail;
    rcptContext->BBData.OrgId = response.OrgId;
    rcptContext->BBData.IsHosted = response.Hosted;
    rcptContext->BBData.CatchAll = response.CatchAll;
    rcptContext->BBData.IsMaillist = response.IsMailList;
    rcptContext->BBData.IsCorpList = response.IsCorpList;
    rcptContext->BBData.RegistrationDate = response.RegistrationDate;

    NWLOG_CTX(notice, context, "BB", "rcpt=" + rcptContext->Email + ", uid=" + response.Uid
        + ", suid=" + std::to_string(response.Suid));

    yieldCtx(TErrorCode {});
}

void TSendMail::CheckMailFromHandler(
    envelope_ptr mailEnvelope,
    TContextPtr context,
    TYieldCtx yieldCtx,
    TErrorCode ec,
    NBlackBox::TResponse response
) {
    if (ec) {
        return yieldCtx(std::move(ec));
    }

    TBlackBoxResult result;
    result.Suid = response.Suid;
    result.Db = response.Mdb;
    result.Karma = response.Karma;
    result.KarmaStatus = response.KarmaStatus;
    result.TimeStamp = response.BornDate;
    result.Country = response.Country;
    result.Uid = response.Uid;
    result.DefaultEmail = response.DefaultEmail;
    result.PhoneConfirmed = response.PhoneConfirmed;

    mailEnvelope->m_sender_uid = response.Uid;
    mailEnvelope->m_sender_default_email = response.DefaultEmail;
    mailEnvelope->senderInfo = std::move(result);

    NWLOG_CTX(notice, context, "BB", "from=" + mailEnvelope->m_sender + ", uid=" + response.Uid
        + ", suid=" + std::to_string(result.Suid));
    NWLOG_CTX(notice, context, "BB", "org_id=" + (response.OrgId.empty() ? "missing" : response.OrgId));

    yieldCtx(TErrorCode {});
}

void TSendMail::ResolveMLHandler(
    envelope_ptr mailEnvelope,
    TRcptContextPtr rcptContext,
    TContextPtr context,
    TConfigPtr config,
    TYieldCtx yieldCtx,
    TErrorCode ec,
    NML::TResponse response
) {
    using NBigML::EErrorCode;

    const auto& bbData = rcptContext->BBData;
    const auto& subscribers = response.Subscribers;

    if (!config->CorpList.ignoreErrors && ec && ec != make_error_condition(EError::MLNotFound)) {
        return yieldCtx(std::move(ec));
    }

    if (
        (ec && (config->CorpList.ignoreErrors || ec == make_error_condition(EError::MLNotFound))) ||
        subscribers.empty()
    ) {
        auto added = mailEnvelope->add_recipient(rcptContext->Email, bbData.Suid, false, bbData.Uid, bbData.Uid,
            bbData.DefaultEmail, bbData.Country, rcptContext->DomainType,
            rcptContext->NotifyMode, bbData.PhoneConfirmed, bbData.IsMaillist, bbData.RegistrationDate);

        if (!added) {
            return yieldCtx(EError::BadRecipient);
        }

        return yieldCtx(TErrorCode {});
    }


    auto domain = GetDomain(mailEnvelope->m_sender);

    bool inWriters = response.WhoCanWrite.count(boost::to_lower_copy(mailEnvelope->m_sender));

    std::string result;
    if (response.Readonly && !inWriters) {
        ec = EError::ReadOnlyMailList;
        result = "maillist read-only and user not in writers";
    } else if (response.Internal && mailEnvelope->m_sender_uid.empty()) {
        ec = EError::MailListWriteProhibited;
        result = "external user can't write to internal list";
    } else {
        result = "list expanded to " + std::to_string(subscribers.size()) + " recipients";
    }

    NWLOG_CTX(notice, context, Where, "from=" + mailEnvelope->m_sender + ", domain=" + domain + ", read_only=" +
        std::to_string(response.Readonly) + ", writers=" + std::to_string(inWriters) + ", internal=" +
        std::to_string(response.Internal) + ", result=" + result);

    if (ec) {
        return yieldCtx(std::move(ec));
    }

    if (mailEnvelope->m_sender_uid.empty() && response.Internal) {
         NWLOG_CTX(notice, context, "ML", "from=" + mailEnvelope->m_sender + " to=" + rcptContext->Email);
    }

    mailEnvelope->is_mailing_list_ |= bbData.IsCorpList;

    if (mailEnvelope->is_mailing_list_ && mailEnvelope->ml_address_.empty()) {
        mailEnvelope->ml_address_ = rcptContext->Email;
    }

    boost::range::for_each(subscribers, [&](auto& subscriber){
        mailEnvelope->add_recipient(subscriber, 0, true, "",
            bbData.Uid, bbData.DefaultEmail, "",
            DomainType::UNKNOWN, dsn::Options::NEVER);
    });

    mailEnvelope->m_no_local_relay |= !subscribers.empty();

    yieldCtx(TErrorCode {});
}

void TSendMail::ResolveBigMLHandler(
    envelope_ptr mailEnvelope,
    TRcptContextPtr rcptContext,
    TContextPtr context,
    TConfigPtr config,
    TYieldCtx yieldCtx,
    TErrorCode ec,
    NBigML::TResponse response
) {
    using NBigML::EErrorCode;

    const auto& bbData = rcptContext->BBData;
    const auto& subscribers = response.Subscribers;

    if (ec == make_error_condition(EError::MLNotFound) || (!ec && subscribers.empty())) {
        ExpandAlias(*rcptContext, mailEnvelope, config->DeliveryToSenderControl.use);
        return yieldCtx(TErrorCode {});
    }

    if (ec) {
        NWLOG_CTX(notice, context, Where,
            "from=" + mailEnvelope->m_sender + ", read_only="
            + std::to_string(ec == make_error_condition(EError::MLPermissionDenied)) + ", result="
            + ec.message());
        return yieldCtx(std::move(ec));
    }

    if (bbData.OrgId.empty()) {
        NWLOG_CTX(notice, context, "ORGID", std::format("status=empty_org_id, rcpt={}", rcptContext->Email));
    }

    boost::range::for_each(subscribers, [&](auto& subscriber){
        mailEnvelope->add_recipient(subscriber.first, 0, true, subscriber.second,
            bbData.Uid, bbData.DefaultEmail, "",
            DomainType::LOCAL, dsn::Options::NEVER, false, true);
    });

    mailEnvelope->m_no_local_relay = true;

    if (NUtil::IsCheckSenderInRcpts(config->DeliveryToSenderControl.use, mailEnvelope->m_sender_uid)) {
        auto it = boost::range::find_if(subscribers, [&](auto& subscriber) {
            return subscriber.second == mailEnvelope->m_sender_uid;
        });

        if (it != subscribers.end()) {
            mailEnvelope->is_sender_in_rcpts_ = true;
        }
    }

    std::string result = "list expanded to " + std::to_string(subscribers.size()) + " recipients";
    NWLOG_CTX(notice, context, Where,
        "from=" + mailEnvelope->m_sender + ", read_only=0" + ", result=" + result);

    yieldCtx(TErrorCode {});
}

void TSendMail::AddSenderToRcpts() {
    if (Config->DeliveryToSenderControl.use
        && !MailEnvelope->is_sender_in_rcpts_
        && !MailEnvelope->m_sender_uid.empty()
    ) {
        MailEnvelope->add_recipient(MailEnvelope->m_sender, 0, true, MailEnvelope->m_sender_uid,
            "", "", "", DomainType::LOCAL, dsn::Options::NEVER,
            false, false, 0, false);
        MailEnvelope->sender_added_for_control = true;
        NWLOG_L(notice, Where, "added sender to rcpts");
    }
}

void TSendMail::AddReceivedHeader() {
    NReceived::TReceived received;

    received.Date = NUtil::GetRfc822DateNow();
    received.RemoteIp = Args->RemoteIp;
    received.RemoteHost = Args->RemoteIp;
    received.HeloHost = Args->RemoteIp;
    received.LocalHost = boost::asio::ip::host_name();
    received.Protocol = "HTTP";
    received.SessionId = Context->GetSessionId();
    received.ClusterName = Config->ClusterName;

    MailEnvelope->received_headers_ = NReceived::BuildReceivedHeader(received);
}

void TSendMail::Delivery(TYieldCtx yieldCtx) {
    auto dlvRequest = NAsyncDlv::TRequest {
        Args->Host.value_or(""),
        Args->ContextId,
        MailEnvelope->m_sender,
        Args->Host.value_or(""),
        MailEnvelope,
        Args->RawIp,
        false,
        {},
        {},
        SessionStartTime
    };

    DeliveryClient->Run(
        Context,
        std::move(dlvRequest),
        std::bind(DeliveryHandler, Context, std::move(yieldCtx), ph::_1, ph::_2)
    );
}

void TSendMail::DeliveryHandler(
    TContextPtr context,
    TYieldCtx yieldCtx,
    TErrorCode ec,
    std::string response
) {
    NWLOG_CTX(notice, context, Where, response);
    yieldCtx(std::move(ec));
}

void StartSendMail(
    THandler&& handler,
    TArgsPtr args,
    boost::asio::io_context& io
) {
    std::shared_ptr<RouterCall> routerClient;
    if (NUtil::IsRouting()) {
        routerClient = std::make_shared<RouterCall>(
            io,
            gconfig->aresOpts.resolverOptions,
            gconfig->delivery.routing
        );
    }

    auto blackboxClient = std::make_shared<NBlackBox::NClient::TClientImpl>(
        yplatform::find<yhttp::cluster_client, std::shared_ptr>("blackbox_client"),
        yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("tvm"),
        io
    );
    auto blackBoxChecks = std::make_shared<NBlackBox::TBBChecksImpl>(
        NBlackBox::MakeConfig(gconfig),
        std::move(blackboxClient)
    );

    auto bigMLClient = std::make_shared<NBigML::TClient>(
       gconfig->bigMlOpts,
       yplatform::find<ymod_httpclient::cluster_call, std::shared_ptr>("big_ml_client"),
       io
    );

    auto mlClient = std::make_shared<NML::TClient>(
        gconfig->corpList,
        yplatform::find<yhttp::simple_call, std::shared_ptr>("http_client"),
        io
    );

    auto soClient = std::make_shared<NSO::TSOClient>(
        yplatform::find<yhttp::cluster_client, std::shared_ptr>(gconfig->soOpts.SoClient),
        gconfig->soOpts.SoClient == "so_out_client" ? NSO::ESOType::SOOut : NSO::ESOType::SOIn,
        io
    );

    auto smtpClient = std::make_shared<SmtpClient::TSmtpClient>(
        yplatform::find<ymod_smtpclient::Call, std::shared_ptr>("smtp_client"),
        io
    );

    auto avirCheckClient = std::make_shared<NAvir::TAvirCheckClient>(
        std::make_shared<avir_client>(io, gconfig->avir.client_opts),
        io
    );

    auto recognizer = yplatform::find<TRecognizer, std::shared_ptr>("recognizer");

    auto yarmClient = std::make_shared<NYarm::TClientImpl>(
        gconfig->yarm,
        yplatform::find<ymod_httpclient::cluster_call, std::shared_ptr>("yarm_client"),
        io
    );

    auto deliveryClient = std::make_shared<NAsyncDlv::TAsyncDelivery>(
        smtpClient,
        soClient,
        avirCheckClient,
        recognizer,
        yarmClient,
        routerClient,
        io
    );

    yplatform::spawn(
        io.get_executor(),
        std::make_shared<TSendMail>(
            std::move(handler),
            std::move(routerClient),
            std::move(blackBoxChecks),
            std::move(bigMLClient),
            std::move(mlClient),
            std::move(deliveryClient),
            NSendMail::MakeConfig(gconfig),
            std::move(args)
        )
    );
}

} // namespace NNwSmtp::NWeb::NSendMail
