#pragma once

#include "args.h"
#include "config.h"
#include "error_code.h"

#include <mail/nwsmtp/src/big_ml/client_impl.h>
#include <mail/nwsmtp/src/blackbox/bb_checks_impl.h>
#include <mail/nwsmtp/src/blackbox/bb_client_impl.h>
#include <mail/nwsmtp/src/delivery/async/interface.h>
#include <mail/nwsmtp/src/envelope.h>
#include <mail/nwsmtp/src/ml/client_impl.h>
#include <mail/nwsmtp/src/rcpt_context.h>
#include <mail/nwsmtp/src/router/router.h>
#include <mail/nwsmtp/src/types.h>

#include <yplatform/coroutine.h>

#include <boost/asio.hpp>

#include <functional>
#include <memory>
#include <vector>

namespace NNwSmtp::NWeb::NSendMail {

using THandler = std::function<void(boost::system::error_code)>;

class TSendMail {
    using TYieldCtx = yplatform::yield_context<TSendMail>;

public:
    TSendMail(
        THandler handler,
        TRouterPtr routerClient,
        NBlackBox::TBBChecksPtr blackBoxChecks,
        NBigML::TClientPtr bigMLClient,
        NML::TClientPtr mlClient,
        NAsyncDlv::TAsyncDeliveryPtr deliveryClient,
        TConfigPtr config,
        TArgsPtr args
    )
        : Handler(std::move(handler))
        , RouterClient(std::move(routerClient))
        , BlackBoxChecks(std::move(blackBoxChecks))
        , BigMLClient(std::move(bigMLClient))
        , MLClient(std::move(mlClient))
        , DeliveryClient(std::move(deliveryClient))
        , Config(std::move(config))
        , Args(std::move(args))
    { }

    void operator()(
        TYieldCtx yieldCtx,
        TErrorCode ec = {}
    );

private:
    void CallRouter(TYieldCtx yieldCtx);
    void BlackBoxCheckRecipient(TYieldCtx yieldCtx);
    void BlackBoxCheckMailFrom(TYieldCtx yieldCtx);
    void ExpandMaillist(TYieldCtx yieldCtx);
    void AddNotifyMode();
    void CheckSenderInRcpts();
    void AddSenderToRcpts();
    void AddReceivedHeader();
    void Delivery(TYieldCtx yieldCtx);

    static void CheckRecipientHandler(
        TRcptContextPtr rcptContext,
        TContextPtr context,
        TYieldCtx yieldCtx,
        TErrorCode ec,
        NBlackBox::TResponse response
    );
    static void CheckMailFromHandler(
        envelope_ptr mailEnvelope,
        TContextPtr context,
        TYieldCtx yieldCtx,
        TErrorCode ec,
        NBlackBox::TResponse response
    );
    static void ResolveMLHandler(
        envelope_ptr mailEnvelope,
        TRcptContextPtr rcptContext,
        TContextPtr context,
        TConfigPtr config,
        TYieldCtx yieldCtx,
        TErrorCode ec,
        NML::TResponse response
    );
    static void ResolveBigMLHandler(
        envelope_ptr mailEnvelope,
        TRcptContextPtr rcptContext,
        TContextPtr context,
        TConfigPtr config,
        TYieldCtx yieldCtx,
        TErrorCode ec,
        NBigML::TResponse response
    );
    static void DeliveryHandler(
        TContextPtr context,
        TYieldCtx yieldCtx,
        TErrorCode ec,
        std::string response
    );

    THandler Handler;
    TRouterPtr RouterClient;
    NBlackBox::TBBChecksPtr BlackBoxChecks;
    NBigML::TClientPtr BigMLClient;
    NML::TClientPtr MLClient;
    NAsyncDlv::TAsyncDeliveryPtr DeliveryClient;
    TConfigPtr Config;
    TArgsPtr Args;

    TRcptContextPtr RcptContext;
    envelope_ptr MailEnvelope = boost::make_shared<envelope>();
    TContextPtr Context = boost::make_shared<TContext>(
        Args->ContextId,
        MailEnvelope->m_id,
        Config->ClusterName,
        Config->HostName
    );
    std::vector<std::string>::const_iterator RecipientIt;
    std::time_t SessionStartTime = 0;

    constexpr static std::int32_t MAX_LENGTH_ADDRESS = 256;
};

void StartSendMail(
    THandler&& handler,
    TArgsPtr args,
    boost::asio::io_context& io
);

} // namespace NNwSmtp::NWeb::NSendMail
