#include "utils.h"

#include <mail/nwsmtp/src/delivery/async/error_code.h>
#include <mail/nwsmtp/src/router/router.h>

#include <boost/range/algorithm.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/adaptor/transformed.hpp>

#include <optional>
#include <utility>

namespace NNwSmtp::NWeb::NSendMail {

bool CheckAt(const std::string& rcpt) {
    const std::string::size_type dogPos = rcpt.find('@');
    if (dogPos == std::string::npos || dogPos == 0 || dogPos == rcpt.length() - 1) {
        return false;
    }
    return true;
}

bool CheckStrictAscii(const std::string& rcpt, const TConfigPtr& config) {
    if (config->StrictAsciiRecipient && std::count_if(rcpt.begin(), rcpt.end(), is_invalid) > 0) {
        return false;
    }
    return true;
}

bool CheckPercent(const std::string& rcpt, const TConfigPtr& config) {
    const std::string::size_type percPos = rcpt.find('%');
    if (percPos != std::string::npos && (!config->AllowPercentHack || percPos == 0)) {
        return false;
    }
    return true;
}

std::string TransformPercentRecipient(const std::string& recipient, const TConfigPtr& config) {
    const std::string::size_type percPos = recipient.find('%');
    const std::string::size_type dogPos = recipient.find('@');
    if (
        percPos != std::string::npos && percPos != 0 &&
        config->AllowPercentHack
    ) {
        return recipient.substr(0, percPos)
            + "@" + recipient.substr(percPos + 1, dogPos - percPos - 1);
    }
    return recipient;
}

bool IsRouting(const TConfigPtr& config, const TRcptContextPtr& rcptContext) {
    return config->Primary != RoutingSettings::NONE && !rcptContext->IsAlias;
}

bool IsBlackBoxRecipientCheck(
    const TConfigPtr& config,
    const TRcptContextPtr& rcptContext
) {
    return config->CheckRcpt &&
        (
            config->Primary == RoutingSettings::NONE ||
            rcptContext->DomainType == DomainType::LOCAL
        ) &&
        !rcptContext->IsAlias;
}

bool ValidateAddress(const std::string& address, const TConfigPtr& config) {
    return address_valid(address, config->CheckSenderSyntax);
}

bool IsBlackBoxMailFromCheck(const TConfigPtr& config, bool checkMailFrom) {
    return config->CheckSender && checkMailFrom;
}

std::string GetDomain(const std::string& address) {
    std::string::size_type at = address.find('@');
    std::string domain = (at != std::string::npos ? address.substr(at + 1) : "");
    std::transform(domain.begin(), domain.end(), domain.begin(), ::tolower);
    return domain;
}

std::optional<std::string> GetBanReason(const TErrorCode& ec) {
    if (ec == NAsyncDlv::EError::RfcFailRejected) {
        return "RfcFail";
    } else if (ec == NAsyncDlv::EError::UrlRblRejected) {
        return "UrlRbl";
    } else if (ec == NAsyncDlv::EError::BadKarmaRejected) {
        return "BadKarma";
    } else if (ec == NAsyncDlv::EError::MailLimitsRejected) {
        return "MailLimits";
    } else if (ec == NAsyncDlv::EError::PddAdminKarmaRejected) {
        return "PddAdminKarma";
    } else if (ec == NAsyncDlv::EError::BouncesRejected) {
        return "Bounces";
    } else if (ec == NAsyncDlv::EError::SpamComplRejected) {
        return "SpamCompl";
    }
    return std::nullopt;
}

} // namespace NNwSmtp::NWeb::NSendMail
