#include "tvm_guard.h"

#include "handle_base.h"
#include "handle_restore.h"
#include "handle_save.h"
#include "handle_store.h"
#include "handle_store_mailish.h"
#include "handle_send_mail.h"
#include "handle_send_system_mail.h"
#include "web.h"

#include <ymod_webserver/server.h>

#include <yplatform/module_registration.h>
#include <yplatform/find.h>

namespace NNwSmtp::NWeb {

void TWebServer::init(const yplatform::ptree& config) {
    using ymod_webserver::transformer;
    using ymod_webserver::argument;

    auto reactor = yplatform::find_reactor(config.get<std::string>("reactor"));
    for (size_t i = 0; i < reactor->size(); ++i) {
        if ((*reactor)[i]->size() != 1)
            throw std::runtime_error("TWebServer requires pool_count=N and io_threads=1");
    }
    auto webserver = yplatform::find<ymod_webserver::server>("web_server");

    auto tvmGuard = MakeTvmGuard(
        config.get_child("tvm_guard"),
        yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("tvm")
    );

    auto handleStoreMailish = TBaseHandle<TStoreMailish>{
        tvmGuard,
        "/mail/store_mailish",
        EHttpMethod::mth_post
    };

    auto handleStore = TBaseHandle<TStore>{
        tvmGuard,
        "/mail/store",
        EHttpMethod::mth_post
    };

    auto handleSendMail = TBaseHandle<TSendMail>{
        tvmGuard,
        "/mail/send_mail",
        EHttpMethod::mth_post
    };

    auto handleSendSystemMail = TBaseHandle<TSendSystemMail>{
        tvmGuard,
        "/mail/send_system_mail",
        EHttpMethod::mth_post
    };

    auto handleRestore = TBaseHandle<TRestore>{
        tvmGuard,
        "/mail/restore",
        EHttpMethod::mth_post
    };

    auto handleSave = TBaseHandle<TSave>{
        tvmGuard,
        "/mail/save",
        EHttpMethod::mth_post
    };

    webserver->bind("", {"/ping"}, [](auto stream) {
        stream->set_code(EHttpCode::ok);
        stream->result_body("pong");
    });

    webserver->bind("", {handleStoreMailish.GetUri()},
        [=](auto stream, auto uid, auto fid, auto external_imap_id) {
            handleStoreMailish(stream, stream->request(),
                std::move(uid), std::move(fid), std::move(external_imap_id));
        },
        transformer(
            argument<std::string>("uid"),
            argument<std::string>("fid"),
            argument<std::string>("external_imap_id")
        )
    );

    webserver->bind("", {handleStore.GetUri()},
        [=](auto stream, auto uid, auto service) {
            handleStore(stream, stream->request(),
                std::move(uid), std::move(service));
        },
        transformer(
            argument<std::string>("uid"),
            argument<std::string>("service")
        )
    );

    auto reactorSendMail = yplatform::find_reactor("global");

    webserver->bind("", {handleSendMail.GetUri()},
        [=](auto stream, auto uid, auto fromEmail, auto service, auto detectSpam, auto detectVirus) {
            handleSendMail(stream, stream->request(),
                std::move(uid), std::move(fromEmail), std::move(service),
                std::move(detectSpam), std::move(detectVirus), *reactorSendMail->io());
        },
        transformer(
            argument<std::string>("uid"),
            argument<std::string>("from_email"),
            argument<std::string>("service"),
            argument<std::string>("detect_spam"),
            argument<std::string>("detect_virus")
        )
    );

    webserver->bind("", {handleSendSystemMail.GetUri()},
        [=](auto stream, auto uid, auto fromEmail, auto service) {
            handleSendSystemMail(stream, stream->request(), std::move(uid),
                std::move(fromEmail), std::move(service), *reactorSendMail->io());
        },
        transformer(
            argument<std::string>("uid"),
            argument<std::string>("from"),
            argument<std::string>("service")
        )
    );

    webserver->bind("", {handleRestore.GetUri()},
        [=](auto stream, auto uid, auto requestId) {
            handleRestore(stream, stream->request(), std::move(uid), std::move(requestId),
                *reactorSendMail->io());
        },
        transformer(
            argument<std::string>("uid"),
            argument<std::string>("request_id")
        )
    );

    webserver->bind("", {handleSave.GetUri()},
        [=](auto stream, auto uid, auto email, auto requestId) {
            handleSave(stream, stream->request(), std::move(uid), std::move(email),
                std::move(requestId));
        },
        transformer(
            argument<std::string>("uid"),
            argument<std::string>("email"),
            argument<std::string>("request_id")
        )
    );

}

}  // namespace NNwSmtp::NWeb

DEFINE_SERVICE_OBJECT(NNwSmtp::NWeb::TWebServer)
