#include "client_impl.h"

#include "error_code.h"
#include "types_reflection.h"

#include <mail/nwsmtp/src/log.h>
#include <mail/yreflection/include/yamail/data/deserialization/json_reader.h>

#include <format>

namespace NNwSmtp::NYarm {

namespace {

const std::string WHERE = "YARM";

void LogSuccess(const TContextPtr& context, const std::string& email) {
    NWLOG_CTX(notice, context, WHERE, std::format("email={}, status=ok", email));
}

void LogError(const TContextPtr& context, const std::string& email, const TErrorCode& errc, std::string_view reason) {
    NWLOG_CTX(error, context, WHERE, std::format("email={}, status=error ({}: {})", email, errc.message(), reason));
}

yhttp::request MakeRequest(const TRequest& request, const std::string& secret) {
    return yhttp::request::GET(
        "/api/v2/smtp_data" + yhttp::url_encode({
            {"uid", request.Uid},
            {"email", request.Email},
            {"secret", secret},
        })
    );
}

} // namespace anonymous

TClientImpl::TClientImpl(const Options::YarmOpts& options, THttpClientPtr httpClient, boost::asio::io_service& io)
    : Secret{options.secret}
    , HttpClient{httpClient}
    , Io{io}
{}

void TClientImpl::AsyncRun(TContextPtr context, TRequest request, TCallback callback) {
    auto clientContext = context->CreateTaskContext(WHERE + ":");
    auto handler = [ex = Io.get_executor(), context = std::move(context), email = request.Email, callback = std::move(callback)]
        (TErrorCode errc, yhttp::response response) {
            boost::asio::post(
                ex,
                std::bind(HandleAsyncRun, context, email, callback, errc, response)
            );
        };
    HttpClient->async_run(std::move(clientContext), MakeRequest(request, Secret), std::move(handler));
}

void TClientImpl::HandleAsyncRun(
    TContextPtr context,
    std::string email,
    TCallback callback,
    TErrorCode errc,
    yhttp::response httpResponse
) {
    if (errc) {
        LogError(context, email, errc, httpResponse.body);
        return callback(std::move(errc), TResponse {});
    }

    if (httpResponse.status != 200 && httpResponse.status != 500) {
        errc = make_error_code(EError::OtherError);
        LogError(context, email, errc, "Bad status code " + std::to_string(httpResponse.status));
        return callback(std::move(errc), TResponse {});
    }

    TSmtpDataResponse parsedData;
    try {
        yamail::data::deserialization::fromJson<TSmtpDataResponse>(httpResponse.body, parsedData);
    } catch (const std::exception& e) {
        errc = make_error_code(EError::ParseError);
        LogError(context, email, errc, e.what());
        return callback(std::move(errc), TResponse {});
    }

    TResponse response;
    std::string errorMsg;
    if (parsedData.SmtpData.has_value()) {
        auto& smtpData = parsedData.SmtpData.value();
        response = {
            .Login = std::move(smtpData.Login),
            .PasswordOrToken = std::move(smtpData.PasswordOrToken),
            .IsOauth = smtpData.IsOauth,
        };
    } else {
        errc = make_error_code(EError::OtherError);
        errorMsg = "No smtp_data in response";
    }

    if (parsedData.Error.has_value()) {
        errc = make_error_code(EError::OtherError);
        errorMsg = parsedData.Error->Description;
    }

    if (httpResponse.status == 500) {
        if (errorMsg == "no such collector") {
            errc = make_error_code(EError::NotFound);
        } else {
            errc = make_error_code(EError::OtherError);
            if (errorMsg.empty()) {
                errorMsg = "Bad status code 500";
            }
        }
    }

    if (errc) {
        LogError(context, email, errc, errorMsg);
        return callback(std::move(errc), std::move(response));
    }

    if (response.Login.empty() || response.PasswordOrToken.empty()) {
        errc = make_error_code(EError::OtherError);
        LogError(context, email, errc, "Login and/or password/token are empty");
        return callback(std::move(errc), std::move(response));
    }

    LogSuccess(context, email);
    return callback(std::move(errc), std::move(response));
}

} // namespace NNwSmtp::NYarm
