#include <mail/nwsmtp/src/address_parser.h>
#include <mail/nwsmtp/src/utils.h>
#include <mail/nwsmtp/src/types.h>

#include <string>

#include <gtest/gtest.h>

using NNwSmtp::parse_address;
using NNwSmtp::mailbox_list;

class TParseAddressTest : public testing::Test {
public:
    void SetUp() override {
    }

    auto MakeHeaderRange(const std::string& header) {
        Buffer = NNwSmtp::NUtil::MakeSegment(header);
        return NNwSmtp::TBufferRange(Buffer.cbegin(), Buffer.cend());
    }

    std::pair<bool, mailbox_list> Parse(const std::string& headerStr) {
        mailbox_list mailboxes;
        auto header = MakeHeaderRange(headerStr);
        auto res = parse_address(header, mailboxes);
        return {res, mailboxes};
    }
private:
    NNwSmtp::TBuffer Buffer;
};

TEST_F(TParseAddressTest, SingleAddrSpecSuccess) {
    auto [res, mailboxes] = Parse("foo@yandex.ru");
    ASSERT_TRUE(res);

    ASSERT_EQ(mailboxes.size(), 1u);
    auto [displayName, address] = mailboxes.front();
    // It's a bug
    ASSERT_EQ(std::string(displayName.begin(), displayName.end()), "foo");
    ASSERT_EQ(std::string(address.begin(), address.end()), "foo@yandex.ru");
}

TEST_F(TParseAddressTest, TwoAddrSpecsSuccess) {
    auto [res, mailboxes] = Parse("foo@yandex.ru, bar@yandex.ru");
    ASSERT_TRUE(res);

    ASSERT_EQ(mailboxes.size(), 2u);
    auto [displayName, address] = mailboxes.front();
    ASSERT_EQ(std::string(address.begin(), address.end()), "foo@yandex.ru");

    std::tie(displayName, address) = *std::next(mailboxes.begin());
    ASSERT_EQ(std::string(address.begin(), address.end()), "bar@yandex.ru");
}

TEST_F(TParseAddressTest, SingleNameAddrSuccess) {
    auto [res, mailboxes] = Parse("Display Name <foo@yandex.ru>");
    ASSERT_TRUE(res);

    ASSERT_EQ(mailboxes.size(), 1u);
    auto [displayName, address] = mailboxes.front();
    ASSERT_EQ(std::string(displayName.begin(), displayName.end()), "Display Name");
    ASSERT_EQ(std::string(address.begin(), address.end()), "foo@yandex.ru");
}

TEST_F(TParseAddressTest, EmptyDisplayNameAndSingleAngleAddrSucces) {
    auto [res, mailboxes] = Parse("<foo@yandex.ru>");
    ASSERT_TRUE(res);

    ASSERT_EQ(mailboxes.size(), 1u);
    auto [displayName, address] = mailboxes.front();
    ASSERT_EQ(std::string(displayName.begin(), displayName.end()), "");
    ASSERT_EQ(std::string(address.begin(), address.end()), "foo@yandex.ru");
}

TEST_F(TParseAddressTest, FoldingHeaderSuccess) {
    // https://tools.ietf.org/html/rfc822#section-3.1.1
    auto [res, mailboxes] = Parse("\"Joe & J. Harvey\" \r\n\t <ddd@Org>, JJV@BBN");
    ASSERT_TRUE(res);

    ASSERT_EQ(mailboxes.size(), 2u);
    auto [displayName, address] = mailboxes.front();
    ASSERT_EQ(std::string(displayName.begin(), displayName.end()), "\"Joe & J. Harvey\"");
    ASSERT_EQ(std::string(address.begin(), address.end()), "ddd@Org");

    std::tie(displayName, address) = *std::next(mailboxes.begin());
    // It's a bug
    ASSERT_EQ(std::string(displayName.begin(), displayName.end()), "JJV");
    ASSERT_EQ(std::string(address.begin(), address.end()), "JJV@BBN");
}

TEST_F(TParseAddressTest, TwoNameAddrsSuccess) {
    auto [res, mailboxes] = Parse("First Display Name <foo@yandex.ru>, \"Second Display Name <buz@gmail.com>\" <bar@yandex.ru>");
    ASSERT_TRUE(res);

    ASSERT_EQ(mailboxes.size(), 2u);
    auto [displayName, address] = mailboxes.front();

    ASSERT_EQ(std::string(address.begin(), address.end()), "foo@yandex.ru");
    ASSERT_EQ(std::string(displayName.begin(), displayName.end()), "First Display Name");

    std::tie(displayName, address) = *std::next(mailboxes.begin());

    ASSERT_EQ(std::string(address.begin(), address.end()), "bar@yandex.ru");
    ASSERT_EQ(std::string(displayName.begin(), displayName.end()), "\"Second Display Name <buz@gmail.com>\"");
}

TEST_F(TParseAddressTest, CommentSuccess) {
    auto [res, mailboxes] = Parse("(my comment) <foo@yandex.ru>");
    ASSERT_TRUE(res);

    ASSERT_EQ(mailboxes.size(), 1u);
    auto [displayName, address] = mailboxes.front();
    ASSERT_EQ(std::string(displayName.begin(), displayName.end()), "");
    ASSERT_EQ(std::string(address.begin(), address.end()), "foo@yandex.ru");
}

TEST_F(TParseAddressTest, EmptyHeader) {
    auto [res, mailboxes] = Parse("");
    ASSERT_FALSE(res);
}

TEST_F(TParseAddressTest, OnlyDisplayNameFail) {
    auto [res, mailboxes] = Parse("Display Name");
    ASSERT_FALSE(res);
}

TEST_F(TParseAddressTest, BadCharsInAddSpecAreAllowed) {
    auto [res, mailboxes] = Parse("f^#-oo@yandex.ru");
    ASSERT_TRUE(res);

    ASSERT_EQ(mailboxes.size(), 1u);
    auto [displayName, address] = mailboxes.front();
    ASSERT_EQ(std::string(address.begin(), address.end()), "f^#-oo@yandex.ru");
}

TEST_F(TParseAddressTest, WhiteSpaceNotAllowed) {
    auto [res, mailboxes] = Parse("<foo @yandex.ru>");
    ASSERT_FALSE(res);
}
