#include "auth/impl.h"
#include "ymod_smtpserver/commands.h"
#include <mail/nwsmtp/src/auth/interface.h>
#include <mail/nwsmtp/src/utils.h>

#include <gtest/gtest.h>

#include <string>

namespace {

using namespace testing;
using namespace NNwSmtp::NAuth;
namespace AuthMethods = ymod_smtpserver::commands::AuthMethods;

std::string CreatePlainString(std::string login, std::string password) {
    std::string ret;
    ret += '\0';
    ret += login;
    ret += '\0';
    ret += password;
    return ret;
}

std::string CreateXOAuth2String(std::string login, std::string token) {
    return "user=" + login + "\1auth=Bearer " + token + "\1\1";
}

TEST(AuthPlainConsumer, for_correct_auth_data_should_extract_successfully) {
    auto auth = std::make_shared<NNwSmtp::NAuth::TAuthCredentialsExtractorImpl>();
    auth->SetMethod(AuthMethods::Plain{});

    std::string b64 = encode_to_base64(CreatePlainString("login", "password"));
    auto res = auth->Consume(b64);
    EXPECT_EQ(res.Status, EStatus::Done);
    auto data = auth->GetData();
    EXPECT_EQ(data.Login, "login");
    EXPECT_EQ(data.Password, "password");
}

struct TPlainBadFormed : TestWithParam<std::string> {};

TEST_P(TPlainBadFormed, for_bad_formed_auth_data_should_return_bad_formed) {
    auto auth = std::make_shared<NNwSmtp::NAuth::TAuthCredentialsExtractorImpl>();
    auth->SetMethod(AuthMethods::Plain{});
    std::string b64 = encode_to_base64(GetParam());
    auto res = auth->Consume(b64);
    EXPECT_EQ(res.Status, EStatus::BadFormed);
    EXPECT_EQ(res.ReplyToUser, "VXNlcm5hbWU6");
}

INSTANTIATE_TEST_SUITE_P(AuthPlainConsumerBadFormed, TPlainBadFormed,
    Values(
        CreatePlainString({}, {}),
        CreatePlainString("login", {}),
        CreatePlainString({}, "")
    )
);

TEST(AuthLoginConsumer, for_correct_auth_data_should_extract_successfully) {
    auto auth = std::make_shared<NNwSmtp::NAuth::TAuthCredentialsExtractorImpl>();
    auth->SetMethod(AuthMethods::Login{});

    std::string b64 = encode_to_base64("login");
    auto res = auth->Consume(b64);
    EXPECT_EQ(res.Status, EStatus::NeedMore);

    b64 = encode_to_base64("password");
    res = auth->Consume(b64);

    EXPECT_EQ(res.Status, EStatus::Done);
    auto data = auth->GetData();
    EXPECT_EQ(data.Login, "login");
    EXPECT_EQ(data.Password, "password");
}

TEST(AuthLoginConsumer, for_correct_auth_data_with_incorrect_empty_interrupts_should_extract_successfully) {
    auto auth = std::make_shared<NNwSmtp::NAuth::TAuthCredentialsExtractorImpl>();
    auth->SetMethod(AuthMethods::Login{});

    auto res = auth->Consume(encode_to_base64({}));
    EXPECT_EQ(res.Status, EStatus::BadFormed);
    EXPECT_EQ(res.ReplyToUser, "VXNlcm5hbWU6");

    res = auth->Consume(encode_to_base64("login"));
    EXPECT_EQ(res.Status, EStatus::NeedMore);

    res = auth->Consume(encode_to_base64({}));
    EXPECT_EQ(res.Status, EStatus::BadFormed);
    EXPECT_EQ(res.ReplyToUser, "UGFzc3dvcmQ6");

    res = auth->Consume(encode_to_base64("password"));
    EXPECT_EQ(res.Status, EStatus::Done);

    auto data = auth->GetData();
    EXPECT_EQ(data.Login, "login");
    EXPECT_EQ(data.Password, "password");
}

TEST(AuthLoginConsumer, for_half_of_correct_auth_data_with_incorrect_trash_symbols_interrupts_should_return_bad_formed) {
    auto auth = std::make_shared<NNwSmtp::NAuth::TAuthCredentialsExtractorImpl>();
    auth->SetMethod(AuthMethods::Login{});

    auto res = auth->Consume("%!!");
    EXPECT_EQ(res.Status, EStatus::BadFormed);
    EXPECT_EQ(res.ReplyToUser, "VXNlcm5hbWU6");

    res = auth->Consume(encode_to_base64("login"));
    EXPECT_EQ(res.Status, EStatus::NeedMore);

    res = auth->Consume("%!!");
    EXPECT_EQ(res.Status, EStatus::BadFormed);
    EXPECT_EQ(res.ReplyToUser, "UGFzc3dvcmQ6");
}

TEST(AuthXOAuth2Consumer, for_correct_auth_data_should_extract_successfully) {
    auto auth = std::make_shared<NNwSmtp::NAuth::TAuthCredentialsExtractorImpl>();
    auth->SetMethod(AuthMethods::XOAuth2{});

    auto res = auth->Consume(encode_to_base64(CreateXOAuth2String("login", "token")));
    EXPECT_EQ(res.Status, EStatus::Done);
    auto data = auth->GetData();
    EXPECT_EQ(data.Login, "login");
    EXPECT_EQ(data.Token, "token");
}

TEST(AuthXOAuth2Consumer, for_empty_auth_data_should_return_bad_formed) {
    auto auth = std::make_shared<NNwSmtp::NAuth::TAuthCredentialsExtractorImpl>();
    auth->SetMethod(AuthMethods::XOAuth2{});

    auto res = auth->Consume(encode_to_base64(CreateXOAuth2String({}, {})));
    EXPECT_EQ(res.Status, EStatus::BadFormed);
}

TEST(AuthXOAuth2Consumer, for_incorrect_auth_data_should_return_bad_formed) {
    auto auth = std::make_shared<NNwSmtp::NAuth::TAuthCredentialsExtractorImpl>();
    auth->SetMethod(AuthMethods::XOAuth2{});

    std::string str = "usr=foo\1auth=Bearer bar\1\1";

    auto res = auth->Consume(encode_to_base64(str));
    EXPECT_EQ(res.Status, EStatus::BadFormed);
}

TEST(TestAuthGetMethod, for_plain_auth_method_should_return_plain_method) {
    auto auth = std::make_shared<NNwSmtp::NAuth::TAuthCredentialsExtractorImpl>();
    auth->SetMethod(AuthMethods::Plain{});
    EXPECT_EQ(auth->GetMethod(), "PLAIN");
}

TEST(TestAuthGetMethod, for_login_auth_method_should_return_login_method) {
    auto auth = std::make_shared<NNwSmtp::NAuth::TAuthCredentialsExtractorImpl>();
    auth->SetMethod(AuthMethods::Login{});
    EXPECT_EQ(auth->GetMethod(), "LOGIN");
}

TEST(TestAuthGetMethod, for_xoauth2_auth_method_should_return_xoauth2_method) {
    auto auth = std::make_shared<NNwSmtp::NAuth::TAuthCredentialsExtractorImpl>();
    auth->SetMethod(AuthMethods::XOAuth2{});
    EXPECT_EQ(auth->GetMethod(), "XOAUTH2");
}

TEST(TestAuthGetMethod, for_not_supported_auth_method_should_return_invalid_method) {
    auto auth = std::make_shared<NNwSmtp::NAuth::TAuthCredentialsExtractorImpl>();
    auth->SetMethod(AuthMethods::NotSupported{});
    EXPECT_EQ(auth->GetMethod(), "INVALID");
}

}
