#include "http_cluster_client_mock.h"
#include "test_with_spawn.h"

#include <mail/nwsmtp/src/big_ml/client_impl.h>
#include <mail/nwsmtp/src/big_ml/errors.h>
#include <mail/nwsmtp/src/big_ml/utils.h>
#include <mail/nwsmtp/ut/init_log.h>

#include <library/cpp/resource/resource.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

using namespace NNwSmtp;
using namespace NBigML;
using namespace NTesting;
using Request = yhttp::request;

namespace ymod_httpclient {

static bool operator ==(const request& lhs, const request& rhs) {
    return lhs.method == rhs.method && lhs.url == rhs.url
        && lhs.headers == rhs.headers
        && ((lhs.body && rhs.body && *lhs.body == *rhs.body) || (!lhs.body && !rhs.body));
}

}

TEST(BigMLTest, ParseSuccessfullResponse) {
    std::string httpBody = NResource::Find("recipients.json");
    auto resp = ParseResponse(httpBody);
    ASSERT_EQ(resp.Status, "ok");
    ASSERT_EQ(resp.Code, "");
    ASSERT_EQ(resp.Message, "");
    std::vector<std::pair<std::string, std::string>> expected{
        {"user1@bigmltest.yaconnect.com", "1130000000918058"},
        {"user2@bigmltest.yaconnect.com", "1130000000918056"}
    };
    ASSERT_TRUE(resp.Subscribers == expected);
}

TEST(BigMLTest, ParseFailedResponse) {
    std::string httpBody = R"({
        "response" : {
            "params" : {},
            "code" : "unhandled_exception",
            "message" : "Unhandled exception"
        },
        "status" : "error"
    })";
    auto resp = ParseResponse(httpBody);
    ASSERT_EQ(resp.Status, "error");
    ASSERT_EQ(resp.Code, "unhandled_exception");
    ASSERT_EQ(resp.Message, "Unhandled exception");
    ASSERT_EQ(resp.Subscribers.empty(), true);
}

TEST(BigMLTest, ParseNoStatusResponseThrowsException) {
    std::string httpBody = R"({
        "response" : {
            "subscriptions": [
                {
                    "email": "user1@bigmltest.yaconnect.com",
                    "uid": 1130000000918058
                }
            ]
        }
    })";
    ASSERT_THROW(ParseResponse(httpBody), std::exception);
}

TEST(BigMLTest, ParseUidAsEmptyStringThrowsException) {
    std::string httpBody = R"({
        "response" : {
            "subscriptions": [
                {
                    "email": "user1@bigmltest.yaconnect.com",
                    "uid": ""
                }
            ]
        },
        "status": "ok"
    })";
    ASSERT_THROW(ParseResponse(httpBody), std::exception);
}

TEST(BigMLTest, ParseEmptyEmailThrowsException) {
    std::string httpBody = R"({
        "response" : {
            "subscriptions": [
                {
                    "email": "",
                    "uid": 1130000000918058
                }
            ]
        },
        "status": "ok"
    })";
    ASSERT_THROW(ParseResponse(httpBody), std::exception);
}

TEST(BigMLTest, ParseNoUidThrowsException) {
    std::string httpBody = R"({
        "response" : {
            "subscriptions": [
                {
                    "email": "user1@bigmltest.yaconnect.com"
                }
            ]
        },
        "status": "ok"
    })";
    ASSERT_THROW(ParseResponse(httpBody), std::exception);
}

TEST(BigMLTest, ParseNoEmailThrowsException) {
    std::string httpBody = R"({
        "response" : {
            "subscriptions": [
                {
                    "uid": 1130000000918058
                }
            ]
        },
        "status": "ok"
    })";
    ASSERT_THROW(ParseResponse(httpBody), std::exception);
}

TEST(BigMLTest, BuildUrlWhenBaseUriDoesNotContainQuestion) {
    auto url = BuildUrl(
    "/api/v1/recipients",
    "from@ya.ru",
    "to@ya.ru");
    ASSERT_EQ(
    url,
    "/api/v1/recipients?email_from=from@ya.ru&email_to=to@ya.ru&version=2");
}

TEST(BigMLTest, BuildUrlWhenBaseUriContainsQuestion) {
    auto url = BuildUrl(
    "/api/v1/recipients?foo=bar",
    "from@ya.ru",
    "to@ya.ru");
    ASSERT_EQ(
    url,
    "/api/v1/recipients?foo=bar&email_from=from@ya.ru&email_to=to@ya.ru&version=2");
}

struct TTestBigMLClient: TTestWithSpawn {
    void SetUp() override {
        NTesting::InitGlobalLog();
    }

    const std::shared_ptr<StrictMock<TClusterClientMock>> HttpClient = GetStrictMockedClusterClient();
    TClientPtr BigMLClient = std::make_shared<TClient>(TConfig {true, true, "/api/v1/recipients", {}, {}},
        HttpClient, Io);

    const Request BigMLRequest = Request::GET(
        "/api/v1/recipients?email_from=pepa@pig.ru&email_to=george@pig.ru&version=2"
    );

    template <typename THandler>
    TResponse CallResolveMaillist(
        TRequest request,
        THandler handler
    ) {
        boost::asio::async_completion<THandler, void(TErrorCode, TResponse)> init(handler);
        BigMLClient->Run(boost::make_shared<TContext>("", "", "", ""), request,
            init.completion_handler);
        return init.result.get();
    }
};

TEST_F(TTestBigMLClient, for_request_ended_with_error_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        TErrorCode ec;

        EXPECT_CALL(*HttpClient, async_run(_, BigMLRequest, _, _))
            .WillOnce(InvokeArgument<3>(ymod_httpclient::http_error::code::ssl_error, yhttp::response {}));

        auto result = CallResolveMaillist({"pepa@pig.ru", "george@pig.ru"}, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
    });
}

TEST_F(TTestBigMLClient, for_bad_request_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        TErrorCode ec;

        EXPECT_CALL(*HttpClient, async_run(_, BigMLRequest, _, _))
            .WillOnce(InvokeArgument<3>(TErrorCode {}, yhttp::response {400, {}, {}, {}}));

        auto result = CallResolveMaillist({"pepa@pig.ru", "george@pig.ru"}, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EErrorCode::EC_BAD_REQUEST);
    });
}

TEST_F(TTestBigMLClient, for_unsuccessful_parsing_response_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        TErrorCode ec;

        EXPECT_CALL(*HttpClient, async_run(_, BigMLRequest, _, _))
            .WillOnce(InvokeArgument<3>(TErrorCode {}, yhttp::response {200, {}, {""}, {}}));

        auto result = CallResolveMaillist({"pepa@pig.ru", "george@pig.ru"}, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EErrorCode::EC_EXCEPTION);
    });
}

TEST_F(TTestBigMLClient, for_good_request_should_return_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        TErrorCode ec;

        EXPECT_CALL(*HttpClient, async_run(_, BigMLRequest, _, _))
            .WillOnce(InvokeArgument<3>(TErrorCode {}, yhttp::response {200, {}, NResource::Find("recipients.json"), {}}));

        auto result = CallResolveMaillist({"pepa@pig.ru", "george@pig.ru"}, yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result.Status, "ok");
        ASSERT_THAT(result.Subscribers,
            ElementsAre(
                Pair("user1@bigmltest.yaconnect.com", "1130000000918058"),
                Pair("user2@bigmltest.yaconnect.com", "1130000000918056")
            )
        );
    });
}
