#include "bb_client_mock.h"

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <ymod_httpclient/errors.h>

#include <mail/nwsmtp/src/context.h>
#include <mail/nwsmtp/src/blackbox/config.h>
#include <mail/nwsmtp/src/blackbox/bb_checks_impl.h>
#include <mail/nwsmtp/src/blackbox/error_code.h>
#include <mail/nwsmtp/ut/test_with_spawn.h>

#include <memory>
#include <chrono>

namespace {

using namespace testing;
using namespace NNwSmtp;
using namespace NNwSmtp::NBlackBox;

const auto DONT_ALLOW_LIST{false};
const auto GET_ORG_ID{true};

auto MakeConfig() {
    return std::make_shared<TConfig>(BlackBoxOpts {}, true, false);
}

NClient::THttpRequest MakeUserInfoRequest(
    const std::string& login,
    const std::string& userip,
    bool allowList = false,
    bool getOrgId = false
) {
    return BuildUserInfoRequest(login, userip, allowList, getOrgId).BuildUserInfoRequest();
}

NClient::THttpRequest MakeLoginRequest(TConfigPtr &config, TAuthData &info) {
    return BuildAuthRequest(config, info).BuildLoginRequest();
}

NClient::THttpRequest MakeOauthRequest(TConfigPtr &config, TAuthData &info) {
    return BuildAuthRequest(config, info).BuildOauthRequest();
}

struct TTestBBChecks: TTestWithSpawn {

    TConfigPtr Config = MakeConfig();

    TContextPtr Context = boost::make_shared<TContext>("","","","");

    std::shared_ptr<StrictMock<NClient::NTest::TBBClientMock>> Client
        = std::make_shared<StrictMock<NClient::NTest::TBBClientMock>>();

    std::shared_ptr<TBBChecksImpl> BbChecks = std::make_shared<TBBChecksImpl>(
        Config,
        Client
    );

    template <typename THandler>
    TResponse CallCheckRecipient(
        TContextPtr context,
        const std::string& email,
        const std::string& ip,
        bool allowList,
        bool getOrgId,
        THandler handler
    ) {
        boost::asio::async_completion<
            THandler,
            void(TErrorCode, TResponse)
        > init(handler);

        BbChecks->CheckRecipient(
            context, email, ip, allowList, getOrgId, init.completion_handler);
        return init.result.get();
    }

    template <typename THandler>
    TResponse CheckMailFrom(
        TContextPtr context,
        const std::string& email,
        const std::string& ip,
        THandler handler
    ) {
        boost::asio::async_completion<
            THandler,
            void(TErrorCode, TResponse)
        > init(handler);

        BbChecks->CheckMailFrom(
            context, email, ip, init.completion_handler);
        return init.result.get();
    }

    template <typename THandler>
    TResponse CheckAuth(
        TContextPtr context,
        TAuthData info,
        THandler handler
    ) {
        boost::asio::async_completion<
            THandler,
            void(TErrorCode, TResponse)
        > init(handler);

        BbChecks->CheckAuth(
            context, info, init.completion_handler);
        return init.result.get();
    }
};

TEST_F(TTestBBChecks, for_recipient_check_ended_with_TErrorCode_should_return_error_code) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", true), _))
            .WillOnce(InvokeArgument<2>(ymod_httpclient::http_error::code::ssl_error, TResponse {}));
        auto result = CallCheckRecipient(Context, "hello@kitty.com", "127.0.0.1", true, false, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::HttpError);
    });
}

TEST_F(TTestBBChecks, for_copr_list_recipient_should_return_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.IsCorpList = true};
        Config->CorpListUse = true;

        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", true), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CallCheckRecipient(Context, "hello@kitty.com", "127.0.0.1", true, false, yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST_F(TTestBBChecks,
        for_recipient_with_empty_uid_and_not_allow_unknown_recipient_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Uid = ""};
        Config->Blackbox.allowUnknownRcpt = false;

        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", true), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CallCheckRecipient(Context, "hello@kitty.com", "127.0.0.1", true, false, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::UserNotFound);
    });
}

TEST_F(TTestBBChecks,
        for_recipient_with_empty_uid_and_allow_unknown_recipient_should_return_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Uid = ""};
        Config->Blackbox.allowUnknownRcpt = true;

        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", true), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CallCheckRecipient(Context, "hello@kitty.com", "127.0.0.1", true, false, yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST_F(TTestBBChecks, for_blocked_recipient_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Uid = "42", .IsBlocked = true};

        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", true), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CallCheckRecipient(Context, "hello@kitty.com", "127.0.0.1", true, false, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::UserBlocked);
    });
}

TEST_F(TTestBBChecks, for_recipient_with_norm_karma_should_return_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Uid = "42", .IsBlocked = false, .Karma = 1};

        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", true), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CallCheckRecipient(Context, "hello@kitty.com", "127.0.0.1", true, false, yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST_F(TTestBBChecks,
        for_recipient_with_bad_karma_and_allow_receive_with_bad_karma_should_return_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Uid = "42", .IsBlocked = false, .Karma = 1000};
        Config->Blackbox.allowReceiveWithBadKarma = true;

        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", true), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CallCheckRecipient(Context, "hello@kitty.com", "127.0.0.1", true, false, yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST_F(TTestBBChecks, for_recipient_with_temp_bad_karma_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {
            .Uid = "42",
            .IsBlocked = false,
            .Karma = 1000,
            .KarmaBanTime = std::numeric_limits<std::int32_t>::max()
        };

        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", true), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CallCheckRecipient(Context, "hello@kitty.com", "127.0.0.1", true, false, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::TempBanUser);
    });
}

TEST_F(TTestBBChecks, for_recipient_check_get_org_id_should_return_org_id) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {
            .Uid = "42",
            .OrgId = "1234567",
        };

        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", true, true), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CallCheckRecipient(Context, "hello@kitty.com", "127.0.0.1", true, true, yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST_F(TTestBBChecks, for_empty_email_should_return_error_code) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;

        auto result = CheckMailFrom(Context, "", "127.0.0.1", yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::EmptySender);
    });
}

TEST_F(TTestBBChecks, for_mail_from_check_ended_with_error_should_return_error_code) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;

        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", DONT_ALLOW_LIST,
            GET_ORG_ID), _)).WillOnce(InvokeArgument<2>(ymod_httpclient::http_error::code::ssl_error,
            TResponse {}));
        auto result = CheckMailFrom(Context, "hello@kitty.com", "127.0.0.1", yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::HttpError);
    });
}

TEST_F(TTestBBChecks, for_mail_from_with_empty_uid_not_allow_unknown_sender_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        auto response = TResponse {.Uid = ""};
        TErrorCode ec;

        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", DONT_ALLOW_LIST,
            GET_ORG_ID), _)).WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckMailFrom(Context, "hello@kitty.com", "127.0.0.1", yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::UserNotFound);
    });
}

TEST_F(TTestBBChecks, for_mail_from_with_empty_uid_allow_unknown_sender_should_return_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        auto response = TResponse {.Uid = ""};
        TErrorCode ec;
        Config->Blackbox.allowUnknownSender = true;
        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", DONT_ALLOW_LIST,
            GET_ORG_ID), _)).WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckMailFrom(Context, "hello@kitty.com", "127.0.0.1", yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST_F(TTestBBChecks, for_blocked_mail_from_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Uid = "42", .IsBlocked = true};

        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", DONT_ALLOW_LIST,
            GET_ORG_ID), _)).WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckMailFrom(Context, "hello@kitty.com", "127.0.0.1", yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::UserBlocked);
    });
}

TEST_F(TTestBBChecks, for_good_mail_from_should_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Uid = "42", .IsBlocked = false};

        EXPECT_CALL(*Client, UserInfo(_, MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", DONT_ALLOW_LIST,
            GET_ORG_ID), _)).WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckMailFrom(Context, "hello@kitty.com", "127.0.0.1", yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST_F(TTestBBChecks, for_auth_with_empty_token_and_login_or_password_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        auto response = TResponse {};
        TErrorCode ec;

        auto result = CheckAuth(Context, {"hello@kitty.com", "", "", "127.0.0.1", "1010", "login"}, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::EmptyAuthData);
    });
};

TEST_F(TTestBBChecks, for_auth_check_ended_with_TErrorCode_should_return_error_code) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};
        auto request = MakeLoginRequest(Config, info);

        EXPECT_CALL(*Client, Login(_, request, _))
            .WillOnce(InvokeArgument<2>(ymod_httpclient::http_error::code::ssl_error, TResponse {}));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::HttpError);
    });
}

TEST_F(TTestBBChecks, for_blocked_user_auth_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.IsBlocked = true};
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Login(_, MakeLoginRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::UserBlocked);
    });
}

TEST_F(TTestBBChecks, for_not_authsuccess_should_return_error_code) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.IsBlocked = false, .AuthSuccess = false};
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Login(_, MakeLoginRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::NoAuthSuccess);
    });
}

TEST_F(TTestBBChecks, for_accessor_and_deny_auth_for_accessors_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Suid = 42, .IsBlocked = false, .IsAssessor = true, .AuthSuccess = true};
        Config->Blackbox.denyAuthForAssessors = true;
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Login(_, MakeLoginRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::ForbiddenForAssessors);
    });
}

TEST_F(TTestBBChecks, for_accessor_and_disabled_deny_auth_for_accessors_should_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Suid = 42, .IsBlocked = false, .IsAssessor = true, .AuthSuccess = true};
        Config->Blackbox.denyAuthForAssessors = false;
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Login(_, MakeLoginRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST_F(TTestBBChecks, for_oauth_with_mismatched_scopes_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        TErrorCode ec;
        auto response = TResponse {.Suid = 42, .IsBlocked = false, .Scopes = {"one_scope"}, .AuthSuccess = true};
        Config->Blackbox.oauthScopes = {"another_scope"};
        TAuthData info{"hello@kitty.com", "password", "token", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Oauth(_, MakeOauthRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::NotFoundSmtpScope);
    });
}

TEST_F(TTestBBChecks, for_good_authdata_oauth_should_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        TErrorCode ec;
        auto response = TResponse {.Suid = 42, .IsBlocked = false, .Scopes = {"test_scope"}, .AuthSuccess = true};
        Config->Blackbox.oauthScopes = {"test_scope"};
        TAuthData info{"hello@kitty.com", "password", "token", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Oauth(_, MakeOauthRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST_F(TTestBBChecks, for_hosted_user_without_pddsid_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {
            .Suid = 42,
            .Hosted = true,
            .IsBlocked = false,
            .HasPDDsid = false,
            .AuthSuccess = true,
        };
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Login(_, MakeLoginRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::NoPddeula);
    });
}

TEST_F(TTestBBChecks, for_hosted_user_with_pddsid_should_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {
            .Suid = 42,
            .Hosted = true,
            .IsBlocked = false,
            .HasPDDsid = true,
            .AuthSuccess = true,
        };
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Login(_, MakeLoginRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST_F(TTestBBChecks, for_user_with_zero_suid_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Suid = 0, .IsBlocked = false, .AuthSuccess = true};
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Login(_, MakeLoginRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::NoSuid);
    });
}

TEST_F(TTestBBChecks, for_user_with_mdbreg_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Suid = 42, .IsBlocked = false, .Mdb = "mdbreg", .AuthSuccess = true};
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Login(_, MakeLoginRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::Mdbreg);
    });
}

TEST_F(TTestBBChecks, for_user_with_bad_karma_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Suid = 42, .IsBlocked = false, .Karma = 85, .AuthSuccess = true};
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Login(_, MakeLoginRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_TRUE(ec);
        ASSERT_EQ(ec, EError::BadKarma);
    });
}

TEST_F(TTestBBChecks, for_user_with_bad_karma_with_pddsid_should_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {
            .Suid = 42,
            .IsBlocked = false,
            .HasPDDsid = true,
            .Karma = 85,
            .AuthSuccess = true,
        };
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Login(_, MakeLoginRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST_F(TTestBBChecks, for_user_with_bad_karma_and_confirmed_phone_should_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {
            .Suid = 42,
            .PhoneConfirmed = true,
            .IsBlocked = false,
            .Karma = 85,
            .AuthSuccess = true,
        };
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Login(_, MakeLoginRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST_F(TTestBBChecks, for_good_authdata_login_should_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;
        TErrorCode ec;
        auto response = TResponse {.Suid = 42, .IsBlocked = false, .AuthSuccess = true};
        TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};

        EXPECT_CALL(*Client, Login(_, MakeLoginRequest(Config, info), _))
            .WillOnce(InvokeArgument<2>(TErrorCode {}, response));
        auto result = CheckAuth(Context, info, yield[ec]);
        ASSERT_FALSE(ec);
        ASSERT_EQ(result, response);
    });
}

TEST(TestBuildUserInfoRequest, test_build_user_info_request) {
    ASSERT_EQ(MakeUserInfoRequest("hello@kitty.com", "127.0.0.1").Url,
        R"(?method=userinfo&format=json&login=hello%40kitty.com&sid=smtp&userip=127.0.0.1)"
        R"(&emails=getdefault&dbfields=account_info.reg_date.uid%2csubscription.suid)"
        R"(.-%2csubscription.login_rule.2%2csubscription.born_date.2%2csubscription.login.-)"
        R"(%2caccount_info.country.uid%2cuserphones.confirmed.uid&attributes=13)"
    );
}

TEST(TestBuildUserInfoRequest, test_build_user_info_request_with_allow_list) {
    ASSERT_EQ(MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", true).Url,
        R"(?method=userinfo&format=json&login=hello%40kitty.com&sid=smtp&userip=127.0.0.1)"
        R"(&emails=getdefault&dbfields=account_info.reg_date.uid%2csubscription.suid)"
        R"(.-%2csubscription.login_rule.2%2csubscription.born_date.2%2csubscription.login.-)"
        R"(%2caccount_info.country.uid%2cuserphones.confirmed.uid%2csubscription.suid.1000)"
        R"(&attributes=13)"
    );
}

TEST(TestBuildUserInfoRequest, test_build_user_info_request_with_ord_id_attribute) {
    ASSERT_EQ(MakeUserInfoRequest("hello@kitty.com", "127.0.0.1", false, true).Url,
        R"(?method=userinfo&format=json&login=hello%40kitty.com&sid=smtp&userip=127.0.0.1)"
        R"(&emails=getdefault&dbfields=account_info.reg_date.uid%2csubscription.suid)"
        R"(.-%2csubscription.login_rule.2%2csubscription.born_date.2%2csubscription.login.-)"
        R"(%2caccount_info.country.uid%2cuserphones.confirmed.uid&attributes=13%2c1031)"
    );
}

TEST(TestBuildAuthRequest, test_build_oauth_request) {
    TAuthData info{"hello@kitty.com", "password", "token", "127.0.0.1", "1010", "login"};
    TConfigPtr Config = MakeConfig();
    ASSERT_EQ(MakeOauthRequest(Config, info).Url,
        R"(?method=oauth&format=json&userip=127.0.0.1&emails=getdefault)"
        R"(&dbfields=subscription.suid.102%2csubscription.login_rule.8)"
        R"(%2csubscription.login_rule.2%2chosts.db_id.-%2caccount_info.country.uid)"
        R"(%2cuserphones.confirmed.uid%2csubscription.suid.2%2csubscription.login.2)"
        R"(&attributes=)"
    );
}

TEST(TestBuildAuthRequest, test_build_login_request) {
    TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};
    TConfigPtr Config = MakeConfig();
    ASSERT_EQ(MakeLoginRequest(Config, info).Url,
        R"(?method=login&format=json&login=hello%40kitty.com&sid=smtp&authtype=smtp)"
        R"(&userip=127.0.0.1&emails=getdefault&dbfields=subscription.suid.102)"
        R"(%2csubscription.login_rule.8%2csubscription.login_rule.2)"
        R"(%2chosts.db_id.-%2caccount_info.country.uid%2cuserphones.confirmed.uid)"
        R"(%2csubscription.suid.-%2csubscription.login.-&attributes=)"
    );
}

TEST(TestBuildAuthRequest, test_build_login_request_with_denyAuthForAssessors) {
    TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};
    TConfigPtr Config = MakeConfig();
    Config->Blackbox.denyAuthForAssessors = true;
    ASSERT_EQ(MakeLoginRequest(Config, info).Url,
        R"(?method=login&format=json&login=hello%40kitty.com&sid=smtp&authtype=smtp)"
        R"(&userip=127.0.0.1&emails=getdefault&dbfields=subscription.suid.102)"
        R"(%2csubscription.login_rule.8%2csubscription.login_rule.2)"
        R"(%2chosts.db_id.-%2caccount_info.country.uid%2cuserphones.confirmed.uid)"
        R"(%2csubscription.suid.-%2csubscription.login.-&attributes=189)"
    );
}

TEST(TestBuildAuthRequest, test_build_login_request_with_authSettingsOpts) {
    TAuthData info{"hello@kitty.com", "password", "", "127.0.0.1", "1010", "login"};
    TConfigPtr Config = MakeConfig();
    Config->AuthSettingsOptsUse = true;
    ASSERT_EQ(MakeLoginRequest(Config, info).Url,
        R"(?method=login&format=json&login=hello%40kitty.com&sid=smtp&authtype=smtp)"
        R"(&userip=127.0.0.1&emails=getdefault&dbfields=subscription.suid.102)"
        R"(%2csubscription.login_rule.8%2csubscription.login_rule.2)"
        R"(%2chosts.db_id.-%2caccount_info.country.uid%2cuserphones.confirmed.uid)"
        R"(%2csubscription.suid.-%2csubscription.login.-&attributes=107)"
    );
}

} // namespace anonymous
