#include <mail/nwsmtp/src/blackbox/bb_client_impl.h>

#include "bb_client_mock.h"

#include <mail/nwsmtp/src/utils.h>
#include <mail/nwsmtp/ut/init_log.h>
#include <mail/nwsmtp/ut/util/http_client_mock.h>
#include <mail/nwsmtp/ut/util/tvm2_module_mock.h>

#include <boost/date_time/c_local_time_adjustor.hpp>
#include <boost/date_time/posix_time/posix_time_types.hpp>
#include <gtest/gtest.h>

#include <ctime>

namespace {

std::time_t GetLocalTimeZoneOffset() {
    const auto utcNow = boost::posix_time::second_clock::universal_time();
    const auto localNow = boost::date_time::c_local_adjustor<boost::posix_time::ptime>::utc_to_local(utcNow);
    return (localNow - utcNow).total_seconds();
}

using namespace NNwSmtp;
using namespace NNwSmtp::NTests;
using namespace NNwSmtp::NBlackBox;
using namespace NNwSmtp::NBlackBox::NClient;

TEST(TestMakeRequest, ForEmptyBodyIsGet) {
    const auto result = MakeRequest({
        .Url = "?hello=darkness&my=old-friend",
    }, "myTvmTicket");
    const auto request = yhttp::request::GET(
        "/?hello=darkness&my=old-friend",
        {
            {"X-Ya-Service-Ticket", "myTvmTicket"},
        }
    );
    EXPECT_EQ(result, request);
}

TEST(TestMakeRequest, ForNonEmptyBodyIsPost) {
    const auto result = MakeRequest({
        .Url = "?hello=darkness&my=old-friend",
        .Body = "password=LoremIpsum",
    }, "myTvmTicket");
    const auto request = yhttp::request::POST(
        "/?hello=darkness&my=old-friend",
        {
            {"Content-Type", "application/x-www-form-urlencoded"},
            {"X-Ya-Service-Ticket", "myTvmTicket"},
        },
        "password=LoremIpsum");
    EXPECT_EQ(result, request);
}

struct TTestBbClient: Test {
    void SetUp() override {
        NTesting::InitGlobalLog();
    }

    void UserInfo(IBBClient::TResponseCallback callback) {
        BlackboxClient->UserInfo(Context, Request, std::move(callback));
        Io.run();
    }

    void Login(IBBClient::TResponseCallback callback) {
        BlackboxClient->Login(Context, Request, std::move(callback));
        Io.run();
    }

    void Oauth(IBBClient::TResponseCallback callback) {
        BlackboxClient->Oauth(Context, Request, std::move(callback));
        Io.run();
    }

    const std::shared_ptr<StrictMock<TClusterClientMock>> ClusterClient = GetStrictMockedClusterClient();
    const std::shared_ptr<StrictMock<TTvm2ModuleMock>> Tvm2Module = GetStrictMockedTvm2Module();
    const TBBClientPtr BlackboxClient = std::make_shared<TClientImpl>(ClusterClient, Tvm2Module, Io);
    const TContextPtr Context = boost::make_shared<TContext>("", "", "", "");

    const std::string TvmServiceName = "blackbox";
    const std::string TvmServiceTicket = "myTvmServiceTicket";
    const THttpRequest Request = THttpRequest {
        .Url = "?hello=darkness&my=old-friend",
    };
    const yhttp::request ClientRequest = yhttp::request::GET(
        "/?hello=darkness&my=old-friend",
        {
            {"X-Ya-Service-Ticket", TvmServiceTicket},
        }
    );

    boost::asio::io_context Io;
};

TEST_F(TTestBbClient, ForShouldPassTvmModuleError) {
    const TErrorCode tvm2ModuleError = ymod_httpclient::http_error::no_service_ticket;

    InSequence seq;
    EXPECT_CALL(*Tvm2Module, get_service_ticket(TvmServiceName, _))
        .WillOnce(Return(tvm2ModuleError));

    UserInfo([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, ymod_httpclient::http_error::no_service_ticket);
    });
}

TEST_F(TTestBbClient, ForShouldPassRequestError) {
    const TErrorCode clientError = ymod_httpclient::http_error::code::ssl_error;

    InSequence seq;
    EXPECT_CALL(*Tvm2Module, get_service_ticket(TvmServiceName, _))
        .WillOnce(DoAll(SetArgReferee<1>(TvmServiceTicket), Return(TErrorCode {})));
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(clientError, yhttp::response {}));

    UserInfo([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, ymod_httpclient::http_error::code::ssl_error);
    });
}

TEST_F(TTestBbClient, ForShouldPassStatusIfNotOk) {
    const auto clientResponse = yhttp::response {
        .status = 500,
    };

    InSequence seq;
    EXPECT_CALL(*Tvm2Module, get_service_ticket(TvmServiceName, _))
        .WillOnce(DoAll(SetArgReferee<1>(TvmServiceTicket), Return(TErrorCode {})));
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    UserInfo([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, ymod_httpclient::http_error::server_status_error);
    });
}

TEST_F(TTestBbClient, ForShouldBeUnableToParseBrokenJsonResponse) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"broken":"json")",
    };

    InSequence seq;
    EXPECT_CALL(*Tvm2Module, get_service_ticket(TvmServiceName, _))
        .WillOnce(DoAll(SetArgReferee<1>(TvmServiceTicket), Return(TErrorCode {})));
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    UserInfo([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, ymod_httpclient::http_error::parse_response_error);
    });
}

TEST_F(TTestBbClient, ForShouldPassStatusIfOkAndExceptionInResponse) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"error":"BlackBox error: Missing userip argument","exception":{"id":2,"value":"INVALID_PARAMS"}})",
    };

    InSequence seq;
    EXPECT_CALL(*Tvm2Module, get_service_ticket(TvmServiceName, _))
        .WillOnce(DoAll(SetArgReferee<1>(TvmServiceTicket), Return(TErrorCode {})));
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    Login([](auto errc, auto) {
        ASSERT_TRUE(errc);
        EXPECT_EQ(errc, ymod_httpclient::http_error::server_response_error);
    });
}

TEST_F(TTestBbClient, ForGoodUserInfoResponse) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"users":[{"id":"88005553535","uid":{"value":"88005553535","lite":false,)"
                R"("hosted":true,"domid":"5553535","domain":"my.old.friend","mx":"1",)"
                R"("domain_ena":"1","catch_all":false},"login":"hello_darkness@my.old.friend",)"
                R"("have_password":true,"have_hint":true,"karma":{"value":85},"karma_status":{)"
                R"("value":0},"dbfields":{"account_info.country.uid":"ru",)"
                R"("account_info.reg_date.uid":"2017-12-11 18:55:31","subscription.born_date.2":)"
                R"("2017-12-11 18:55:31","subscription.login.-":"hello_darkness@my.old.friend",)"
                R"("subscription.login_rule.2":"1","subscription.suid.-":"5552725",)"
                R"("userphones.confirmed.uid":"2017-12-11 18:55:31","subscription.suid.1000":""},)"
                R"("attributes":{"1031":"1234567"},"address-list":[{"address":"hello_darkness@my.old.friend",)"
                R"("validated":true,"default":true,"rpop":false,"silent":false,"unsafe":false,"native":true,)"
                R"("born-date":"2017-12-11 18:55:31"}]}]})",
    };
    const auto response = TResponse {
        .Login = "hello_darkness@my.old.friend",
        .Suid = 5552725,
        .Uid = "88005553535",
        .DefaultEmail = "hello_darkness@my.old.friend",
        .Hosted = true,
        .CatchAll = true,
        .Domain = "my.old.friend",
        .Country = "ru",
        .OrgId = "1234567",
        .PhoneConfirmed = true,
        .IsCorpList = true,
        .RegistrationDate = 1513018531 - GetLocalTimeZoneOffset(),
        .BornDate = 1513018531 - GetLocalTimeZoneOffset(),
        .Karma = 85,
    };

    InSequence seq;
    EXPECT_CALL(*Tvm2Module, get_service_ticket(TvmServiceName, _))
        .WillOnce(DoAll(SetArgReferee<1>(TvmServiceTicket), Return(TErrorCode {})));
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    UserInfo([&](auto errc, auto result) {
        ASSERT_FALSE(errc);
        EXPECT_EQ(result, response);
    });
}

TEST_F(TTestBbClient, ForGoodLoginResponse) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"status":{"value":"VALID","id":0},"error":"OK","uid":{"value":"88005553535",)"
                R"("lite":false,"hosted":true,"domid":"5553535","domain":"my.old.friend","mx":"1",)"
                R"("domain_ena":"","catch_all":true},"login":"hello_darkness@my.old.friend",)"
                R"("have_password":true,"have_hint":true,"karma":{"value":85,"allow-until":1513018531},)"
                R"("karma_status":{"value":0},"dbfields":{"account_info.country.uid":"ru","hosts.db_id.-":"pg",)"
                R"("subscription.login.-":"hello_darkness@my.old.friend","subscription.login_rule.2":"0",)"
                R"("subscription.login_rule.8":"1","subscription.suid.-":"5552725","subscription.suid.102":"1",)"
                R"("userphones.confirmed.uid":"2017-12-11 18:55:31"},"attributes":{"13":"1","107":"1"},)"
                R"("address-list":[{"address":"hello_darkness@my.old.friend","validated":true,)"
                R"("default":true,"rpop":false,"silent":false,"unsafe":false,"native":true,)"
                R"("born-date":"2017-12-11 18:55:31"}]})",
    };
    const auto response = TResponse {
        .Login = "hello_darkness@my.old.friend",
        .Suid = 5552725,
        .Uid = "88005553535",
        .DefaultEmail = "hello_darkness@my.old.friend",
        .Hosted = true,
        .CatchAll = true,
        .Domain = "my.old.friend",
        .Country = "ru",
        .PhoneConfirmed = true,
        .IsMailList = true,
        .IsBlocked = true,
        .HasPDDsid = true,
        .AppPasswordEnabled = true,
        .Mdb = "pg",
        .Karma = 85,
        .KarmaBanTime = 1513018531,
        .AuthSuccess = true,
    };

    InSequence seq;
    EXPECT_CALL(*Tvm2Module, get_service_ticket(TvmServiceName, _))
        .WillOnce(DoAll(SetArgReferee<1>(TvmServiceTicket), Return(TErrorCode {})));
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    Login([&](auto errc, auto result) {
        ASSERT_FALSE(errc);
        EXPECT_EQ(result, response);
    });
}

TEST_F(TTestBbClient, ForGoodOauthResponse) {
    const auto clientResponse = yhttp::response {
        .status = 200,
        .body = R"({"oauth":{"uid":"88005553535","token_id":"3332166863","device_id":"",)"
                R"("device_name":"","scope":"login:birthday login:email mail:smtp",)"
                R"("ctime":"2017-12-11 18:55:31","issue_time":"2017-12-11 18:55:31",)"
                R"("expire_time":"2018-12-11 18:55:31","is_ttl_refreshable":true,"client_id":"abcdefg",)"
                R"("client_name":"MyClient","client_icon":"","client_homepage":"",)"
                R"("client_ctime":"2017-12-11 18:55:31","client_is_yandex":false,"xtoken_id":"",)"
                R"("meta":""},"status":{"value":"VALID","id":0},"error":"OK",)"
                R"("uid":{"value":"88005553535","lite":false,"hosted":false},)"
                R"("login":"hello_darkness","have_password":true,"have_hint":false,)"
                R"("karma":{"value":0},"karma_status":{"value":6000},"dbfields":{)"
                R"("account_info.country.uid":"ru","hosts.db_id.-":"pg",)"
                R"("subscription.login.2":"hello_darkness","subscription.login_rule.2":"1",)"
                R"("subscription.login_rule.8":"4","subscription.suid.102":"1",)"
                R"("subscription.suid.2":"5552725","userphones.confirmed.uid":"2017-12-11 18:55:31"},)"
                R"("attributes":{"189":"1"},"address-list":[{"address":"hello_darkness@my.old.friend",)"
                R"("validated":true,"default":true,"rpop":false,"silent":false,"unsafe":false,)"
                R"("native":true,"born-date":"2017-12-11 18:55:31"}],"connection_id":"t:5555566666"})",
    };
    const auto response = TResponse {
        .Login = "hello_darkness",
        .Suid = 5552725,
        .Uid = "88005553535",
        .DefaultEmail = "hello_darkness@my.old.friend",
        .Country = "ru",
        .PhoneConfirmed = true,
        .IsBlocked = true,
        .IsAssessor = true,
        .HasPDDsid = true,
        .Mdb = "pg",
        .KarmaStatus = 6000,
        .Scopes = {
            "login:birthday",
            "login:email",
            "mail:smtp",
        },
        .AuthSuccess = true,
    };

    InSequence seq;
    EXPECT_CALL(*Tvm2Module, get_service_ticket(TvmServiceName, _))
        .WillOnce(DoAll(SetArgReferee<1>(TvmServiceTicket), Return(TErrorCode {})));
    EXPECT_CALL(*ClusterClient, async_run(_, ClientRequest, _, _))
        .WillOnce(InvokeArgument<3>(TErrorCode {}, clientResponse));

    Oauth([&](auto errc, auto result) {
        ASSERT_FALSE(errc);
        EXPECT_EQ(result, response);
    });
}

} // namespace anonymous
