#include <mail/nwsmtp/src/rbl/http_client.h>
#include <mail/nwsmtp/ut/util/http_client_mock.h>
#include <ymod_httpclient/errors.h>

#include <gtest/gtest.h>

using TResponse = yhttp::response;

using NNwSmtp::TErrorCode;

using namespace ::testing;
using namespace NNwSmtp;

namespace {

struct TTestRbl : public Test {
    TTestRbl()
        : Ctx(boost::make_shared<TContext>("conn", "env", "cluster", "host"))
    {
        NNwSmtp::glog = std::make_shared<TLog>();
    }

    static TResponse MakeResponse(std::int32_t status, std::string body) {
        TResponse resp;
        resp.status = status;
        resp.body = std::move(body);
        return resp;
    }

    void CheckRblClient(NSO::NRBL::IClientPtr client, std::string ip, NSO::NRBL::TCallback cb) {
         client->AsyncCheck(Ctx, std::move(ip), std::move(cb));
         IoContext.run();
    }

    boost::asio::io_context IoContext;
    TContextPtr Ctx;
};

struct TTestHttpRbl : public TTestRbl, public WithParamInterface<std::tuple<std::string /* responseBody */, bool /* isSpam */>>{
    TTestHttpRbl()
        : HttpClient(std::make_shared<NSO::NRBL::THttpClient>(ClusterCall, IoContext))
    {
    }

    std::shared_ptr<NTests::TClusterClientMock> ClusterCall{NTests::GetStrictMockedClusterClient()};
    NSO::NRBL::THttpClientPtr HttpClient;
};

} // namespace

TEST_P(TTestHttpRbl, for_response_200_and_should_be_ok_and_correct_found_in_spam) {
    const auto& [responseBody, isSpam] = GetParam();
    EXPECT_CALL(*ClusterCall, async_run(_, _, _, _)).WillOnce(
        InvokeArgument<3>(yhttp::errc::success, MakeResponse(200, responseBody))
    );

    CheckRblClient(HttpClient, "1.2.3.4", [isSpam = isSpam](TErrorCode ec, bool foundInSpam) {
        EXPECT_EQ(ec, NSO::NRBL::make_error_code(NSO::NRBL::EError::Ok));
        EXPECT_EQ(foundInSpam, isSpam);
    });
}

INSTANTIATE_TEST_SUITE_P(Check200, TTestHttpRbl, 
    Values( 
        std::make_tuple(R"({"infos": {}, "checks": {"spamsource": true}})", true),
        std::make_tuple(R"({"infos": {}, "checks": {"spamsource": false}})", false)
    )
);

TEST_F(TTestHttpRbl, for_response_200_and_invalid_body_should_be_parsing_error) {
    EXPECT_CALL(*ClusterCall, async_run(_, _, _, _)).WillOnce(
        InvokeArgument<3>(yhttp::errc::success, MakeResponse(200, R"({"infos": {}, "checks": {"combined-bl": }})"))
    );

    CheckRblClient(HttpClient, "1.2.3.4", [](TErrorCode ec, bool foundInSpam) {
        EXPECT_EQ(ec, NSO::NRBL::make_error_code(NSO::NRBL::EError::ResponseParseError));
        EXPECT_FALSE(foundInSpam);
    });
}

TEST_F(TTestHttpRbl, for_response_503_should_be_error_failed) {
    EXPECT_CALL(*ClusterCall, async_run(_, _, _, _)).WillOnce(
        InvokeArgument<3>(yhttp::errc::success, MakeResponse(503, R"(Too many requests)"))
    );

    CheckRblClient(HttpClient, "1.2.3.4", [](TErrorCode ec, bool foundInSpam) {
        EXPECT_EQ(ec, NSO::NRBL::make_error_code(NSO::NRBL::EError::Failed));
        EXPECT_FALSE(foundInSpam);
    });
}
