#include <mail/nwsmtp/src/delivery/sync/hint.h>
#include <mail/nwsmtp/src/utils.h>

#include <gtest/gtest.h>

using namespace NNwSmtp::NDlv;

using NNwSmtp::THeaderStorage;
using NNwSmtp::TBufferRange;

using NNwSmtp::NHint::TParamsContainer;

const TParamsContainer DEFAULT_SYNC_HINT{
    {"filters", {"0"}},
    {"sync_dlv", {"1"}},
    {"disable_push", {"1"}},
    {"allow_duplicates", {"0"}},
    {"skip_loop_prevention", {"1"}},
    {"no_such_fid_fail", {"1"}},
    {"imap", {"0"}},
    {"skip_meta_msg", {"0"}}
};

TEST(WriteLablesToHint, Success) {
    TParamsContainer params;
    TLabels labels{
        .System = std::vector<std::string>{"news"},
        .Imap = std::vector<std::string>{"RED"},
        .Symbol = std::vector<std::string>{"seen"},
        .Lids = std::vector<std::string>{"1"},
        .User = std::vector<std::string>{"userlabel1"}
    };
    WriteToHint(labels, params);
    ASSERT_EQ(params, TParamsContainer({
        {"label", {"SystMetkaSO:news"}},
        {"imaplabel", {"RED"}},
        {"label", {"symbol:seen"}},
        {"lid", {"1"}},
        {"userlabel", {"userlabel1"}}
    }));
}

TEST(WriteMailInfoToHint, Success) {
    TParamsContainer params;
    TMailInfo mailInfo{
        .Fid = "1",
        .FolderPath = "path",
        .FolderSpamPath = "spam_path",
        .FolderPathDelim = "path_delim",
        .ReceivedDate = 2,
        .Labels = TLabels{},
        .ExternalImapId = "123456",
        .OldMid = "100500"
    };
    WriteToHint(mailInfo, params);
    ASSERT_EQ(params, TParamsContainer({
        {"fid", {"1"}},
        {"received_date", {"2"}},
        {"external_imap_id", {"123456"}},
        {"mid", {"100500"}}
    }));
}

TEST(WriteMailInfoToHintWithFolderPath, Success) {
    TParamsContainer params;
    TMailInfo mailInfo{
        .FolderPath = "path",
        .FolderSpamPath = "spam_path",
        .FolderPathDelim = "path_delim",
    };
    WriteToHint(mailInfo, params);
    ASSERT_EQ(params, TParamsContainer({
        {"folder_path", "path"},
        {"folder_path_delim", "path_delim"},
        {"folder_spam_path", "spam_path"},
    }));
}

TEST(WriteOptionsToHint, Success) {
    TParamsContainer params;
    TOptions options{
        .EnablePush = true,
        .DetectSpam = true,
        .DetectVirus = true,
        .DetectCycle = true,
        .UseFilters = true,
        .AllowDuplicates = true,
        .SkipMetaMsg = true
    };
    WriteToHint(options, params);
    ASSERT_EQ(params, TParamsContainer({
        {"disable_push", {"0"}},
        {"skip_loop_prevention", {"1"}},
        {"filters", {"1"}},
        {"allow_duplicates", {"1"}},
        {"imap", {"0"}},
        {"skip_meta_msg", {"1"}}
    }));
}

TEST(BuildHint, DefaultFieldsSuccess) {
    auto hint = BuildHint({}, {.Fid = "1"}, {});
    auto expectedHint = DEFAULT_SYNC_HINT;
    expectedHint.emplace("fid", "1");
    ASSERT_EQ(hint, expectedHint);
}

TEST(BuildHint, WriteOptionsMailInfoSoResultSuccess) {
    TOptions options{
        .EnablePush = true,
    };
    TLabels labels{
        .System = std::vector<std::string>{"news"},
    };
    TMailInfo mailInfo{
        .Fid = "1",
        .Labels = labels,
        .ExternalImapId = "123",
        .OldMid = "100500"
    };

    NNwSmtp::NSO::TResponse soResponse{};
    soResponse.SoClasses = {"domain_vtnrf0vkcom", "people"};

    auto hint = BuildHint(options, mailInfo, {soResponse});

    auto expectedHint = DEFAULT_SYNC_HINT;
    expectedHint.emplace("fid", "1");
    expectedHint.emplace("label", "SystMetkaSO:news");
    expectedHint.emplace("label", "domain_vtnrf0vkcom");
    expectedHint.emplace("label", "SystMetkaSO:people");
    expectedHint.emplace("external_imap_id", "123");
    expectedHint.emplace("mid", "100500");
    auto iter = expectedHint.find("disable_push");
    ASSERT_NE(iter, expectedHint.end());
    iter->second = "0";
    ASSERT_EQ(hint, expectedHint);
}
