#include <mail/nwsmtp/src/so/types_reflection.h>

#include <mail/nwsmtp/src/delivery/async/so.h>
#include <mail/nwsmtp/src/delivery/async/error_code.h>
#include <mail/nwsmtp/src/delivery/async/read_headers.cpp>
#include <mail/nwsmtp/src/dmarc.h>
#include <mail/nwsmtp/src/bb_result.h>
#include <mail/nwsmtp/src/envelope.h>
#include <mail/nwsmtp/src/types.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <boost/make_shared.hpp>

#include <memory>
#include <optional>
#include <set>

namespace NNwSmtp::NSO {

static bool operator ==(const TConnectInfo& lhs, const TConnectInfo& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

static bool operator ==(const TEmail& lhs, const TEmail& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

static bool operator ==(const TEmailInfo& lhs, const TEmailInfo& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

static bool operator ==(const TEnvelope& lhs, const TEnvelope& rhs) {
    return lhs.ConnectInfo == rhs.ConnectInfo
        && lhs.EmailType == rhs.EmailType
        && lhs.MailFrom == rhs.MailFrom
        && lhs.Rcpts == rhs.Rcpts;
}

static bool operator ==(const TRequest& lhs, const TRequest& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

} // namespace NNwSmtp::NSO

namespace {

using namespace testing;
using namespace NNwSmtp;
using namespace NNwSmtp::NAsyncDlv;

NSO::TOptions MakeConfig() {
    NSO::TOptions options;
    options.ClusterName = "cluster_name";
    options.InternalHeaders = std::unordered_set<std::string>{"header-1", "header-2"};
    options.RejectForInactive = true;
    options.InactivityThreshold = 20;
    options.RejectedTypes = {{"rejected_types1"}, {"rejected_types2"}};
    options.AddXspamFlag = true;
    options.ReplyTextMalicious = "malicious";
    options.ReplyTextDiscard = "discard";
    options.ReplyTextSpam = "spam";
    options.DryRun = true;
    return options;
}

void MakeEnvelope(envelope_ptr envelope, const NNwSmtp::TBuffer& message) {
    envelope->orig_message_ = message;

    auto [_, iter] = parse_header(
        TBufferRange{envelope->orig_message_.begin(), envelope->orig_message_.end()},
        [&envelope]
        (const auto& name, const auto&, const auto& value) {
            envelope->header_storage_->Add(name, value);
        }
    );
    envelope->m_sender = "sender";
    envelope->m_sender_uid = "sender_uid";
    envelope->sender_added_for_control = true;
    envelope->orig_message_body_beg_ = iter;
    envelope->added_message_id_ = true;
    envelope->message_id_ = "message_id";
    envelope->m_id = "id";
    envelope->senderInfo = TBlackBoxResult {
        .Suid = 228,
        .Karma = 1,
        .KarmaStatus = 2,
        .Country = "country",
        .Uid = "sender_uid",
    };
    envelope->add_recipient("rcpt1", 301, false, "rcpt_uid1", "", "", "country1",
        DomainType::LOCAL, dsn::Options::NEVER, false, true, 1000);
    envelope->add_recipient("rcpt2", 302, false, "rcpt_uid2", "", "", "country2",
        DomainType::LOCAL, dsn::Options::NEVER, false, false, 1001);
    envelope->add_recipient("rcpt3", 303, false, "sender_uid", "", "", "country3",
        DomainType::LOCAL, dsn::Options::NEVER, false, false, 1002);
}

NSO::TResponse MakeSoResponse() {
    return {
        .Resolution = NSO::EResolution::SO_RESOLUTION_SPAM,
        .DenyGraylist = true,
        .OutParameters = NSO::TOutParameters {
            .Type = NSO::EForwardType::FORWARD_TYPE_WHITE,
            .ForeignMx = true
        },
        .ActivityInfos = std::vector<NSO::TActivityInfo> {
            {
                .Uid = "rcpt_uid1",
                .Status = 10,
            },
            {
                .Uid = "rcpt_uid2",
                .Status = 100,
            },
            {
                .Uid = "rcpt_uid3",
                .Status = 1000,
            },
        },
        .SoClasses = {"class1", "class2", "domain_class"},
        .PersonalResolutions = std::vector<NSO::TPersonalResolution> {
            {
                .Uid = "rcpt_uid1",
                .Resolution = NSO::EResolution::SO_RESOLUTION_ACCEPT,
                .SoClasses = {"class3", "class4"}
            },
            {
                .Uid = "rcpt_uid2",
                .Resolution = NSO::EResolution::SO_RESOLUTION_REJECT,
                .SoClasses = {"class5", "class6", "pf_spam"}
            },
        }
    };
}

struct TTestSOAsyncDelivery: Test {
    void SetUp() override {
        MakeEnvelope(Envelope, Message);
    }

    NSO::TOptions Options = MakeConfig();
    NNwSmtp::TBuffer Message = NUtil::MakeSegment(
        "Header-1: 1\r\nHeader-2: 2\r\nHeader-3: 3\r\n\r\nBody"
    );
    envelope_ptr Envelope = boost::make_shared<envelope>();
    NSO::TResponse SOResponse = MakeSoResponse();
};

TEST_F(TTestSOAsyncDelivery, for_build_so_headers_should_return_so_headers) {
    Envelope->senderInfo = {};
    EXPECT_EQ(
        NSO::BuildSoHeaders(
            {
                .SoClusterName = Options.ClusterName,
                .Spf = "spf_result (spf_expl) envelope-from=smtp_from",
                .Dmarc = "quarantine dmarc_domain",
                .AuthResults = "auth_results",
                .MessageId = Envelope->message_id_,
                .Headers = *Envelope->header_storage_,
                .InternalHeaders = Options.InternalHeaders,
            },
            "session_id-id"
        ),
        "X-Yandex-QueueID: session_id-id\r\n"
        "X-Yandex-So-Front: cluster_name\r\n"
        "Received-SPF: spf_result (spf_expl) envelope-from=smtp_from\r\n"
        "X-DMARC: quarantine dmarc_domain\r\n"
        "X-Yandex-Authentication-Results: auth_results\r\n"
        "Message-Id: message_id\r\n"
        "Header-3: 3\r\n"
    );
}

NSO::TEmailInfo MakeSoEmailInfo() {
    return {
        .Address = {.Email = "sender"},
        .Uid = "sender_uid",
        .Suid = "228",
        .Country = "country",
        .Karma = "1",
        .KarmaStatus = "2"
    };
}

std::vector<NSO::TEmailInfo> MakeSoRecipients() {
    return {
        {
            .Address = {.Email = "rcpt1"},
            .Uid = "rcpt_uid1",
            .Suid = "301",
            .Country = "country1",
            .IsMaillist = true,
        },
        {
            .Address = {.Email = "rcpt2"},
            .Uid = "rcpt_uid2",
            .Suid = "302",
            .Country = "country2",
            .IsMaillist = false,
        },
    };
}

TEST_F(TTestSOAsyncDelivery, for_build_so_mail_from_return_so_mail_from) {
    EXPECT_EQ(
        BuildMailFrom(Envelope),
        MakeSoEmailInfo()
    );
}

TEST_F(TTestSOAsyncDelivery, for_build_so_recipients_return_so_recipients) {
    EXPECT_EQ(
        BuildRecipients(Envelope),
        MakeSoRecipients()
    );
}

NSO::TRequest MakeSoHttpRequest() {
    return {
        NSO::TEnvelope {
            .ConnectInfo = {
                .Host = "remote_host",
                .Ip = std::string("\xc0\x00\x02\xeb", 4),
                .Domain = "hello_host",
                .SessionId = "session_id-id"
            },
            .MailFrom = MakeSoEmailInfo(),
            .Rcpts = MakeSoRecipients(),
        },
        "X-Yandex-QueueID: session_id-id\r\n"
        "X-Yandex-So-Front: cluster_name\r\n"
        "X-DMARC: quarantine dmarc_domain\r\n"
        "X-Yandex-Authentication-Results: auth_results\r\n"
        "Message-Id: message_id\r\n"
        "Header-3: 3\r\n"
        "\r\n"
        "Body",
        true
    };
}

TEST_F(TTestSOAsyncDelivery, for_build_so_http_request_should_return_so_http_request) {
    auto request = MakeSoHttpRequest();
    EXPECT_THAT(
        BuildSoRequest(
            {
                .HeloHost = "hello_host",
                .SessionId = "session_id",
                .SmtpFrom = "smtp_from",
                .RemoteHost = "remote_host",
                .Envelope = Envelope,
                .Ip = boost::asio::ip::make_address_v4("192.0.2.235"),
                .SpfResult = "spf_result",
                .SpfExpl = "spf_expl",
            },
            "auth_results",
            dmarc::record::QUARANTINE,
            "dmarc_domain",
            Options
        ),
        Pointee(request)
    );
}

TRcpt MakeRcpt(
    bool isMaillist,
    std::time_t date,
    std::uint32_t inactiveFor,
    NSO::EResolution spamStatus
) {
    auto rcpt = TRcpt {"email", "uid"};
    rcpt.registrationDate = date;
    rcpt.is_maillist_ = isMaillist;
    rcpt.inactive_for_ = inactiveFor;
    rcpt.m_spam_status = spamStatus;
    return rcpt;
}

TEST_F(TTestSOAsyncDelivery, for_check_inactivity_for_maillist_recipient_should_return_true) {
    bool isMaillist = true;
    auto registrationDate = 0;
    auto inactivity = Options.InactivityThreshold;
    auto spamStatus = NSO::EResolution::SO_RESOLUTION_ACCEPT;
    EXPECT_TRUE(CheckInactivity(MakeRcpt(isMaillist, registrationDate, inactivity, spamStatus), {}, Options));
}

TEST_F(TTestSOAsyncDelivery, for_check_inactivity_for_new_recipient_should_return_true) {
    bool isMaillist = false;
    auto registrationDate = std::numeric_limits<std::time_t>::max();
    auto inactivity = Options.InactivityThreshold;
    auto spamStatus = NSO::EResolution::SO_RESOLUTION_ACCEPT;
    EXPECT_TRUE(CheckInactivity(MakeRcpt(isMaillist, registrationDate, inactivity, spamStatus), {}, Options));
}

TEST_F(TTestSOAsyncDelivery,
        for_check_inactivity_for_recipient_with_inactive_less_than_threshold_should_return_true) {
    bool isMaillist = false;
    auto registrationDate = 0;
    auto inactivity = Options.InactivityThreshold - 1;
    auto spamStatus = NSO::EResolution::SO_RESOLUTION_ACCEPT;
    EXPECT_TRUE(CheckInactivity(MakeRcpt(isMaillist, registrationDate, inactivity, spamStatus), {}, Options));
}

TEST_F(TTestSOAsyncDelivery, for_check_inactivity_for_recipient_with_spam_resolution_should_return_false) {
    bool isMaillist = false;
    auto registrationDate = 0;
    auto inactivity = Options.InactivityThreshold + 1;
    auto spamStatus = NSO::EResolution::SO_RESOLUTION_SPAM;
    EXPECT_FALSE(CheckInactivity(MakeRcpt(isMaillist, registrationDate, inactivity, spamStatus), {}, Options));
}

TEST_F(TTestSOAsyncDelivery, for_check_inactivity_for_recipient_with_malicious_spam_resolution_should_return_false) {
    bool isMaillist = false;
    auto registrationDate = 0;
    auto inactivity = Options.InactivityThreshold + 1;
    auto spamStatus = NSO::EResolution::SO_RESOLUTION_REJECT;
    EXPECT_FALSE(CheckInactivity(MakeRcpt(isMaillist, registrationDate, inactivity, spamStatus), {}, Options));
}

TEST_F(TTestSOAsyncDelivery, for_check_inactivity_for_recipient_with_rejected_types_should_return_false) {
    bool isMaillist = false;
    auto registrationDate = 0;
    auto inactivity = Options.InactivityThreshold + 1;
    auto spamStatus = NSO::EResolution::SO_RESOLUTION_ACCEPT;
    std::set<std::string> types = {"rejected_types1"};
    EXPECT_FALSE(CheckInactivity(MakeRcpt(isMaillist, registrationDate, inactivity, spamStatus), types, Options));
}

TEST_F(TTestSOAsyncDelivery, for_false_reject_inactive_option_for_recipient_should_return_true) {
    bool isMaillist = false;
    auto registrationDate = 0;
    auto inactivity = Options.InactivityThreshold + 1;
    auto spamStatus = NSO::EResolution::SO_RESOLUTION_ACCEPT;
    std::set<std::string> types = {"rejected_types1"};
    Options.RejectForInactive = false;
    EXPECT_TRUE(CheckInactivity(MakeRcpt(isMaillist, registrationDate, inactivity, spamStatus), types, Options));
}

TEST_F(TTestSOAsyncDelivery, test_check_inactives) {
    auto* rcpt1 = Envelope->m_rcpts.GetByUid("rcpt_uid1");
    auto* rcpt2 = Envelope->m_rcpts.GetByUid("rcpt_uid2");
    rcpt2->m_spam_status = NSO::EResolution::SO_RESOLUTION_SPAM;
    CheckInactives(SOResponse, Envelope, Options);
    ASSERT_TRUE(rcpt1);
    ASSERT_TRUE(rcpt2);
    EXPECT_EQ(rcpt1->inactive_for_, 10u);
    EXPECT_EQ(rcpt1->m_delivery_status, check::CHK_TEMPFAIL);
    EXPECT_EQ(rcpt2->inactive_for_, 100u);
    EXPECT_EQ(rcpt2->m_delivery_status, check::CHK_REJECT);
}

TEST_F(TTestSOAsyncDelivery, for_invalid_so_resolution_should_return_so_temp_fail_error_code) {
    const auto& [ec, answer] = MakeSoStatus(
        NSO::EResolution::SO_RESOLUTION_INVALID,
        {"class3", "class4"},
        "dmarc_domain",
        dmarc::record::QUARANTINE,
        Options,
        false
    );
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::SoTempFail);
    EXPECT_EQ(answer, "451 4.7.1 Service unavailable - try again later");
}

TEST_F(TTestSOAsyncDelivery,
        for_reject_so_resolution_and_malicious_rejected_and_dmarc_reject_should_return_malicious_rejected_error_code) {
    Options.ActionMalicious = NSO::ESpamPolicy::REJECT;
    const auto& [ec, answer] = MakeSoStatus(
        NSO::EResolution::SO_RESOLUTION_REJECT,
        {"class3", "class4"},
        "dmarc_domain",
        dmarc::record::REJECT,
        Options,
        false
    );
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::MaliciousRejected);
    EXPECT_EQ(answer, "550 5.7.1 Email rejected per DMARC policy for dmarc_domain");
}

TEST_F(TTestSOAsyncDelivery,
        for_reject_so_resolution_and_malicious_rejected_should_return_malicious_rejected_error_code) {
    Options.ActionMalicious = NSO::ESpamPolicy::REJECT;
    const auto& [ec, answer] = MakeSoStatus(
        NSO::EResolution::SO_RESOLUTION_REJECT,
        {"class3", "class4"},
        "dmarc_domain",
        dmarc::record::QUARANTINE,
        Options,
        false
    );
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::MaliciousRejected);
    EXPECT_EQ(answer, "554 5.7.1 malicious");
}

TEST_F(TTestSOAsyncDelivery,
        for_reject_so_resolution_and_malicious_rejected_with_so_type_res_url_rbl_should_return_url_rbl_rejected_error_code) {
    Options.ActionMalicious = NSO::ESpamPolicy::REJECT;
    const auto& [ec, answer] = MakeSoStatus(
        NSO::EResolution::SO_RESOLUTION_REJECT,
        {"res_url_rbl"},
        "dmarc_domain",
        dmarc::record::QUARANTINE,
        Options,
        false
    );
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::UrlRblRejected);
    EXPECT_EQ(answer, "554 5.7.1 malicious");
}

TEST_F(TTestSOAsyncDelivery,
        for_reject_so_resolution_and_malicious_discarded_should_return_malicious_discarded_error_code) {
    Options.ActionMalicious = NSO::ESpamPolicy::DISCARD;
    const auto& [ec, answer] = MakeSoStatus(
        NSO::EResolution::SO_RESOLUTION_REJECT,
        {"class3", "class4"},
        "dmarc_domain",
        dmarc::record::QUARANTINE,
        Options,
        false
    );
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::MaliciousDiscarded);
    EXPECT_EQ(answer, "250 2.0.0 discard");
}

TEST_F(TTestSOAsyncDelivery, for_spam_so_resolution_and_captcha_and_spam_reject_should_no_return_error_code) {
    Options.ActionSpam = NSO::ESpamPolicy::REJECT;
    const auto& [ec, answer] = MakeSoStatus(
        NSO::EResolution::SO_RESOLUTION_SPAM,
        {"class3", "class4"},
        "dmarc_domain",
        dmarc::record::REJECT,
        Options,
        true
    );
    ASSERT_FALSE(ec);
}

TEST_F(TTestSOAsyncDelivery,
        for_spam_so_resolution_and_not_captcha_and_spam_reject_and_dmarc_reject_should_return_spam_rejected_error_code) {
    Options.ActionSpam = NSO::ESpamPolicy::REJECT;
    const auto& [ec, answer] = MakeSoStatus(
        NSO::EResolution::SO_RESOLUTION_SPAM,
        {"class3", "class4"},
        "dmarc_domain",
        dmarc::record::REJECT,
        Options,
        false
    );
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::SpamRejected);
    EXPECT_EQ(answer, "550 5.7.1 Email rejected per DMARC policy for dmarc_domain");
}

TEST_F(TTestSOAsyncDelivery,
        for_spam_so_resolution_and_not_captcha_and_spam_reject_should_return_spam_rejected_error_code) {
    Options.ActionSpam = NSO::ESpamPolicy::REJECT;
    const auto& [ec, answer] = MakeSoStatus(
        NSO::EResolution::SO_RESOLUTION_SPAM,
        {"class3", "class4"},
        "dmarc_domain",
        dmarc::record::QUARANTINE,
        Options,
        false
    );
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::SpamRejected);
    EXPECT_EQ(answer, "554 5.7.1 spam");
}

TEST_F(TTestSOAsyncDelivery,
        for_spam_so_resolution_and_not_captcha_and_spam_discarded_should_return_spam_discard_error_code) {
    Options.ActionSpam = NSO::ESpamPolicy::DISCARD;
    const auto& [ec, answer] = MakeSoStatus(
        NSO::EResolution::SO_RESOLUTION_SPAM,
        {"class3", "class4"},
        "dmarc_domain",
        dmarc::record::QUARANTINE,
        Options,
        false
    );
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::SpamDiscarded);
    EXPECT_EQ(answer, "250 2.0.0 discard");
}

TEST_F(TTestSOAsyncDelivery,
        for_spam_so_resolution_and_not_captcha_and_field_mark_spam_should_no_return_error_code) {
    Options.ActionSpam = NSO::ESpamPolicy::FIELD_MARK;
    const auto& [ec, answer] = MakeSoStatus(
        NSO::EResolution::SO_RESOLUTION_SPAM,
        {"class3", "class4"},
        "dmarc_domain",
        dmarc::record::QUARANTINE,
        Options,
        false
    );
    ASSERT_FALSE(ec);
}

TEST_F(TTestSOAsyncDelivery, for_accept_so_resolution_should_no_return_error_code) {
    const auto& [ec, answer] = MakeSoStatus(
        NSO::EResolution::SO_RESOLUTION_ACCEPT,
        {"class3", "class4"},
        "dmarc_domain",
        dmarc::record::QUARANTINE,
        Options,
        false
    );
    ASSERT_FALSE(ec);
}

TEST_F(TTestSOAsyncDelivery, test_make_so_personal_status) {
    MakeSoPersonalStatus(SOResponse, Envelope);
    EXPECT_THAT(
        Envelope->xyandex_hint_header_value_personal_,
        ElementsAre(
            "email=rcpt1\nreplace_so_labels=1\nlabel=SystMetkaSO:class3\nlabel=SystMetkaSO:class4\n",
            "email=rcpt2\nreplace_so_labels=1\nlabel=SystMetkaSO:class5\nlabel=SystMetkaSO:class6\nlabel=SystMetkaSO:pf_spam\n"
        )
    );
    auto* rcpt1 = Envelope->m_rcpts.GetByUid("rcpt_uid1");
    auto* rcpt2 = Envelope->m_rcpts.GetByUid("rcpt_uid2");
    ASSERT_TRUE(rcpt1);
    ASSERT_TRUE(rcpt2);
    EXPECT_EQ(rcpt1->m_spam_status, NSO::EResolution::SO_RESOLUTION_ACCEPT);
    EXPECT_EQ(rcpt2->m_spam_status, NSO::EResolution::SO_RESOLUTION_REJECT);
}

TEST_F(TTestSOAsyncDelivery, test_make_personal_spam_header) {
    EXPECT_EQ(MakePersonalSpamHeader(SOResponse, Envelope), "X-Yandex-Personal-Spam: cmNwdF91aWQy\r\n");
}

TEST_F(TTestSOAsyncDelivery, test_make_spam_header) {
    EXPECT_EQ(MakeSpamHeader(NSO::EResolution::SO_RESOLUTION_SPAM), "X-Yandex-Spam: 4\r\n");
}

TEST_F(TTestSOAsyncDelivery, test_make_uid_status_header) {
    EXPECT_EQ(MakeUidStatusHeader(SOResponse, Envelope),
        "X-Yandex-Uid-Status: 1 rcpt_uid1,256 rcpt_uid2\r\n");
}

TEST_F(TTestSOAsyncDelivery, for_empty_response_out_parameters_should_return_empty_foreign_mx) {
    SOResponse.OutParameters = std::optional<NSO::TOutParameters>{};
    EXPECT_EQ(MakeForeignMx(SOResponse), "");
}

TEST_F(TTestSOAsyncDelivery, for_response_out_parameters_with_white_mx_type_should_return_foreign_mx) {
    SOResponse.OutParameters = NSO::TOutParameters {
        .Type = NSO::EForwardType::FORWARD_TYPE_WHITE,
        .ForeignMx = true
    };
    EXPECT_EQ(MakeForeignMx(SOResponse), "whi");
}

TEST_F(TTestSOAsyncDelivery, for_response_out_parameters_with_black_mx_type_should_return_foreign_mx) {
    SOResponse.OutParameters = NSO::TOutParameters {
        .Type = NSO::EForwardType::FORWARD_TYPE_MXBACK,
        .ForeignMx = true
    };
    EXPECT_EQ(MakeForeignMx(SOResponse), "bla");
}

TEST_F(TTestSOAsyncDelivery, for_response_out_parameters_with_gray_mx_type_should_return_foreign_mx) {
    SOResponse.OutParameters = NSO::TOutParameters {
        .Type = NSO::EForwardType::FORWARD_TYPE_GRAY,
        .ForeignMx = true
    };
    EXPECT_EQ(MakeForeignMx(SOResponse), "gre");
}

TEST_F(TTestSOAsyncDelivery, for_response_out_parameters_with_invalid_mx_type_should_return_foreign_mx) {
    SOResponse.OutParameters = NSO::TOutParameters {
        .Type = NSO::EForwardType::FORWARD_TYPE_INVALID,
        .ForeignMx = true
    };
    EXPECT_EQ(MakeForeignMx(SOResponse), "gre");
}

TEST_F(TTestSOAsyncDelivery, for_spam_resolution_should_return_spam_flag_header_with_yes_value) {
    EXPECT_EQ(MakeSpamFlagHeader(NSO::EResolution::SO_RESOLUTION_SPAM), "X-Spam-Flag: YES\r\n");
}

TEST_F(TTestSOAsyncDelivery, for_reject_resolution_should_return_spam_flag_header_with_yes_value) {
    EXPECT_EQ(MakeSpamFlagHeader(NSO::EResolution::SO_RESOLUTION_REJECT), "X-Spam-Flag: YES\r\n");
}

TEST_F(TTestSOAsyncDelivery, for_accept_resolution_should_return_spam_flag_header_with_no_value) {
    EXPECT_EQ(MakeSpamFlagHeader(NSO::EResolution::SO_RESOLUTION_ACCEPT), "X-Spam-Flag: NO\r\n");
}

TEST_F(TTestSOAsyncDelivery, for_foreign_mx_value_should_return_foreign_mx_header) {
    EXPECT_EQ(MakeForeignMXHeader("gre"), "X-Yandex-ForeignMX: gre\r\n");
}

TEST_F(TTestSOAsyncDelivery, test_format_so_classes) {
    EXPECT_EQ(
        FormatSoClasses(SOResponse),
        "label=SystMetkaSO:class1\nlabel=SystMetkaSO:class2\nlabel=domain_class\n"
    );
}

TEST_F(TTestSOAsyncDelivery, for_so_request_with_reject_resolution_should_return_error_code) {
    SOResponse.Resolution = NSO::EResolution::SO_RESOLUTION_REJECT;
    const auto& [ec, answer] = MakeSoResult(
        SOResponse,
        Envelope,
        "dmarc_domain",
        dmarc::record::QUARANTINE,
        Options,
        false
    );
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::MaliciousRejected);
}

TEST_F(TTestSOAsyncDelivery, for_so_request_with_spam_resolution_should_return_error_code) {
    SOResponse.Resolution = NSO::EResolution::SO_RESOLUTION_SPAM;
    Options.ActionSpam = NSO::ESpamPolicy::REJECT;
    const auto& [ec, answer] = MakeSoResult(
        SOResponse,
        Envelope,
        "dmarc_domain",
        dmarc::record::QUARANTINE,
        Options,
        false
    );
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::SpamRejected);
}

TEST_F(TTestSOAsyncDelivery, for_so_request_with_spam_resolution_and_captcha_should_no_return_error_code) {
    SOResponse.Resolution = NSO::EResolution::SO_RESOLUTION_SPAM;
    SOResponse.OutParameters->Type = NSO::EForwardType::FORWARD_TYPE_MXBACK;
    const auto& [ec, answer] = MakeSoResult(
        SOResponse,
        Envelope,
        "dmarc_domain",
        dmarc::record::QUARANTINE,
        Options,
        true
    );
    ASSERT_FALSE(ec);
    EXPECT_TRUE(Envelope->m_spam);
    EXPECT_EQ(Envelope->so_status, NSO::EResolution::SO_RESOLUTION_SPAM);
    auto addedHeaders = std::string{
        Envelope->added_headers_.begin(),
        Envelope->added_headers_.end()
    };
    EXPECT_EQ(
        addedHeaders,
        "X-Yandex-Spam: 4\r\n"
        "X-Yandex-ForeignMX: bla\r\n"
        "X-Yandex-Uid-Status: 1 rcpt_uid1,256 rcpt_uid2\r\n"
        "X-Yandex-Personal-Spam: cmNwdF91aWQy\r\n"
        "X-Spam-Flag: YES\r\n"
    );
}

TEST_F(TTestSOAsyncDelivery, for_so_request_with_accept_resolution_should_no_return_error_code) {
    SOResponse.Resolution = NSO::EResolution::SO_RESOLUTION_ACCEPT;
    SOResponse.OutParameters = std::nullopt;
    const auto& [ec, answer] = MakeSoResult(
        SOResponse,
        Envelope,
        "dmarc_domain",
        dmarc::record::QUARANTINE,
        Options,
        false
    );
    ASSERT_FALSE(ec);
    EXPECT_FALSE(Envelope->m_spam);
    EXPECT_EQ(Envelope->so_status, NSO::EResolution::SO_RESOLUTION_ACCEPT);
    EXPECT_EQ(Envelope->foreign_mx, "");
    EXPECT_THAT(
        Envelope->so_labels_,
        ElementsAre(
            "class1",
            "class2",
            "domain_class"
        )

    );
    EXPECT_THAT(
        Envelope->xyandex_hint_header_value_personal_,
        ElementsAre(
            "email=rcpt1\nreplace_so_labels=1\nlabel=SystMetkaSO:class3\nlabel=SystMetkaSO:class4\n",
            "email=rcpt2\nreplace_so_labels=1\nlabel=SystMetkaSO:class5\nlabel=SystMetkaSO:class6\nlabel=SystMetkaSO:pf_spam\n"
        )
    );
    EXPECT_EQ(
        Envelope->xyandex_hint_header_value_,
        "label=SystMetkaSO:class1\nlabel=SystMetkaSO:class2\nlabel=domain_class\n"
    );

    auto addedHeaders = std::string{
        Envelope->added_headers_.begin(),
        Envelope->added_headers_.end()
    };
    EXPECT_EQ(
        addedHeaders,
        "X-Yandex-Spam: 1\r\n"
        "X-Yandex-Uid-Status: 1 rcpt_uid1,256 rcpt_uid2\r\n"
        "X-Yandex-Personal-Spam: cmNwdF91aWQy\r\n"
        "X-Spam-Flag: NO\r\n"
    );
    auto* rcpt1 = Envelope->m_rcpts.GetByUid("rcpt_uid1");
    auto* rcpt2 = Envelope->m_rcpts.GetByUid("rcpt_uid2");
    ASSERT_TRUE(rcpt1);
    ASSERT_TRUE(rcpt2);
    EXPECT_EQ(rcpt1->m_spam_status, NSO::EResolution::SO_RESOLUTION_ACCEPT);
    EXPECT_EQ(rcpt2->m_spam_status, NSO::EResolution::SO_RESOLUTION_REJECT);
    EXPECT_EQ(rcpt1->inactive_for_, 10u);
    EXPECT_EQ(rcpt1->m_delivery_status, check::CHK_TEMPFAIL);
    EXPECT_EQ(rcpt2->inactive_for_, 100u);
    EXPECT_EQ(rcpt2->m_delivery_status, check::CHK_REJECT);
}

struct TTestIsSpam: TestWithParam<std::tuple<NSO::EResolution, bool>> {
};

TEST_P(TTestIsSpam, test_is_spam) {
    EXPECT_EQ(
        IsSpam(std::get<0>(GetParam())),
        std::get<1>(GetParam())
    );
}

INSTANTIATE_TEST_SUITE_P(
    test_is_spam,
    TTestIsSpam,
    Values(
        std::make_tuple(
            NSO::EResolution::SO_RESOLUTION_ACCEPT,
            false
        ),
        std::make_tuple(
            NSO::EResolution::SO_RESOLUTION_REJECT,
            true
        ),
        std::make_tuple(
            NSO::EResolution::SO_RESOLUTION_SPAM,
            true
        ),
        std::make_tuple(
            NSO::EResolution::SO_RESOLUTION_INVALID,
            false
        )
    )
);

struct TTestGetMaliciousErrorCode: TestWithParam<std::tuple<std::vector<std::string>, TErrorCode>> {
};

TEST_P(TTestGetMaliciousErrorCode, test_get_malicious_error_code) {
    EXPECT_EQ(
        GetMaliciousErrorCode(std::get<0>(GetParam())),
        std::get<1>(GetParam())
    );
}

INSTANTIATE_TEST_SUITE_P(
    test_get_malicious_error_code,
    TTestGetMaliciousErrorCode,
    Values(
        std::make_tuple(
            std::vector<std::string>{},
            EError::MaliciousRejected
        ),
        std::make_tuple(
            std::vector<std::string>{"res_rfc_fail"},
            EError::RfcFailRejected
        ),
        std::make_tuple(
            std::vector<std::string>{"res_url_rbl"},
            EError::UrlRblRejected
        ),
        std::make_tuple(
            std::vector<std::string>{"res_bad_karma"},
            EError::BadKarmaRejected
        ),
        std::make_tuple(
            std::vector<std::string>{"res_mail_limits"},
            EError::MailLimitsRejected
        ),
        std::make_tuple(
            std::vector<std::string>{"res_pdd_admin_karma"},
            EError::PddAdminKarmaRejected
        ),
        std::make_tuple(
            std::vector<std::string>{"res_bounces"},
            EError::BouncesRejected
        ),
        std::make_tuple(
            std::vector<std::string>{"res_spam_compl"},
            EError::SpamComplRejected
        ),
        std::make_tuple(
            std::vector<std::string>{"tutu"},
            EError::MaliciousRejected
        )
    )
);

} // namespace anonymous
