#include "sync_delivery_mocks.h"

#include <mail/nwsmtp/src/delivery/sync/errors.h>
#include <mail/nwsmtp/src/delivery/sync/hint.h>
#include <mail/nwsmtp/src/delivery/sync/sync.h>
#include <mail/nwsmtp/src/delivery/sync/utils.h>
#include <mail/nwsmtp/src/header_storage.h>
#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/so/config.h>
#include <mail/nwsmtp/src/types.h>
#include <mail/nwsmtp/src/utils.h>

#include <mail/ymod_httpclient/include/ymod_httpclient/errors.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <boost/range/algorithm_ext.hpp>

#include <memory>

namespace {

using namespace testing;
using namespace NTesting::NMocks;
using namespace NNwSmtp;
using namespace NNwSmtp::NDlv;

struct TTestSyncDelivery: Test {
    TTestSyncDelivery() {
        glog = std::make_shared<TLog>();
        std::tie(Headers, Body) = ParseMessage(MessageWithHeaders);
    }

    TEnvelopeInfo MakeEnvelopeInfo() {
        return {
            "request_id",
            "session_id",
            "env_id",
            "192.0.2.235",
            ""
        };
    }

    TMailInfo MakeMailInfo() {
        return {"1", "path", "delimeter", "spam_path", 228, TLabels {}, "", "", "stid", "news"};
    }

    TUserInfo MakeUserInfo() {
        return {"15500", "email"};
    }

    TConfigPtr MakeConfig() {
        NSO::TOptions soOptions;
        soOptions.ClusterName = "so_server";
        return std::make_shared<TConfig>(
            TCaseInsensitiveHashSet{},
            soOptions,
            std::unordered_set<std::string>{},
            Options::DecyclerOpts{true, true, 64}
        );
    }

    TOptions MakeOptions() {
        return TOptions {
            .DetectSpam = true,
            .DetectVirus = true,
            .DetectCycle = true
        };
    }

    NNwSmtp::NMds::TPutRequest MakeMdsRequest() {
        return {"15500", false, Headers, AddedHeaders, Body, {}};
    }

    NSO::TRequest MakeSoRequest(
        std::string soType = "EMAIL_TYPE_REGULAR",
        bool dryRun = false
    ) {
        return {
            NSO::TEnvelope {
                .ConnectInfo = {
                    .Host = "remote_host",
                    .Ip = std::string("\xc0\x00\x02\xeb", 4),
                    .Domain = "remote_host",
                    .SessionId = "session_id-env_id"
                },
                .EmailType = soType,
                .Rcpts = {{
                    .Address = {.Email = "email"},
                    .Uid = "15500"
                }},
                .Timestamp = "228",
            },
            "X-Yandex-QueueID: session_id-env_id\r\n"
            "X-Yandex-So-Front: so_server\r\n"
            "Header-1: 1\r\n"
            "Header-2: 2\r\n"
            "\r\n"
            "Body",
            dryRun
        };
    }

    NNsls::TRequest MakeNslsRequest() {
        auto hint = BuildHint(Options, MailInfo,
            NSO::TResponse{.Resolution = NSO::EResolution::SO_RESOLUTION_ACCEPT, .SoClasses = {"so_label1", "t_news"}});
        auto envelopeInfo = EnvelopeInfo;
        envelopeInfo.RemoteHost = "remote_host";
        return BuildStoreRequest({}, envelopeInfo, UserInfo, "stid", false, std::move(hint));
    }

    std::shared_ptr<StrictMock<TMdsClientMock>> MdsClientMock
        = std::make_shared<StrictMock<TMdsClientMock>>();
    std::shared_ptr<StrictMock<TResolverClient>> ResolveClientMock
        = std::make_shared<StrictMock<TResolverClient>>();
    std::shared_ptr<StrictMock<TNslsClient>> NslsClientMock
        = std::make_shared<StrictMock<TNslsClient>>();
    std::shared_ptr<StrictMock<TAvirClient>> AvirClientMock
        = std::make_shared<StrictMock<TAvirClient>>();
    std::shared_ptr<StrictMock<TSOClientMock>> SOInClientMock
        = std::make_shared<StrictMock<TSOClientMock>>();
    std::shared_ptr<StrictMock<TSOClientMock>> SOOutClientMock
        = std::make_shared<StrictMock<TSOClientMock>>();

    std::string RawMessage = "Header-1: 1\r\nHeader-2: 2\r\n\r\nBody";
    NNwSmtp::TBuffer MessageWithHeaders = NUtil::MakeSegment("Header-1: 1\r\nHeader-2: 2\r\n\r\nBody");
    NNwSmtp::TBuffer MessageWithFwdHeader = NUtil::MakeSegment("X-Yandex-Fwd: 64\r\n\r\nBody");
    NNwSmtp::TBuffer MessageWithoutHeaders = NUtil::MakeSegment("Body");
    NNwSmtp::TBuffer AddedHeaders = NUtil::MakeSegment("X-Yandex-Fwd: 1\r\n");
    TBufferRange Body;
    THeaderStorage<TBufferRange> Headers;

    TContextPtr Context = boost::make_shared<NNwSmtp::TContext>("","","","");
    TOptions Options = MakeOptions();
    TMailInfo MailInfo = MakeMailInfo();
    TEnvelopeInfo EnvelopeInfo = MakeEnvelopeInfo();
    TUserInfo UserInfo = MakeUserInfo();
    TConfigPtr Config = MakeConfig();

    void CallSendMail(
        const NNwSmtp::TBuffer& message,
        TSyncResultCallback&& handler
    ) {
        auto sendMailHandle = std::make_shared<TSyncDelivery>(
            Context,
            Options,
            EnvelopeInfo,
            MailInfo,
            UserInfo,
            message,
            NslsClientMock,
            MdsClientMock,
            SOInClientMock,
            SOOutClientMock,
            ResolveClientMock,
            AvirClientMock,
            Config,
            std::move(handler)
        );

        yplatform::spawn(sendMailHandle);
    }
};

TEST_F(TTestSyncDelivery, for_bad_message_should_return_error) {
    const InSequence s;
    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    CallSendMail(
        MessageWithoutHeaders,
        [](auto ec, auto) {
            ASSERT_TRUE(ec);
            EXPECT_EQ(ec, EError::BadMessage);
        }
    );
}

TEST_F(TTestSyncDelivery, for_detect_cycle_should_return_error) {
    const InSequence s;
    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    CallSendMail(
        MessageWithFwdHeader,
        [](auto ec, auto) {
            ASSERT_TRUE(ec);
            EXPECT_EQ(ec, EError::CycleDetected);
        }
    );
}

TEST_F(TTestSyncDelivery, for_failed_so_in_request_on_store_should_return_error) {
    const InSequence s;
    auto soRequest = MakeSoRequest();

    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    EXPECT_CALL(*SOInClientMock, Check(_, Pointee(soRequest), _))
        .WillOnce(
            InvokeArgument<2>(
                ymod_httpclient::http_error::code::ssl_error,
                NSO::TResponse{}
            )
        );

    CallSendMail(
        MessageWithHeaders,
        [](auto ec, auto) {
            ASSERT_TRUE(ec);
            EXPECT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
        }
    );
}

TEST_F(TTestSyncDelivery, for_so_in_response_with_spam_resolution_on_store_should_return_error) {
    const InSequence s;
    auto soRequest = MakeSoRequest();

    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    EXPECT_CALL(*SOInClientMock, Check(_, Pointee(soRequest), _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NSO::TResponse{.Resolution = NSO::EResolution::SO_RESOLUTION_SPAM}
            )
        );

    CallSendMail(
        MessageWithHeaders,
        [](auto ec, auto) {
            ASSERT_TRUE(ec);
            EXPECT_EQ(ec, EError::Spam);
        }
    );
}

TEST_F(TTestSyncDelivery, for_so_out_response_with_spam_resolution_on_store_should_return_error) {
    const InSequence s;
    auto soRequest = MakeSoRequest();
    Options.SoOut = true;

    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    EXPECT_CALL(*SOOutClientMock, Check(_, Pointee(soRequest), _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NSO::TResponse{.Resolution = NSO::EResolution::SO_RESOLUTION_SPAM}
            )
        );
    CallSendMail(
        MessageWithHeaders,
        [](auto ec, auto) {
            ASSERT_TRUE(ec);
            EXPECT_EQ(ec, EError::Spam);
        }
    );
}

TEST_F(TTestSyncDelivery, for_failed_avir_request_on_store_should_return_error) {
    const InSequence s;
    auto soRequest = MakeSoRequest();

    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    EXPECT_CALL(*SOInClientMock, Check(_, Pointee(soRequest), _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NSO::TResponse{.Resolution = NSO::EResolution::SO_RESOLUTION_ACCEPT}
            )
        );
    EXPECT_CALL(*AvirClientMock, Check(_, MessageWithHeaders, _))
        .WillOnce(
            InvokeArgument<2>(
                ymod_httpclient::http_error::code::ssl_error,
                NAvir::TStatus::unknown
            )
        );
    CallSendMail(
        MessageWithHeaders,
        [](auto ec, auto) {
            ASSERT_TRUE(ec);
            EXPECT_EQ(ec, EError::AvirError);
        }
    );
}

TEST_F(TTestSyncDelivery, for_avir_response_with_infected_resolution_on_store_should_return_error) {
    const InSequence s;
    auto soRequest = MakeSoRequest();

    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    EXPECT_CALL(*SOInClientMock, Check(_, Pointee(soRequest), _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NSO::TResponse{.Resolution = NSO::EResolution::SO_RESOLUTION_ACCEPT}
            )
        );
    EXPECT_CALL(*AvirClientMock, Check(_, MessageWithHeaders, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NAvir::TStatus::infected
            )
        );
    CallSendMail(
        MessageWithHeaders,
        [](auto ec, auto) {
            ASSERT_TRUE(ec);
            EXPECT_EQ(ec, EError::Virus);
        }
    );
}

TEST_F(TTestSyncDelivery, for_failed_mds_put_request_on_store_should_return_error) {
    const InSequence s;
    auto soRequest = MakeSoRequest();
    auto mdsRequest = MakeMdsRequest();

    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    EXPECT_CALL(*SOInClientMock, Check(_, Pointee(soRequest), _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NSO::TResponse{
                    .Resolution = NSO::EResolution::SO_RESOLUTION_ACCEPT,
                    .SoClasses = {"so_label1", "t_news"}
                }
            )
        );
    EXPECT_CALL(*AvirClientMock, Check(_, MessageWithHeaders, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NAvir::TStatus::clean
            )
        );
    EXPECT_CALL(*MdsClientMock, Put(_, mdsRequest, _))
        .WillOnce(
            InvokeArgument<2>(
                ymod_httpclient::http_error::code::ssl_error,
                std::string{}
            )
        );

    CallSendMail(
        MessageWithHeaders,
        [](auto ec, auto) {
            ASSERT_TRUE(ec);
            EXPECT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
        }
    );
}

TEST_F(TTestSyncDelivery, for_failed_nsls_store_request_on_store_should_return_error) {
    const InSequence s;
    auto soRequest = MakeSoRequest();
    auto mdsRequest = MakeMdsRequest();
    auto nslsRequest = MakeNslsRequest();

    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    EXPECT_CALL(*SOInClientMock, Check(_, Pointee(soRequest), _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NSO::TResponse{
                    .Resolution = NSO::EResolution::SO_RESOLUTION_ACCEPT,
                    .SoClasses = {"so_label1", "t_news"}
                }
            )
        );
    EXPECT_CALL(*AvirClientMock, Check(_, MessageWithHeaders, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NAvir::TStatus::clean
            )
        );
    EXPECT_CALL(*MdsClientMock, Put(_, mdsRequest, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                std::string{"stid"}
            )
        );
    EXPECT_CALL(*NslsClientMock, Store(_, nslsRequest, _))
        .WillOnce(
            InvokeArgument<2>(
                ymod_httpclient::http_error::code::ssl_error,
                NNsls::TResponse{}
            )
        );

    CallSendMail(
        MessageWithHeaders,
        [](auto ec, auto) {
            ASSERT_TRUE(ec);
            EXPECT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
        }
    );
}

TEST_F(TTestSyncDelivery, for_successful_store_should_return_response) {
    const InSequence s;
    auto soRequest = MakeSoRequest();
    auto mdsRequest = MakeMdsRequest();
    auto nslsRequest = MakeNslsRequest();
    auto result = TSyncResult{"mid", "imap"};

    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    EXPECT_CALL(*SOInClientMock, Check(_, Pointee(soRequest), _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NSO::TResponse{
                    .Resolution = NSO::EResolution::SO_RESOLUTION_ACCEPT,
                    .SoClasses = {"so_label1", "t_news"}
                }
            )
        );
    EXPECT_CALL(*AvirClientMock, Check(_, MessageWithHeaders, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NAvir::TStatus::clean
            )
        );
    EXPECT_CALL(*MdsClientMock, Put(_, mdsRequest, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                std::string{"stid"}
            )
        );
    EXPECT_CALL(*NslsClientMock, Store(_, nslsRequest, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NNsls::TResponse{
                    "success",
                    NNsls::TNslsSuccessResponse{"mid", "imap", false, false},
                    NNsls::TNslsErrorResponse{}
                }
            )
        );

    CallSendMail(
        MessageWithHeaders,
        [&](auto ec, auto response) {
            ASSERT_FALSE(ec);
            EXPECT_EQ(response, result);
        }
    );
}

TEST_F(TTestSyncDelivery, for_failed_mds_get_request_on_restore_should_return_error) {
    const InSequence s;
    Options.DlvType = EDlvType::Restore;
    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    EXPECT_CALL(*MdsClientMock, Get(_, "stid", _))
        .WillOnce(
            InvokeArgument<2>(
                ymod_httpclient::http_error::code::ssl_error,
                std::string{}
            )
        );

    CallSendMail(
        MessageWithHeaders,
        [](auto ec, auto) {
            ASSERT_TRUE(ec);
            EXPECT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
        }
    );
}

TEST_F(TTestSyncDelivery, for_failed_so_request_on_restore_should_return_error) {
    const InSequence s;
    Options.DlvType = EDlvType::Restore;
    auto soRequest = MakeSoRequest("EMAIL_TYPE_RESTORE", true);

    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    EXPECT_CALL(*MdsClientMock, Get(_, "stid", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                RawMessage
            )
        );
    EXPECT_CALL(*SOInClientMock, Check(_, Pointee(soRequest), _))
        .WillOnce(
            InvokeArgument<2>(
                ymod_httpclient::http_error::code::ssl_error,
                NSO::TResponse{}
            )
        );

    CallSendMail(
        MessageWithHeaders,
        [](auto ec, auto) {
            ASSERT_TRUE(ec);
            EXPECT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
        }
    );
}

TEST_F(TTestSyncDelivery, for_failed_nsls_store_request_on_restore_should_return_error) {
    const InSequence s;
    Options.DlvType = EDlvType::Restore;
    auto soRequest = MakeSoRequest("EMAIL_TYPE_RESTORE", true);
    auto nslsRequest = MakeNslsRequest();

    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    EXPECT_CALL(*MdsClientMock, Get(_, "stid", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                RawMessage
            )
        );
    EXPECT_CALL(*SOInClientMock, Check(_, Pointee(soRequest), _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NSO::TResponse{
                    .Resolution = NSO::EResolution::SO_RESOLUTION_ACCEPT,
                    .SoClasses = {"so_label1", "t_news"}
                }
            )
        );
    EXPECT_CALL(*NslsClientMock, Store(_, nslsRequest, _))
        .WillOnce(
            InvokeArgument<2>(
                ymod_httpclient::http_error::code::ssl_error,
                NNsls::TResponse{}
            )
        );

    CallSendMail(
        MessageWithHeaders,
        [](auto ec, auto) {
            ASSERT_TRUE(ec);
            EXPECT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
        }
    );
}

TEST_F(TTestSyncDelivery, for_successful_restore_should_return_response) {
    const InSequence s;
    Options.DlvType = EDlvType::Restore;
    auto soRequest = MakeSoRequest("EMAIL_TYPE_RESTORE", true);
    auto nslsRequest = MakeNslsRequest();
    auto result = TSyncResult{"mid", "imap"};

    EXPECT_CALL(*ResolveClientMock, ResolveIp(_, "192.0.2.235", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                "remote_host"
            )
        );
    EXPECT_CALL(*MdsClientMock, Get(_, "stid", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                RawMessage
            )
        );

    EXPECT_CALL(*SOInClientMock, Check(_, Pointee(soRequest), _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NSO::TResponse{
                    .Resolution = NSO::EResolution::SO_RESOLUTION_ACCEPT,
                    .SoClasses = {"so_label1", "t_news"}
                }
            )
        );
    EXPECT_CALL(*NslsClientMock, Store(_, nslsRequest, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode{},
                NNsls::TResponse{
                    "success",
                    NNsls::TNslsSuccessResponse{"mid", "imap", false, false},
                    NNsls::TNslsErrorResponse{}
                }
            )
        );

    CallSendMail(
        MessageWithHeaders,
        [&](auto ec, auto response) {
            ASSERT_FALSE(ec);
            EXPECT_EQ(response, result);
        }
    );
}

} // namespace anonymous
