#include <mail/nwsmtp/src/so/types_reflection.h>

#include <mail/nwsmtp/src/delivery/sync/errors.h>
#include <mail/nwsmtp/src/delivery/sync/utils.h>
#include <mail/nwsmtp/src/delivery/sync/types.h>
#include <mail/nwsmtp/src/nsls/types.h>
#include <mail/nwsmtp/src/so/config.h>
#include <mail/nwsmtp/src/so/types.h>
#include <mail/nwsmtp/src/types.h>

#include <boost/fusion/include/equal_to.hpp>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

namespace NNwSmtp::NDlv {

static bool operator ==(const TSyncResult& lhs, const TSyncResult& rhs) {
    return lhs.Mid == rhs.Mid
        && lhs.ImapId == rhs.ImapId;
}

}

namespace NNwSmtp::NSO {

static bool operator ==(const TConnectInfo& lhs, const TConnectInfo& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

static bool operator ==(const TEmail& lhs, const TEmail& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

static bool operator ==(const TEmailInfo& lhs, const TEmailInfo& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

static bool operator ==(const TEnvelope& lhs, const TEnvelope& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

static bool operator ==(const TRequest& lhs, const TRequest& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

} // namespace NNwSmtp::NSO

namespace {

using namespace testing;
using namespace NNwSmtp;
using namespace NNwSmtp::NDlv;
using namespace NNwSmtp::NSO;

auto MakeSuccessfulNslsResponse(bool isDuplicate, bool isDecycled) {
    return NNsls::TResponse{
        "success",
        NNsls::TNslsSuccessResponse{"mid", "imap", isDuplicate, isDecycled},
        NNsls::TNslsErrorResponse {}
    };
}

auto MakeWrongNslsResponse(bool isPermanent) {
    return NNsls::TResponse{
        "error",
        NNsls::TNslsSuccessResponse{},
        NNsls::TNslsErrorResponse {isPermanent, "", ""}
    };
}

TEST(TestDlvUtils, for_successful_nsls_response_with_duplicate_flag_should_return_error) {
    TErrorCode ec;
    TSyncResult result;
    std::tie(ec, result) = MakeStoreResult(MakeSuccessfulNslsResponse(true, false));
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::DuplicateFound);
}

TEST(TestDlvUtils, for_successful_nsls_response_with_decycled_flag_should_return_error) {
    TErrorCode ec;
    TSyncResult result;
    std::tie(ec, result) = MakeStoreResult(MakeSuccessfulNslsResponse(false, true));
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::CycleDetected);
}

TEST(TestDlvUtils, for_successful_nsls_response_should_return_delivery_result) {
    TErrorCode ec;
    TSyncResult result;
    auto response = TSyncResult{"mid", "imap"};
    std::tie(ec, result) = MakeStoreResult(MakeSuccessfulNslsResponse(false, false));
    ASSERT_FALSE(ec);
    EXPECT_EQ(result, response);
}

TEST(TestDlvUtils, for_wrong_nsls_response_with_permanent_flag_should_return_error) {
    TErrorCode ec;
    TSyncResult result;
    std::tie(ec, result) = MakeStoreResult(MakeWrongNslsResponse(true));
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::InvalidFid);
}

TEST(TestDlvUtils, for_wrong_nsls_response_should_return_error) {
    TErrorCode ec;
    TSyncResult result;
    std::tie(ec, result) = MakeStoreResult(MakeWrongNslsResponse(false));
    ASSERT_TRUE(ec);
    EXPECT_EQ(ec, EError::NslsTemporaryError);
}

TEST(TestDlvUtils, for_empty_so_response_should_return_empty_so_response) {
    std::optional<NSO::TResponse> response;
    ChangeTab(response, "news");
    ASSERT_FALSE(response.has_value());
}

auto MakeSoResponse(
    const std::string& tab = "",
    NSO::EResolution status = NSO::EResolution::SO_RESOLUTION_ACCEPT
) {
    auto response = NSO::TResponse{.Resolution = status, .SoClasses = {"so_label"}};
    if (!tab.empty()) {
        response.SoClasses.push_back(tab);
    }
    return response;
}

TEST(TestDlvUtils, for_empty_input_tab_and_so_response_with_tab_should_return_so_response_with_so_tab) {
    std::optional<NSO::TResponse> response = MakeSoResponse("t_social");
    ChangeTab(response, "");
    ASSERT_TRUE(response.has_value());
    EXPECT_THAT(
        response->SoClasses,
        ElementsAre(
            "so_label",
            "t_social"
        )
    );

}

TEST(TestDlvUtils, for_input_tab_and_so_response_with_tab_should_return_so_response_with_input_tab) {
    std::optional<NSO::TResponse> response = MakeSoResponse("t_social");
    ChangeTab(response, "news");
    ASSERT_TRUE(response.has_value());
    EXPECT_THAT(
        response->SoClasses,
        ElementsAre(
            "so_label",
            "t_news"
        )
    );

}

TEST(TestDlvUtils, for_input_tab_and_so_response_without_tab_should_return_so_response_with_input_tab) {
    std::optional<NSO::TResponse> response = MakeSoResponse();
    ChangeTab(response, "news");
    ASSERT_TRUE(response.has_value());
    EXPECT_THAT(
        response->SoClasses,
        ElementsAre(
            "so_label",
            "t_news"
        )
    );

}

TEST(TestDlvUtils, for_empty_so_result_should_no_return_error) {
    ASSERT_FALSE(MakeSoResult({}, {}, false));
}

TEST(TestDlvUtils, for_so_result_with_ham_resolution_should_no_return_error) {
    ASSERT_FALSE(MakeSoResult(MakeSoResponse("", NSO::EResolution::SO_RESOLUTION_ACCEPT), {}, false));
}

TEST(TestDlvUtils, for_so_result_with_spam_resolution_should_return_error) {
    auto result = MakeSoResult(MakeSoResponse("", NSO::EResolution::SO_RESOLUTION_SPAM), {}, false);
    ASSERT_TRUE(result);
    EXPECT_EQ(result, EError::Spam);
}

TEST(TestDlvUtils, for_so_result_with_spam_resolution_and_with_captcha_should_return_error) {
    auto result = MakeSoResult(MakeSoResponse("", NSO::EResolution::SO_RESOLUTION_SPAM), {}, true);
    ASSERT_TRUE(result);
    EXPECT_EQ(result, EError::Spam);
}

TEST(TestDlvUtils, for_so_result_with_spam_resolution_and_with_save_type_and_not_captcha_should_return_error) {
    auto options = NDlv::TOptions{.DlvType = EDlvType::Save};
    auto result = MakeSoResult(MakeSoResponse("", NSO::EResolution::SO_RESOLUTION_SPAM), options, false);
    ASSERT_TRUE(result);
    EXPECT_EQ(result, EError::Spam);
}

TEST(TestDlvUtils, for_so_result_with_spam_resolution_and_with_save_type_and_captcha_should_no_return_error) {
    auto options = NDlv::TOptions{.DlvType = EDlvType::Save};
    auto result = MakeSoResult(MakeSoResponse("", NSO::EResolution::SO_RESOLUTION_SPAM), options, true);
    ASSERT_FALSE(result);
}

TEST(TestDlvUtils, for_so_result_with_spam_resolution_and_with_mailish_type_should_no_return_error) {
    auto options = NDlv::TOptions{.DlvType = EDlvType::Mailish};
    auto result = MakeSoResult(MakeSoResponse("", NSO::EResolution::SO_RESOLUTION_SPAM), options, false);
    ASSERT_FALSE(result);
}

TEST(TestDlvUtils, for_so_result_with_malicious_resolution_and_with_not_mailish_type_should_return_error) {
    auto result = MakeSoResult(MakeSoResponse("", NSO::EResolution::SO_RESOLUTION_REJECT), {}, false);
    ASSERT_TRUE(result);
    EXPECT_EQ(result, EError::Malicious);
}

TEST(TestDlvUtils, for_not_virus_avir_result_should_no_return_error) {
    ASSERT_FALSE(MakeAvirResult(NAvir::TStatus::clean));
}

TEST(TestDlvUtils, for_virus_avir_result_should_return_error) {
    auto result = MakeAvirResult(NAvir::TStatus::infected);
    ASSERT_TRUE(result);
    EXPECT_EQ(result, EError::Virus);
}

TEnvelopeInfo MakeEnvelopeInfo() {
    return {
        .RequestId = "request_id",
        .SessionId = "session_id",
        .EnvId = "env_id",
        .RemoteIp = "192.0.2.235",
        .RemoteHost = "remote_host"
    };
}

TUserInfo MakeUserInfo() {
    return {
        .Uid = "uid",
        .Email = "email",
    };
}

NDlv::TOptions MakeOptions(EDlvType dlvType = EDlvType::Mailish) {
    return {
        .DetectCycle = false,
        .DlvType = dlvType,
    };
}

TConfigPtr MakeConfig(bool addTrustHeaderCaptcha = false) {
    auto trustHeaders = std::unordered_set<std::string>{};
    NSO::TOptions soOptions;
    soOptions.ClusterName = "so_server";
    soOptions.InternalHeaders = std::unordered_set<std::string>{"header-1", "header-2"};

    if (addTrustHeaderCaptcha) {
        trustHeaders.insert("x-yandex-captcha-entered");
    }
    return std::make_shared<TConfig>(
        TCaseInsensitiveHashSet{},
        soOptions,
        trustHeaders,
        Options::DecyclerOpts{true, true, 64}
    );
}

TMailInfo MakeMailInfo() {
    return {
        .Fid = "fid",
        .ReceivedDate = 228,
        .Labels = {},
        .ExternalImapId = "imap_id",
        .OldMid = "old_mid",
        .Stid = "stid",
        .Tab = "tab"
    };
}

NSO::TRequest MakeSoRequest() {
    return {
        NSO::TEnvelope {
            .ConnectInfo = {
                .Host = "remote_host",
                .Ip = std::string("\xc0\x00\x02\xeb", 4),
                .Domain = "remote_host",
                .SessionId = "session_id-env_id"
            },
            .EmailType = "EMAIL_TYPE_MAILISH",
            .Rcpts = {{
                .Address = {.Email = "email"},
                .Uid = "uid"
            }},
            .Timestamp = "228",
        },
        "X-Yandex-QueueID: session_id-env_id\r\n"
        "X-Yandex-So-Front: so_server\r\n"
        "X-Yandex-Mailish: uid\r\n"
        "Header-3: 3\r\n"
        "\r\n"
        "Body",
        true
    };
}

TEST(TestDlvUtils, for_build_so_headers_should_return_so_headers) {
    auto userInfo = MakeUserInfo();
    auto envInfo = MakeEnvelopeInfo();
    auto config = MakeConfig();
    auto message = NUtil::MakeSegment(
        "Header-1: 1\r\nHeader-2: 2\r\nHeader-3: 3\r\n\r\nBody"
    );
    auto [headers, _] = ParseMessage(message);

    EXPECT_EQ(
        BuildSoHeaders(
            {
                .SoClusterName = config->SoOptions.ClusterName,
                .Mailish = userInfo.Uid,
                .Headers = headers,
                .InternalHeaders = config->SoOptions.InternalHeaders,
            },
            "session_id-env_id"
        ),
        "X-Yandex-QueueID: session_id-env_id\r\n"
        "X-Yandex-So-Front: so_server\r\n"
        "X-Yandex-Mailish: uid\r\n"
        "Header-3: 3\r\n"
    );
}

TEST(TestDlvUtils, for_build_so_http_request_should_return_so_http_request) {
    auto userInfo = MakeUserInfo();
    auto envInfo = MakeEnvelopeInfo();
    auto mailInfo = MakeMailInfo();
    auto options = MakeOptions(EDlvType::Mailish);
    auto config = MakeConfig();
    auto message = NUtil::MakeSegment(
        "Header-1: 1\r\nHeader-2: 2\r\nHeader-3: 3\r\n\r\nBody"
    );
    auto [headers, body] = ParseMessage(message);

    auto result = BuildSoRequest(options, envInfo, userInfo, mailInfo, body, headers, config);
    auto request = MakeSoRequest();

    EXPECT_THAT(result, Pointee(request));
}

struct TTestGetSoEmailType: TestWithParam<std::tuple<EDlvType, std::string>> {
};

TEST_P(TTestGetSoEmailType, test_get_so_email_type) {
    EXPECT_EQ(
        GetSoEmailType(std::get<0>(GetParam())),
        std::get<1>(GetParam())
    );
}

INSTANTIATE_TEST_SUITE_P(
    test_get_so_email_type,
    TTestGetSoEmailType,
    Values(
        std::make_tuple(
            EDlvType::Sync,
            "EMAIL_TYPE_REGULAR"
        ),
        std::make_tuple(
            EDlvType::Mailish,
            "EMAIL_TYPE_MAILISH"
        ),
        std::make_tuple(
            EDlvType::Restore,
            "EMAIL_TYPE_RESTORE"
        ),
        std::make_tuple(
            EDlvType::Save,
            "EMAIL_TYPE_DELAYED"
        ),
        std::make_tuple(
            EDlvType::Imap,
            "EMAIL_TYPE_REGULAR"
        ),
        std::make_tuple(
            EDlvType::Collectors,
            "EMAIL_TYPE_REGULAR"
        )
    )
);

struct TTestGetSoDryRun: TestWithParam<std::tuple<EDlvType, bool, bool>> {
};

TEST_P(TTestGetSoDryRun, test_get_so_dry_run) {
    EXPECT_EQ(
        GetSoDryRun(std::get<0>(GetParam()), std::get<1>(GetParam())),
        std::get<2>(GetParam())
    );
}

INSTANTIATE_TEST_SUITE_P(
    test_get_so_dry_run,
    TTestGetSoDryRun,
    Values(
        std::make_tuple(
            EDlvType::Sync,
            false,
            false
        ),
        std::make_tuple(
            EDlvType::Mailish,
            false,
            true
        ),
        std::make_tuple(
            EDlvType::Restore,
            false,
            true
        ),
        std::make_tuple(
            EDlvType::Save,
            false,
            false
        ),
        std::make_tuple(
            EDlvType::Imap,
            false,
            false
        ),
        std::make_tuple(
            EDlvType::Collectors,
            false,
            false
        ),
        std::make_tuple(
            EDlvType::Collectors,
            true,
            true
        )
    )
);

TEST(TestCaptchaWasEntered, for_mail_with_captcha_header_with_yes_value_should_return_true) {
    auto message = NUtil::MakeSegment(
        "Header-1: 1\r\nX-Yandex-Captcha-Entered: yes\r\n\r\nBody"
    );
    auto [headers, body] = ParseMessage(message);

    EXPECT_TRUE(GetCaptchaWasEntered(headers));
}

TEST(TestCaptchaWasEntered, for_mail_with_captcha_header_with_not_yes_value_return_false) {
    auto message = NUtil::MakeSegment(
        "Header-1: 1\r\nX-Yandex-Captcha-Entered: boo\r\n\r\nBody"
    );
    auto [headers, _] = ParseMessage(message);

    EXPECT_FALSE(GetCaptchaWasEntered(headers));
}

TEST(TestCaptchaWasEntered, for_mail_with_not_captcha_header_should_return_false) {
    auto message = NUtil::MakeSegment(
        "Header-1: 1\r\n\r\nBody"
    );
    auto [headers, _] = ParseMessage(message);

    EXPECT_FALSE(GetCaptchaWasEntered(headers));
}

TEST(TestGetDecyclerCounter, for_mail_with_decycler_header_should_return_its_value) {
    auto message = NUtil::MakeSegment(
        "Header-1: 1\r\nX-Yandex-Fwd: 10\r\n\r\nBody"
    );
    auto [headers, _] = ParseMessage(message);

    EXPECT_EQ(GetDecyclerCounter(headers), 10u);
}

TEST(TestGetDecyclerCounter, for_mail_with_decycler_header_with_wrong_value_should_return_zero) {
    auto message = NUtil::MakeSegment(
        "Header-1: 1\r\nX-Yandex-Fwd: wrong_value\r\n\r\nBody"
    );
    auto [headers, _] = ParseMessage(message);

    EXPECT_EQ(GetDecyclerCounter(headers), 0u);
}

TEST(TestGetDecyclerCounter, for_mail_without_decycler_value_should_return_zero) {
    auto message = NUtil::MakeSegment(
        "Header-1: 1\r\n\r\nBody"
    );
    auto [headers, _] = ParseMessage(message);

    EXPECT_EQ(GetDecyclerCounter(headers), 0u);
}

TEST(TestGetDecyclerCounter, test_build_not_empty_decycler_header) {
    EXPECT_EQ(BuildDecyclerHeader(220), "X-Yandex-Fwd: 220\r\n");
}

TEST(TestGetDecyclerCounter, test_build_empty_decycler_header) {
    EXPECT_EQ(BuildDecyclerHeader(0), "");
}

TEST(TestDetectCycle, for_counter_less_than_ttl_should_return_false) {
    auto options = MakeOptions();
    auto config = MakeConfig();
    options.DetectCycle = true;
    config->Decycler.ttl = 1;
    config->Decycler.reject = true;
    EXPECT_FALSE(DetectCycle(config, options, 0));
}

TEST(TestDetectCycle, for_counter_more_than_ttl_should_return_true) {
    auto options = MakeOptions();
    auto config = MakeConfig();
    options.DetectCycle = true;
    config->Decycler.ttl = 1;
    config->Decycler.reject = true;
    EXPECT_TRUE(DetectCycle(config, options, 2));
}

TEST(TestDetectCycle, for_false_detect_cycle_option_should_return_false) {
    auto options = MakeOptions();
    auto config = MakeConfig();
    options.DetectCycle = false;
    config->Decycler.ttl = 1;
    config->Decycler.reject = true;
    EXPECT_FALSE(DetectCycle(config, options, 2));
}

TEST(TestDetectCycle, for_false_decycler_reject_config_should_return_false) {
    auto options = MakeOptions();
    auto config = MakeConfig();
    options.DetectCycle = true;
    config->Decycler.ttl = 1;
    config->Decycler.reject = false;
    EXPECT_FALSE(DetectCycle(config, options, 2));
}

} // namespace anonymous
