#include <mail/nwsmtp/src/dkim/adkim.h>
#include <mail/nwsmtp/src/header_parser.h>
#include <mail/nwsmtp/src/header_storage.h>
#include <mail/nwsmtp/src/types.h>
#include <mail/nwsmtp/ut/utils.h>

#include <nwsmtp/resolver.h>

#include <library/cpp/resource/resource.h>

#include <gtest/gtest.h>


class TDkimCheck : public testing::Test {
public:
    using TResult = std::pair<NNwSmtp::dkim_check::Output, std::string>;

    void SetUp() override {
        NNwSmtp::init_dkim();
        Check = std::make_shared<NNwSmtp::dkim_check>(IoService);
    }

    TResult Run(NNwSmtp::TBuffer message) {
        HeaderStorage.Clear();
        auto [headers, body] = NNwSmtp::ParseMessage(message);

        auto fromHeaders = headers.GetHeaders("From");
        if (fromHeaders) {
            NNwSmtp::find_domain(std::string(fromHeaders.begin()->Value.begin(), fromHeaders.begin()->Value.end()),
                FromDomain);
        }

        NNwSmtp::dkim_check::input input{
            FromDomain,
            std::make_shared<NNwSmtp::THeaderStorageImpl >(headers),
            body,
            [this](auto message) { CheckMessage = message; }};

        NNwSmtp::dkim_check::Output checkResult;
        Check->start(input, {},
            [&checkResult](auto output) {
                checkResult = output;
            });
        IoService.run();
        return {checkResult, CheckMessage};
    }

    TResult Run(const std::string& name) {
        return Run(NTesting::LoadMessageAsBuffer(name));
    }

private:
    boost::asio::io_service IoService;
    NNwSmtp::THeaderStorageImpl HeaderStorage;
    std::string FromDomain;
    std::shared_ptr<NNwSmtp::dkim_check> Check;
    std::string CheckMessage;
};

const std::string YANDEX_PUBLIC_KEY =
    "mail._domainkey.yandex.ru: v=DKIM1; g=*; k=rsa; "
    "p=MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDEc6Lkc9k"
    "LHjIxLkeszz1dYzGIfPH8qaUx2wLojYefUzZiCjyl0s/YT17W"
    "JMfGFZkl0gHgkEj5/I2C72MmaHVtTFzNqD48ZuqVydlDyfLed0"
    "A6vxb+MS34DIbpCgCi0HxQO1QRG7PechKza0iazWTIAQ1xRU24"
    "ZYM70kGDzhFHSwIDAQAB";

const std::string MAILRU_PUBLIC_KEY =
    "mail2._domainkey.mail.ru: v=DKIM1; k=rsa; "
    "p=MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQC8msGcERt9"
    "i1AqEs6Dl5n0btBDj4W3IjzNg1xAExTn1Wb7wjRk9ed8oJ6Xnx"
    "n2jSYwbt3G65lW8LK/8vVdx2arFexHgKmOXT5RKIeiYFkHmLEt"
    "ycrRkyJHr6n7rsjwlFSayXnxrM0xbum3oHXgNJUI1XQXJNoQPm"
    "AXoMCbi2yB7QIDAQAB";

TEST_F(TDkimCheck, Pass) {
    auto [res, msg] = Run("pass.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::pass);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, YANDEX_PUBLIC_KEY);
}

TEST_F(TDkimCheck, ChunkedBody) {
    auto buffer = NTesting::LoadMessageAsBuffer("pass.eml", /*chunks = */ 3);
    ASSERT_GE(buffer.end_fragment() - buffer.begin_fragment(), 3);

    auto [res, msg] = Run(buffer);
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::pass);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, YANDEX_PUBLIC_KEY);
}

TEST_F(TDkimCheck, HeadersReorderedPass) {
    auto [res, msg] = Run("reordered_headers.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::pass);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, YANDEX_PUBLIC_KEY);
}

// Test for messages from Wildberries
TEST_F(TDkimCheck, HeadersBeforeSignature) {
    auto [res, msg] = Run("headers_before_signature.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::pass);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, YANDEX_PUBLIC_KEY);
}

TEST_F(TDkimCheck, DuplicateSubjectHeaderPass) {
    auto [res, msg] = Run("duplicate_header.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::pass);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, YANDEX_PUBLIC_KEY);
}

TEST_F(TDkimCheck, ModifiedNotSignedHeaderPass) {
    auto [res, msg] = Run("modified_not_signed_header.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::pass);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, YANDEX_PUBLIC_KEY);
}

TEST_F(TDkimCheck, NoFromHeader) {
    auto [res, msg] = Run("no_from.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::neutral);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, "verify error: Syntax error");
}

TEST_F(TDkimCheck, NoCrLfInBody) {
    auto [res, msg] = Run("no_crlf.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::neutral);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, "verify error: Syntax error");
}

TEST_F(TDkimCheck, NoTxtRecordForDomainNeutral) {
    auto [res, msg] = Run("no_txt_record_for_domain.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::neutral);
    ASSERT_EQ(res.domain, "mail.yandex.net");
    ASSERT_EQ(res.identity, "@mail.yandex.net");
    ASSERT_EQ(msg, "verify error: failed all DNS requests");
}

TEST_F(TDkimCheck, NonExistentDomainNeutral) {
    auto [res, msg] = Run("non_existent_domain.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::neutral);
    ASSERT_EQ(res.domain, "abc.foo.bar");
    ASSERT_EQ(res.identity, "@abc.foo.bar");
    ASSERT_EQ(msg, "verify error: failed all DNS requests");
}

TEST_F(TDkimCheck, BodyHashMissmatchFail) {
    auto [res, msg] = Run("modified_body.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::fail);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, "verify error: body hash mismatch");
}

TEST_F(TDkimCheck, AbsentSignedHeaderFail) {
    auto [res, msg] = Run("absent_signed_header.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::fail);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, "verify error: Bad signature");
}

TEST_F(TDkimCheck, ModifiedSignedHeaderFail) {
    auto [res, msg] = Run("modified_signed_header.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::fail);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, "verify error: Bad signature");
}

TEST_F(TDkimCheck, SeveralSignatures) {
    auto [res, msg] = Run("several_signatures.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::pass);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, MAILRU_PUBLIC_KEY);
}

TEST_F(TDkimCheck, PrimarySignatureFail) {
    auto [res, msg] = Run("primary_signature_fail.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::fail);
    ASSERT_EQ(res.domain, "mail.ru");
    ASSERT_EQ(res.identity, "@mail.ru");
    ASSERT_EQ(msg, "verify error: Bad signature");
}

TEST_F(TDkimCheck, PrimarySignatureInvalid) {
    auto [res, msg] = Run("primary_signature_invalid.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::pass);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, YANDEX_PUBLIC_KEY);
}

TEST_F(TDkimCheck, PrimarySignatureNeutral) {
    auto [res, msg] = Run("primary_signature_neutral.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::pass);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, YANDEX_PUBLIC_KEY);
}

TEST_F(TDkimCheck, SeveralInvalidSignatures) {
    auto [res, msg] = Run("several_invalid_signatures.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::neutral);
    ASSERT_EQ(res.domain, "");
    ASSERT_EQ(res.identity, "");
    ASSERT_EQ(msg, "verify error: fail to find valid signature");
}

TEST_F(TDkimCheck, SeveralNeutralSignatures) {
    auto [res, msg] = Run("several_neutral_signatures.eml");
    ASSERT_TRUE(res.status == NNwSmtp::dkim_check::neutral);
    ASSERT_EQ(res.domain, "yandex.ru");
    ASSERT_EQ(res.identity, "@yandex.ru");
    ASSERT_EQ(msg, "verify error: failed all DNS requests");
}
