#include <mail/nwsmtp/src/dkim/adkim.h>
#include <mail/nwsmtp/src/dkim/signature_parser.h>
#include <mail/nwsmtp/src/header_parser.h>
#include <mail/nwsmtp/src/header_storage.h>
#include <mail/nwsmtp/src/dkim/dkim_options.h>
#include <mail/nwsmtp/src/utils.h>
#include <mail/nwsmtp/src/types.h>
#include <mail/nwsmtp/ut/utils.h>
#include <mail/nwsmtp/ut/http_cluster_client_test_impl.h>
#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/options.h>

#include <nwsmtp/resolver.h>

#include <boost/property_tree/json_parser.hpp>

#include <util/generic/algorithm.h>
#include <library/cpp/resource/resource.h>

#include <gtest/gtest.h>

using namespace NNwSmtp;

using TFields = NDkim::TSignatureParser::TFields;

DkimOptions::KeyEntry MakeYandexKey(const std::string& secretKey) {
    return {"mail", "yandex.ru", secretKey};
}

class TDkimSign : public testing::Test {
public:
    using TResult = std::pair<std::string, dkim_sign::Output>;

    void SetUp() override {
        init_dkim();
        yplatform::log::init_global_log_console();
        NNwSmtp::glog = std::make_shared<NNwSmtp::TLog>();
        NNwSmtp::gconfig = std::make_shared<NNwSmtp::Options>();
    }

    void SetYandexKey(const std::string& key) {
        Opts.signOpts.keys["yandex.ru"] = MakeYandexKey(key);
    }

    void SetHeadersForSign(const std::vector<std::string>& headers) {
        auto& signHdrs = Opts.signOpts.sign_hdrs;
        signHdrs.clear();
        std::copy(headers.begin(), headers.end(), std::inserter(signHdrs, signHdrs.begin()));
    }

    void EnableFouras() {
        Opts.signOpts.use_fouras = true;
    }

    TResult Sign(
        const std::string& email,
        NNwSmtp::THeaderStorageImpl headers,
        NNwSmtp::TBufferRange body,
        NTesting::THttpClusterClientMock::TAsyncRunImpl callback = {})
    {
        boost::asio::io_context ios;
        std::string resError;
        dkim_sign::Output resOutput;
        auto httpClient = std::make_shared<NTesting::THttpClusterClientMock>(std::move(callback));
        auto sign = std::make_shared<dkim_sign>(ios, Opts.signOpts);
        dkim_sign::input input{
            email,
            std::make_shared<NNwSmtp::THeaderStorageImpl>(headers),
            body, "session_id", "envelope_id",
            [&](const auto& message){
                LogMessage = message;
            }
        };
        sign->start(
            input,
            httpClient,
            [&resError, &resOutput](auto error, auto output) {
                resError = error;
                resOutput = output;
            });
        ios.run();
        return {resError, resOutput};
    }

protected:
    auto MakeHttpHandlerReturning404() {
        return [&](
            yhttp::request, NTesting::THttpClusterClientMock::options,
            NTesting::THttpClusterClientMock::callback_type callback)
        {
            ASSERT_FALSE(HandlerUsed);
            HandlerUsed = true;
            std::string response = R"({
                "response": {
                    "code": "not_found",
                    "message": "Object not found",
                    "params": {}
                },
                "status": "error"
            })";

            const auto status{404};
            callback({}, {status, {}, std::move(response), {}});
        };
    }

    void TestSignAllHeaders(
        const std::string& email,
        NTesting::THttpClusterClientMock::TAsyncRunImpl callback = {})
    {
        auto msg = NTesting::LoadMessageAsBuffer("for_signing.eml");
        auto [headers, body] = ParseMessage(msg);

        SetYandexKey(NResource::Find("mail.private"));
        SetHeadersForSign({});
        auto [error, res] = Sign(email, headers, body, std::move(callback));
        ASSERT_EQ(error, "");
        ASSERT_FALSE(res.hasError);

        ASSERT_TRUE(Contains(res.header, "DKIM-Signature: v=1; a=rsa-sha256; c=relaxed/relaxed; d=yandex.ru; "
                "s=mail; t="));
        TFields expected = {{"date"}, {"from"}, {"message-id"}, {"mime-version"}, {"subject"}, {"to"}};
        ASSERT_EQ(ExtractFields(res.header), expected);
    }

    std::vector<std::string> ExtractFields(const std::string& signHeader) const {
        if (signHeader.empty()) {
            return {};
        }

        auto iter = Find(signHeader, ':');
        std::string headerValue{iter + 1, signHeader.end()};

        TFields fields;
        std::string _;
        NDkim::TSignatureParser(_, _, fields).Parse(headerValue);

        Sort(fields);
        return fields;
    }

protected:
    bool HandlerUsed{false};
    std::string LogMessage;
private:
    DkimOptions Opts;
};

TEST_F(TDkimSign, BadEmailFails) {
    auto [error, output] = Sign("yandex.ru", {}, {});
    ASSERT_EQ(error, "dkim: failed to parse email: yandex.ru");
    ASSERT_TRUE(output.hasError);
}

TEST_F(TDkimSign, NoKeyForDomainFails) {
    auto [error, output] = Sign("foo@yandex.ru", {}, {});
    ASSERT_EQ(error, "dkim: the specified domain not found in dkim key list: yandex.ru");
    ASSERT_FALSE(output.hasError);
}

TEST_F(TDkimSign, NoKeyForDomainFailsWithFouras) {
    EnableFouras();
    auto [error, output] = Sign("foo@yandex.ru", {}, {}, MakeHttpHandlerReturning404());
    ASSERT_TRUE(HandlerUsed);
    ASSERT_EQ(error, "dkim: key for yandex.ru not received from Fouras [Domain not found]"
            "; the specified domain not found in dkim key list: yandex.ru");
    ASSERT_FALSE(output.hasError);
}

TEST_F(TDkimSign, BadKeyFails) {
    SetYandexKey("FOO");

    auto msg = NTesting::LoadMessageAsBuffer("for_signing.eml");
    auto [headers, body] = ParseMessage(msg);

    auto [error, output] = Sign("foo@yandex.ru", headers, body);
    ASSERT_EQ(error, "dkim: d2i_PrivateKey_bio() failed");
    ASSERT_TRUE(output.hasError);
}

TEST_F(TDkimSign, SignAllHeadersWithDomainKey) {
    TestSignAllHeaders("local@yandex.ru");
    ASSERT_EQ(LogMessage, "dkim key got from file system for domain: yandex.ru");
}

TEST_F(TDkimSign, SignAllHeadersWithSubdomainKey) {
    TestSignAllHeaders("local@mail.yandex.ru");
    ASSERT_EQ(LogMessage, "dkim key got from file system for subdomain: yandex.ru");
}

TEST_F(TDkimSign, SignAllHeadersWithSubdomainKeyWithFourasFailed) {
    EnableFouras();
    TestSignAllHeaders("local@mail.yandex.ru", MakeHttpHandlerReturning404());
    ASSERT_EQ(LogMessage, "failed to get dkim key from Fouras for domain: mail.yandex.ru"
            ", got from file system for subdomain: yandex.ru");
}

TEST_F(TDkimSign, SignCustomHeadersSuccess) {
    auto msg = NTesting::LoadMessageAsBuffer("for_signing.eml");
    auto [headers, body] = ParseMessage(msg);

    SetYandexKey(NResource::Find("mail.private"));
    SetHeadersForSign({"from", "date", "subject"});
    auto [error, res] = Sign("foo@yandex.ru", headers, body);
    ASSERT_EQ(error, "");
    ASSERT_FALSE(res.hasError);

    ASSERT_TRUE(Contains(res.header, "DKIM-Signature: v=1; a=rsa-sha256; c=relaxed/relaxed; d=yandex.ru; s=mail; t="));
    TFields expected = {{"date", "from", "subject"}};
    ASSERT_EQ(ExtractFields(res.header), expected);
}

TEST_F(TDkimSign, SignWithGetKeyFromFouras) {
    auto msg = NTesting::LoadMessageAsBuffer("for_signing.eml");
    auto [headers, body] = ParseMessage(msg);

    auto key = MakeYandexKey(NResource::Find("mail.private"));
    auto httpHandler = [&, key](yhttp::request, NTesting::THttpClusterClientMock::options,
        NTesting::THttpClusterClientMock::callback_type callback)
    {
        ASSERT_FALSE(HandlerUsed);
        HandlerUsed = true;

        boost::property_tree::ptree root, response;
        root.put("status", "ok");
        response.put("public_key", "\"v=DKIM1; k=rsa; t=s; p=...");
        response.put("private_key", key.secretkey);
        response.put("domain", key.domain);
        response.put("selector", key.selector);
        response.put<bool>("enabled", true);
        root.put_child("response", response);

        std::stringstream ss;
        boost::property_tree::write_json(ss, root);

        callback({}, {200, {}, ss.str(), ""});
    };

    EnableFouras();
    SetHeadersForSign({});
    auto [error, res] = Sign("foo@yandex.ru", headers, body, httpHandler);
    ASSERT_TRUE(HandlerUsed);
    ASSERT_EQ(error, "");
    ASSERT_FALSE(res.hasError);
    ASSERT_EQ(LogMessage, "dkim key got from Fouras for domain: yandex.ru");

    ASSERT_TRUE(Contains(res.header, "DKIM-Signature: v=1; a=rsa-sha256; c=relaxed/relaxed; d=yandex.ru; s=mail; t="));
    TFields expected = {{"date"}, {"from"}, {"message-id"}, {"mime-version"}, {"subject"}, {"to"}};
    ASSERT_EQ(ExtractFields(res.header), expected);
}
