#include <mail/nwsmtp/src/dkim/signature_parser.h>

#include <gtest/gtest.h>

#include <algorithm>
#include <memory>

using namespace NNwSmtp;

class TTestSignatureParser : public testing::Test {
protected:
    void SetUp() override {
        Domain.clear();
        Selector.clear();
        Fields.clear();
        Parser = std::make_unique<NDkim::TSignatureParser>(Domain, Selector, Fields);
    }

    bool CompareFields(const NDkim::TSignatureParser::TFields& fields) {
        if (Fields.size() != fields.size()) {
            return false;
        }
        return std::is_permutation(Fields.begin(), Fields.end(), fields.begin());
    }

protected:
    std::string Domain;
    std::string Selector;
    NDkim::TSignatureParser::TFields Fields;
    std::unique_ptr<NDkim::TSignatureParser> Parser;
};


TEST_F(TTestSignatureParser, GmailSignature) {
    const std::string signature = R"(
        v=1; a=rsa-sha256; c=relaxed/relaxed;
        d=gmail.com; s=20161025;
        h=mime-version:in-reply-to:references:from:date:message-id:subject:to;
        bh=kZPL0f+eZIw9ZKKH+QZHFy3Zykbio1Vlg2c0YmsUf4Q=;
        b=WzydZWTJ0pKF8liG3cHgsHIipU7IxRtIL5f2gPMk+u8XlyVsTYt3SpBpJKgDU6Dbq4
        VW8oqUIvHpvu4OPBdCbtMebFWdpQTh4Nk6LlQIzTMVRK8iHiPzcRAn+4pg2SRCO/haXt
        7NTi2d3t/XvLb61+OiVsX+OCT9tsP6s0VTXYvtSByjXHF8EsVyYIDGf3/QtspZyfKP+B
        Om0EX5DNyF0sr61LGnyRYGusZYctQ6Yx5mioUDXRxVp0IYuhcx3bnEWjN/mu4m4NcB3Y
        Yf9YhKYiQtS61iAJ2kp0T4sL1bV6oCGHccL67H1EUmn+aKFz6QIGtPtfcNkXlfmvOoxW
        lYNQ==
    )";

    ASSERT_TRUE(Parser->Parse(signature));

    ASSERT_EQ(Domain, "gmail.com");
    ASSERT_EQ(Selector, "20161025");
    ASSERT_TRUE(CompareFields(
        {"mime-version", "in-reply-to", "references", "from", "date", "message-id", "subject", "to"}
    ));
}

TEST_F(TTestSignatureParser, GmailSignatureWithSpaces) {
    const std::string signature = R"(
        v=1; a=rsa-sha256; c=relaxed/relaxed;
        d= gmail.com  ; s=20161025;
        h=mime-version : in-reply-to:  references:from  :date : message-id : subject:to;
        bh=kZPL0f+eZIw9ZKKH+QZHFy3Zykbio1Vlg2c0YmsUf4Q=;
        b=WzydZWTJ0pKF8liG3cHgsHIipU7IxRtIL5f2gPMk+u8XlyVsTYt3SpBpJKgDU6Dbq4
        VW8oqUIvHpvu4OPBdCbtMebFWdpQTh4Nk6LlQIzTMVRK8iHiPzcRAn+4pg2SRCO/haXt
        7NTi2d3t/XvLb61+OiVsX+OCT9tsP6s0VTXYvtSByjXHF8EsVyYIDGf3/QtspZyfKP+B
        Om0EX5DNyF0sr61LGnyRYGusZYctQ6Yx5mioUDXRxVp0IYuhcx3bnEWjN/mu4m4NcB3Y
        Yf9YhKYiQtS61iAJ2kp0T4sL1bV6oCGHccL67H1EUmn+aKFz6QIGtPtfcNkXlfmvOoxW
        lYNQ==
    )";

    ASSERT_TRUE(Parser->Parse(signature));

    ASSERT_EQ(Domain, "gmail.com");
    ASSERT_EQ(Selector, "20161025");
    ASSERT_TRUE(CompareFields(
        {"mime-version", "in-reply-to", "references", "from", "date", "message-id", "subject", "to"}
    ));
}

TEST_F(TTestSignatureParser, YandexSignature) {
    const std::string signature = R"(
        v=1; a=rsa-sha256; c=relaxed/relaxed; d=yandex.ru; s=mail; t=1544355983;
        bh=NYgnuab3ygDvkj1bkSDezaox9JdXq4csfqrTmLSH5wY=;
        h=From:To:Subject:Date:Message-Id;
        b=bH2N0eMi8zpd4gzrPXpJmd4nVCMsg1bakXNGOm7cpBlktrxJ7w+z6U8DLThFBHAMz
        ms8LPLttke+TTZ6bIig0u3fYc/12hkdfL/T81E9yPT6Ef4TcPLFHKWeDFvmvPIdKH4
        AKsOy92I3vSaVSk/82PYQh0R6vPzRifECjqqDxwc=
    )";

    ASSERT_TRUE(Parser->Parse(signature));

    ASSERT_EQ(Domain, "yandex.ru");
    ASSERT_EQ(Selector, "mail");
    ASSERT_TRUE(CompareFields({"from", "to", "subject", "date", "message-id"}));
}

TEST_F(TTestSignatureParser, SignatureWithFieldsListWithDuplicate) {
    const std::string signature = R"(
       v=1; a=rsa-sha256; c=relaxed/relaxed; d=yandex.ru; s=mail; t=1544355983;
       bh=NYgnuab3ygDvkj1bkSDezaox9JdXq4csfqrTmLSH5wY=;
       h=From:To:Subject:suBject:Date:to:Message-Id;
       b=bH2N0eMi8zpd4gzrPXpJmd4nVCMsg1bakXNGOm7cpBlktrxJ7w+z6U8DLThFBHAMz
       ms8LPLttke+TTZ6bIig0u3fYc/12hkdfL/T81E9yPT6Ef4TcPLFHKWeDFvmvPIdKH4
       AKsOy92I3vSaVSk/82PYQh0R6vPzRifECjqqDxwc=
    )";

    ASSERT_TRUE(Parser->Parse(signature));

    ASSERT_EQ(Domain, "yandex.ru");
    ASSERT_EQ(Selector, "mail");
    ASSERT_TRUE(CompareFields({"from", "to", "to", "subject", "subject", "date", "message-id"}));
}

// This is contrary to the RFC, but we specifically allow spaces in header names
TEST_F(TTestSignatureParser, SignatureWithSpacesInHeaderNames) {
    const std::string signature = R"(
       v=1; a=rsa-sha256; c=relaxed/relaxed; d=yandex.ru; s=mail; t=1544355983;
       bh=NYgnuab3ygDvkj1bkSDezaox9JdXq4csfqrTmLSH5wY=;
       h=From:To:Sub ject:Date  :Mess  age-I d;
       b=bH2N0eMi8zpd4gzrPXpJmd4nVCMsg1bakXNGOm7cpBlktrxJ7w+z6U8DLThFBHAMz
       ms8LPLttke+TTZ6bIig0u3fYc/12hkdfL/T81E9yPT6Ef4TcPLFHKWeDFvmvPIdKH4
       AKsOy92I3vSaVSk/82PYQh0R6vPzRifECjqqDxwc=
    )";

    ASSERT_TRUE(Parser->Parse(signature));

    ASSERT_EQ(Domain, "yandex.ru");
    ASSERT_EQ(Selector, "mail");
    ASSERT_TRUE(CompareFields({"from", "to", "sub ject", "date", "mess  age-i d"}));
}

TEST_F(TTestSignatureParser, SignatureWithFieldsListWithEmptyHeaderName) {
    const std::string signature = R"(
       v=1; a=rsa-sha256; c=relaxed/relaxed; d=yandex.ru; s=mail; t=1544355983;
       bh=NYgnuab3ygDvkj1bkSDezaox9JdXq4csfqrTmLSH5wY=;
       h=From::To:Subject:Date:Message-Id;
       b=bH2N0eMi8zpd4gzrPXpJmd4nVCMsg1bakXNGOm7cpBlktrxJ7w+z6U8DLThFBHAMz
       ms8LPLttke+TTZ6bIig0u3fYc/12hkdfL/T81E9yPT6Ef4TcPLFHKWeDFvmvPIdKH4
       AKsOy92I3vSaVSk/82PYQh0R6vPzRifECjqqDxwc=
    )";

    ASSERT_FALSE(Parser->Parse(signature));
}

TEST_F(TTestSignatureParser, SignatureWithEmptyFieldsList) {
    const std::string signature = R"(
       v=1; a=rsa-sha256; c=relaxed/relaxed; d=yandex.ru; s=mail; t=1544355983;
       bh=NYgnuab3ygDvkj1bkSDezaox9JdXq4csfqrTmLSH5wY=;
       h=;
       b=bH2N0eMi8zpd4gzrPXpJmd4nVCMsg1bakXNGOm7cpBlktrxJ7w+z6U8DLThFBHAMz
       ms8LPLttke+TTZ6bIig0u3fYc/12hkdfL/T81E9yPT6Ef4TcPLFHKWeDFvmvPIdKH4
       AKsOy92I3vSaVSk/82PYQh0R6vPzRifECjqqDxwc=
    )";

    ASSERT_FALSE(Parser->Parse(signature));
}

TEST_F(TTestSignatureParser, SignatureWithMissingFieldsList) {
    const std::string signature = R"(
       v=1; a=rsa-sha256; c=relaxed/relaxed; d=yandex.ru; s=mail; t=1544355983;
       bh=NYgnuab3ygDvkj1bkSDezaox9JdXq4csfqrTmLSH5wY=;
       b=bH2N0eMi8zpd4gzrPXpJmd4nVCMsg1bakXNGOm7cpBlktrxJ7w+z6U8DLThFBHAMz
       ms8LPLttke+TTZ6bIig0u3fYc/12hkdfL/T81E9yPT6Ef4TcPLFHKWeDFvmvPIdKH4
       AKsOy92I3vSaVSk/82PYQh0R6vPzRifECjqqDxwc=
    )";

    ASSERT_FALSE(Parser->Parse(signature));
}

TEST_F(TTestSignatureParser, SignatureWithInvalidTagName) {
    const std::string signature = R"(
       v=1; 7ar=rsa-sha256; c=relaxed/relaxed; d=yandex.ru; s=mail; t=1544355983;
       bh=NYgnuab3ygDvkj1bkSDezaox9JdXq4csfqrTmLSH5wY=;
       h=From:To:Subject:Date:Message-Id;
       b=bH2N0eMi8zpd4gzrPXpJmd4nVCMsg1bakXNGOm7cpBlktrxJ7w+z6U8DLThFBHAMz
       ms8LPLttke+TTZ6bIig0u3fYc/12hkdfL/T81E9yPT6Ef4TcPLFHKWeDFvmvPIdKH4
       AKsOy92I3vSaVSk/82PYQh0R6vPzRifECjqqDxwc=
    )";

    ASSERT_FALSE(Parser->Parse(signature));
}
