#include <mail/nwsmtp/src/dmarc/dmarc.h>

#include <gtest/gtest.h>

using namespace NNwSmtp;

TEST(DmarcCheckPolicyTest, PoliciesWhenNoSpfAndDkim) {
    using namespace NOpenDmarc;
    struct TFixture {
        std::string TxtRecord;
        EDmarcPolicy Policy;
    };
    std::vector<TFixture> fixtures{
        {"v=DMARC1; p=none;", EDmarcPolicy::None},
        {"v=DMARC1; p=reject;", EDmarcPolicy::Reject},
        {"v=DMARC1; p=quarantine;", EDmarcPolicy::Quarantine},
    };

    TContext ctx;
    for (const auto& fixture : fixtures) {
        TPolicyManager policyManager(ctx.Get(), "foo.com", fixture.TxtRecord);
        auto policy = policyManager.AdvisePolicy();
        ASSERT_TRUE(policy == fixture.Policy);
        ctx.Reset();
    }
}

TEST(DmarcCheckPolicyTest, PassWhenDkimPass) {
    using namespace NOpenDmarc;

    TContext ctx;
    TPolicyManager policyManager(ctx.Get(), "example.com", "v=DMARC1; p=reject;");
    policyManager.SetDkim("example.com", EOutcome::Pass);

    auto policy = policyManager.AdvisePolicy();
    ASSERT_TRUE(policy == EDmarcPolicy::Pass);
}

TEST(DmarcCheckPolicyTest, PassWhenSpfPass) {
    using namespace NOpenDmarc;

    TContext ctx;
    TPolicyManager policyManager(ctx.Get(), "example.com", "v=DMARC1; p=reject;");
    policyManager.SetSpf("example.com", EOutcome::Pass, ESpfOrigin::MailFrom);

    auto policy = policyManager.AdvisePolicy();
    ASSERT_TRUE(policy == EDmarcPolicy::Pass);
}

TEST(DmarcCheckPolicyTest, GetAlignments) {
    using namespace NOpenDmarc;

    TContext ctx;
    TPolicyManager policyManager(ctx.Get(), "example.com", "v=DMARC1; p=reject;");
    policyManager.SetSpf("example.com", EOutcome::Pass, ESpfOrigin::MailFrom);
    policyManager.SetDkim("example.com", EOutcome::Pass);

    policyManager.AdvisePolicy();

    auto alignments = policyManager.GetAlignments();
    ASSERT_TRUE(alignments.Dkim == EAlignment::Pass);
    ASSERT_TRUE(alignments.Spf == EAlignment::Pass);
}

TEST(DmarcCheckPolicyTest, RejectForSubdomainWhenAlignmentsStrict) {
    using namespace NOpenDmarc;

    TContext ctx;
    TPolicyManager policyManager(ctx.Get(), "bar.com", "v=DMARC1; p=reject; adkim=s; aspf=s;");
    policyManager.SetSpf("foo.bar.com", EOutcome::Pass, ESpfOrigin::MailFrom);
    policyManager.SetDkim("foo.bar.com", EOutcome::Pass);

    auto policy = policyManager.AdvisePolicy();
    ASSERT_TRUE(policy == EDmarcPolicy::Reject);

    auto alignments = policyManager.GetAlignments();
    ASSERT_TRUE(alignments.Dkim == EAlignment::Fail);
    ASSERT_TRUE(alignments.Spf == EAlignment::Fail);
}

TEST(DmarcCheckPolicyTest, PassForSubdomainWhenAlignmentsRelaxed) {
    using namespace NOpenDmarc;

    TContext ctx;
    TPolicyManager policyManager(ctx.Get(), "bar.com", "v=DMARC1; p=reject; adkim=r; aspf=r;");
    policyManager.SetSpf("foo.bar.com", EOutcome::Pass, ESpfOrigin::MailFrom);
    policyManager.SetDkim("foo.bar.com", EOutcome::Pass);

    auto policy = policyManager.AdvisePolicy();
    ASSERT_TRUE(policy == EDmarcPolicy::Pass);

    auto alignments = policyManager.GetAlignments();
    ASSERT_TRUE(alignments.Dkim == EAlignment::Pass);
    ASSERT_TRUE(alignments.Spf == EAlignment::Pass);
}

TEST(DmarcCheckPolicyTest, OrganizationalDomainFallback) {
    using namespace NOpenDmarc;

    struct TFixture {
        std::string Sp;
        EDmarcPolicy ExpectedPolicy;
    };
    std::vector<TFixture> fixtures = {
        {"none", EDmarcPolicy::None},
        {"reject", EDmarcPolicy::Reject},
        {"quarantine", EDmarcPolicy::Quarantine}
    };

    for (const auto& fixture : fixtures) {
        TContext ctx;
        std::string txtRecord = "v=DMARC1; p=reject; sp=";
        txtRecord += fixture.Sp;
        txtRecord += "; adkim=s; aspf=s;";

        TPolicyManager policyManager(ctx.Get(), "example.com", txtRecord, "example.com");
        policyManager.SetSpf("foo.example.com", EOutcome::Fail, ESpfOrigin::MailFrom);
        policyManager.SetDkim("foo.example.com", EOutcome::Fail);

        auto policy = policyManager.AdvisePolicy();
        ASSERT_TRUE(policy == fixture.ExpectedPolicy);

        auto alignments = policyManager.GetAlignments();
        ASSERT_TRUE(alignments.Dkim == EAlignment::Fail);
        ASSERT_TRUE(alignments.Spf == EAlignment::Fail);
    }
}

TEST(DmarcCheckPolicyTest, AspfRelaxedAndNotAligned) {
    using namespace NOpenDmarc;

    TContext ctx;
    // "aspf=r" and "adkim=r" by default.
    std::string txtRecord = "v=DMARC1;p=reject;";

    TPolicyManager policyManager(ctx.Get(), "mail.ru", txtRecord);
    policyManager.SetSpf("anita.ua", EOutcome::Pass, ESpfOrigin::MailFrom);

    // "p=reject" and nothing is aligned, so results to "Reject".
    auto policy = policyManager.AdvisePolicy();
    ASSERT_TRUE(policy == EDmarcPolicy::Reject);

    auto alignments = policyManager.GetAlignments();
    ASSERT_TRUE(alignments.Dkim == EAlignment::Fail);
    // "aspf=r" and spf domain is not subdomain of "mail.ru", so it's not aligned.
    ASSERT_TRUE(alignments.Spf == EAlignment::Fail);
}

TEST(DmarcCheckPolicyTest, AspfStrictAndNotAligned) {
    using namespace NOpenDmarc;

    TContext ctx;
    std::string txtRecord = "v=DMARC1; p=quarantine; adkim=s; aspf=s;";

    TPolicyManager policyManager(ctx.Get(), "contact-center.ru", txtRecord);
    policyManager.SetSpf("vm10.trade.su", EOutcome::Pass, ESpfOrigin::MailFrom);
    policyManager.SetDkim("contact-center.ru", EOutcome::Fail);

    // "p=quarantine" and nothing is aligned, so results to "Quarantine".
    auto policy = policyManager.AdvisePolicy();
    ASSERT_TRUE(policy == EDmarcPolicy::Quarantine);

    auto alignments = policyManager.GetAlignments();
    ASSERT_TRUE(alignments.Dkim == EAlignment::Fail);
    // "aspf=s" and spf domain is not equal to "contact-center.ru", so it's not aligned.
    ASSERT_TRUE(alignments.Spf == EAlignment::Fail);
}

TEST(DmarcCheckPolicyTest, NoChecksAndBothRelaxed) {
    using namespace NOpenDmarc;

    TContext ctx;
    // "aspf=r" and "adkim=r" by default.
    std::string txtRecord = "v=DMARC1; p=quarantine;";

    TPolicyManager policyManager(ctx.Get(), "symmetron.ru", txtRecord);

    // "p=quarantine" and spf/dkim are not aligned, so results to "Quarantine"
    auto policy = policyManager.AdvisePolicy();
    ASSERT_TRUE(policy == EDmarcPolicy::Quarantine);

    auto alignments = policyManager.GetAlignments();
    ASSERT_TRUE(alignments.Dkim == EAlignment::Fail);
    ASSERT_TRUE(alignments.Spf == EAlignment::Fail);
}

TEST(DmarcCheckPolicyTest, AspfRelaxedAndSpfAligned) {
    using namespace NOpenDmarc;

    TContext ctx;
    // "aspf=r" and "adkim=r" by default.
    std::string txtRecord = "v=DMARC1; p=quarantine;";

    TPolicyManager policyManager(ctx.Get(), "symmetron.ru", txtRecord);
    policyManager.SetSpf("mail.symmetron.ru", EOutcome::Pass, ESpfOrigin::MailFrom);

    // "sp" is not specified and spf is aligned, so results to "Pass".
    auto policy = policyManager.AdvisePolicy();
    ASSERT_TRUE(policy == EDmarcPolicy::Pass);

    auto alignments = policyManager.GetAlignments();
    ASSERT_TRUE(alignments.Dkim == EAlignment::Fail);
    // "aspf=r" and spf domain is subdomain of "symmetron.ru", so spf is aligned.
    ASSERT_TRUE(alignments.Spf == EAlignment::Pass);
}

TEST(DmarcCheckPolicyTest, AdkimRelaxedAndDkimAligned) {
    using namespace NOpenDmarc;

    TContext ctx;
    // "aspf=r" and "adkim=r" by default.
    std::string txtRecord = "v=DMARC1; p=reject; pct=100;";

    TPolicyManager policyManager(ctx.Get(), "aol.com", txtRecord);
    policyManager.SetSpf("yandex.com", EOutcome::Pass, ESpfOrigin::MailFrom);
    policyManager.SetDkim("mx.aol.com", EOutcome::Pass);

    // `sp` is not specified and `dkim` is aligned, so results in "Pass".
    auto policy = policyManager.AdvisePolicy();
    ASSERT_TRUE(policy == EDmarcPolicy::Pass);

    auto alignments = policyManager.GetAlignments();
    // `adkim=r` and `dkim` domain is subdomain of "aol.com", so dkim is aligned.
    ASSERT_TRUE(alignments.Dkim == EAlignment::Pass);
    // `aspf=r` and `spf` domain is not subdomain of "aol.com"
    ASSERT_TRUE(alignments.Spf == EAlignment::Fail);
}


TEST(DmarcCheckPolicyTest, PMustBeAppliedToSp) {
    // MPROTO-3605
    using namespace NOpenDmarc;

    TContext ctx;
    // no sp, so p should apply to sp
    std::string txtRecord{R"(v=DMARC1; p=reject; fo=1;
        rua=mailto:dmarc_agg@auth.returnpath.net,mailto:dmarc-rua@yandex.ru;
        ruf=mailto:dmarc_afrf@auth.returnpath.net)"};

    // check for subdomain.
    TPolicyManager policyManager(ctx.Get(), "support.yandex.ru", txtRecord);

    // "p=reject", so results in "Reject".
    auto policy = policyManager.AdvisePolicy();
    ASSERT_TRUE(policy == EDmarcPolicy::Reject);

    auto alignments = policyManager.GetAlignments();
    ASSERT_TRUE(alignments.Dkim == EAlignment::Fail);
    ASSERT_TRUE(alignments.Spf == EAlignment::Fail);
}
