#include <mail/nwsmtp/src/dmarc/dmarc.h>

#include <gtest/gtest.h>

using namespace NNwSmtp;

struct TDmarcParseFixture {
    std::string TxtRecord;
    bool ExpectedResult;
};

using TDmarcParseFixtures = std::vector<TDmarcParseFixture>;

TDmarcParseFixtures MakeDmarcParseFixtures() {
    return {
        {"v=DMARC1; p=none; rf=afrf; rua=mailto:dmarc-a@abuse.net; ruf=mailto:dmarc-f@abuse.net", true},
        {"v=DMARC1; p=none;", true},
        {"v=DMARC1; p=reject;", true},
        {"v=DMARC1; p=quarantine;", true},
        {"v=DMARC1; p=n;", true},
        {"v=DMARC1; p=none; rua=ftp://abuse.com", true},
        {"v=DMARC1; p=none; ruf=mailto://abuse.com", true},
        {"v=DMARC1; p=none; ruf=mailto://abuse.com; foo=bar; buzz=happy;", true},
        {"", false},
        {"v=FOO", false},
        {"v=DMARC1; p=bob;", false},
        {"v=DMARC1; p=none; sp=bob;", false},
        {"v=DMARC1; p=none; adkim=bob;", false},
        {"v=DMARC1; p=none; aspf=bob;", false},
        {"v=DMARC1; p=none; rf=bob;", false},
        {"v=DMARC1; p=none; ri=bob;", false},
        {"v=DMARC1; p=none; pct=500;", false},
        {"v=DMARC1; pct=100;", false},
    };
}

TEST(DmarcParseTest, test_dmarc_record_parse_result) {
    using namespace NOpenDmarc;

    for (const auto& fixture : MakeDmarcParseFixtures()) {
        bool success = true;
        try {
            Parse("abuse.com", fixture.TxtRecord);
        } catch (const TDmarcParseError& e) {
            success = false;
        }
        ASSERT_EQ(success, fixture.ExpectedResult);
    }

}

TEST(DmarcParseTest, ParseDomainPolicy) {
    using namespace NOpenDmarc;

    struct TFixture {
        std::string TxtRecord;
        EDomainPolicy Policy;
    };

    std::vector<TFixture> fixtures{
        {"v=DMARC1; p=none;", EDomainPolicy::None},
        {"v=DMARC1; p=quarantine;", EDomainPolicy::Quarantine},
        {"v=DMARC1; p=reject;", EDomainPolicy::Reject}
    };

    for (const auto& fixture : fixtures) {
        auto record = Parse("", fixture.TxtRecord);
        ASSERT_TRUE(record->DomainPolicy == fixture.Policy);
    }
}

TEST(DmarcParseTest, ParseSubDomainPolicy) {
    using namespace NOpenDmarc;

    struct TFixture {
        std::string TxtRecord;
        EDomainPolicy Policy;
    };

    std::vector<TFixture> fixtures{
        {"v=DMARC1; p=none; sp=none;", EDomainPolicy::None},
        {"v=DMARC1; p=none; sp=quarantine;", EDomainPolicy::Quarantine},
        {"v=DMARC1; p=none; sp=reject;", EDomainPolicy::Reject}
    };

    for (const auto& fixture : fixtures) {
        auto record = Parse("", fixture.TxtRecord);
        ASSERT_TRUE(record->SubDomainPolicy == fixture.Policy);
    }
}

TEST(DmarcParseTest, ParseAdkim) {
    using namespace NOpenDmarc;

    struct TFixture {
        std::string TxtRecord;
        EAlignMode AlignMode;
    };

    std::vector<TFixture> fixtures{
        {"v=DMARC1; p=none;", EAlignMode::Relaxed},
        {"v=DMARC1; p=none; adkim=r;", EAlignMode::Relaxed},
        {"v=DMARC1; p=none; adkim=s;", EAlignMode::Strict}
    };

    for (const auto& fixture : fixtures) {
        auto record = Parse("", fixture.TxtRecord);
        ASSERT_TRUE(record->Adkim == fixture.AlignMode);
    }
}

TEST(DmarcParseTest, ParseAspf) {
    using namespace NOpenDmarc;

    struct TFixture {
        std::string TxtRecord;
        EAlignMode AlignMode;
    };

    std::vector<TFixture> fixtures{
        {"v=DMARC1; p=none;", EAlignMode::Relaxed},
        {"v=DMARC1; p=none; aspf=r;", EAlignMode::Relaxed},
        {"v=DMARC1; p=none; aspf=s;", EAlignMode::Strict}
    };

    for (const auto& fixture : fixtures) {
        auto record = Parse("", fixture.TxtRecord);
        ASSERT_TRUE(record->Aspf == fixture.AlignMode);
    }
}

TEST(DmarcParseTest, ParsePercentage) {
    using namespace NOpenDmarc;

    struct TFixture {
        std::string TxtRecord;
        uint32_t Percentage;
    };

    std::vector<TFixture> fixtures{
        {"v=DMARC1; p=none;", 100},
        {"v=DMARC1; p=none; pct=50;", 50},
        {"v=DMARC1; p=none; pct=99;", 99}
    };

    for (const auto& fixture : fixtures) {
        auto record = Parse("", fixture.TxtRecord);
        ASSERT_EQ(record->Pct, fixture.Percentage);
    }

    ASSERT_THROW(
        Parse("", "v=DMARC1; p=none; pct=101"),
        TDmarcParseError);
}

TEST(DmarcParseTest, ParseReturnAggregateAddresses) {
    using namespace NOpenDmarc;

    using TAddresses = std::vector<std::string>;
    struct TFixture {
        std::string TxtRecord;
        TAddresses Rua;
    };

    std::vector<TFixture> fixtures{
        {"v=DMARC1; p=none;", {}},
        {"v=DMARC1; p=none; rua=mailto:dmarc-a@abuse.net;",
        {"mailto:dmarc-a@abuse.net"}},
        {"v=DMARC1; p=none; rua=mailto:dmarc-a@abuse.net,ftp://dmarc-b@abuse.net;",
        {"mailto:dmarc-a@abuse.net", "ftp://dmarc-b@abuse.net"}},
    };

    for (const auto& fixture : fixtures) {
        auto record = Parse("", fixture.TxtRecord);
        ASSERT_EQ(record->Rua.size(), fixture.Rua.size());

        auto begin = record->Rua.begin();
        for (const auto& addr : fixture.Rua) {
            ASSERT_EQ(*begin, addr);
            ++begin;
        }
    }
}

TEST(DmarcParseTest, ParseFailureOptions) {
    using namespace NOpenDmarc;

    struct TFixture {
        std::string TxtRecord;
        EFailureOptions EFailureOptions;
    };

    std::vector<TFixture> fixtures{
        {"v=DMARC1; p=none;", EFailureOptions::Unspecified},
        {"v=DMARC1; p=none; ruf=mailto:dmarc-a@abuse.net;", EFailureOptions::All},
        {"v=DMARC1; p=none; ruf=mailto:dmarc-a@abuse.net; fo=0", EFailureOptions::All},
        {"v=DMARC1; p=none; ruf=mailto:dmarc-a@abuse.net; fo=1", EFailureOptions::Any},
        {"v=DMARC1; p=none; ruf=mailto:dmarc-a@abuse.net; fo=d", EFailureOptions::Dkim},
        {"v=DMARC1; p=none; ruf=mailto:dmarc-a@abuse.net; fo=s", EFailureOptions::Spf}
    };

    for (const auto& fixture : fixtures) {
        auto record = Parse("", fixture.TxtRecord);
        ASSERT_TRUE(record->FailureOpts == fixture.EFailureOptions);
    }
}

TEST(DmarcParseTest, AccessFieldsWithoutParsingThrows) {
    using namespace NOpenDmarc;

    TDmarcParser parser(nullptr);
    ASSERT_THROW(
        parser.GetDomainPolicy(),
        TDmarcParseError);
}

TEST(DmarcParseTest, SubsequentParsing) {
    using namespace NOpenDmarc;
    TContext ctx;
    TDmarcParser parser = TDmarcParser::Parse(ctx.Get(), "foo@example.com", "v=DMARC1; p=none");
    auto record = parser.MakeRecord();
    ASSERT_TRUE(record->DomainPolicy == EDomainPolicy::None);

    ctx.Reset();
    parser = TDmarcParser::Parse(ctx.Get(), "bar@example.com", "v=DMARC1; p=quarantine");
    record = parser.MakeRecord();
    ASSERT_TRUE(record->DomainPolicy == EDomainPolicy::Quarantine);
}
