#include <mail/nwsmtp/src/fouras/client.h>
#include <mail/nwsmtp/src/fouras/errors.h>

#include <mail/nwsmtp/ut/base_http_test.h>

#include <yplatform/coroutine.h>

#include <gtest/gtest.h>

using namespace NTesting;
using namespace NFouras;

class TTestFourasClient : public TBaseHttpTest<NTesting::THttpClusterClientMock> {
public:
    std::pair<boost::system::error_code, TKeyEntry> Run(std::string domain) {
        boost::system::error_code ec;
        TKeyEntry key;

        auto client = std::make_shared<TClient>(
            HttpMock, domain, boost::make_shared<NNwSmtp::TContext>(Ctx->uniq_id(), std::string{}, std::string{}, std::string{}),
            [&errorCode = ec, &resKey = key](auto ec, auto key) {
                errorCode = std::move(ec);
                resKey = std::move(key);
            });
        yplatform::spawn(client);

        return {ec, key};
    }
};

TEST_F(TTestFourasClient, ExistDomain) {
    std::string domain = "yandex.ru";
    auto httpResp = HTTP_200;
    httpResp.Resp.body = R"({
        "response": {
            "public_key" : "\"v=DKIM1; k=rsa; t=s; p=...",
            "private_key": "super_secret_key",
            "domain": "yandex.ru",
            "selector": "mail_selector",
            "enabled": true
        },
        "status": "ok"
    })";
    InitHttpMock(httpResp);

    auto [ec, key] = Run(domain);

    ASSERT_EQ(HttpMock->CallCount, 1ul);
    ASSERT_TRUE(!ec);
    ASSERT_EQ(key.Domain, domain);
    ASSERT_EQ(key.Selector, "mail_selector");
    ASSERT_EQ(key.SecretKey, "super_secret_key");
}

TEST_F(TTestFourasClient, CyrillicDomain) {
    std::string domain = "xn--d1acpjx3f.xn--p1ai";
    auto httpResp = HTTP_200;
    httpResp.Resp.body = R"({
        "response": {
            "public_key" : "\"v=DKIM1; k=rsa; t=s; p=...",
            "private_key": "super_secret_key",
            "domain": "яндекс.рф",
            "selector": "mail_selector",
            "enabled": true
        },
        "status": "ok"
    })";
    InitHttpMock(httpResp);

    auto [ec, key] = Run(domain);

    ASSERT_EQ(HttpMock->CallCount, 1ul);
    ASSERT_TRUE(!ec);
    ASSERT_EQ(key.Domain, domain);
    ASSERT_EQ(key.Selector, "mail_selector");
    ASSERT_EQ(key.SecretKey, "super_secret_key");
}

TEST_F(TTestFourasClient, NonExistDomain) {
    std::string domain = "yandex.ru";
    auto httpResp = HTTP_404;
    httpResp.Resp.body = R"({
        "response": {
            "code": "not_found",
            "message": "Object not found",
            "params": {}
        },
        "status": "error"
    })";
    InitHttpMock(httpResp);

    auto [ec, key] = Run(domain);

    ASSERT_EQ(HttpMock->CallCount, 1ul);
    ASSERT_TRUE(ec == EC_DOMAIN_NOT_FOUND);
}

TEST_F(TTestFourasClient, DisabledDomain) {
    std::string domain = "yandex.ru";
    auto httpResp = HTTP_200;
    httpResp.Resp.body = R"({
        "response": {
            "public_key" : "\"v=DKIM1; k=rsa; t=s; p=...",
            "private_key": "super_secret_key",
            "domain": "yandex.ru",
            "selector": "mail_selector",
            "enabled": false
        },
        "status": "ok"
    })";
    InitHttpMock(httpResp);

    auto [ec, key] = Run(domain);

    ASSERT_EQ(HttpMock->CallCount, 1ul);
    ASSERT_TRUE(ec == EC_DISABLED);
}

TEST_F(TTestFourasClient, ParseError200) {
    std::string domain = "yandex.ru";
    auto httpResp = HTTP_200;
    httpResp.Resp.body = R"({
        "response": {
            "code": "unknown",
            "message": "Unknown",
            "params": {}
        },
        "status": "unknown"
    })";
    InitHttpMock(httpResp);

    auto [ec, key] = Run(domain);

    ASSERT_EQ(HttpMock->CallCount, 1ul);
    ASSERT_TRUE(ec == EC_PARSE_ERROR);
}

TEST_F(TTestFourasClient, ParseError404) {
    std::string domain = "yandex.ru";
    auto httpResp = HTTP_404;
    httpResp.Resp.body = "Not found";
    InitHttpMock(httpResp);

    auto [ec, key] = Run(domain);

    ASSERT_EQ(HttpMock->CallCount, 1ul);
    ASSERT_TRUE(ec == EC_PARSE_ERROR);
}

TEST_F(TTestFourasClient, EmptyDomain) {
    std::string domain = "yandex.ru";
    auto httpResp = HTTP_200;
    httpResp.Resp.body = R"({
        "response": {
            "public_key" : "\"v=DKIM1; k=rsa; t=s; p=...",
            "private_key": "super_secret_key",
            "domain": "",
            "selector": "mail_selector",
            "enabled": true
        },
        "status": "ok"
    })";
    InitHttpMock(httpResp);

    auto [ec, key] = Run(domain);

    ASSERT_EQ(HttpMock->CallCount, 1ul);
    ASSERT_TRUE(ec == EC_PARSE_ERROR);
}

TEST_F(TTestFourasClient, EmptyKey) {
    std::string domain = "yandex.ru";
    auto httpResp = HTTP_200;
    httpResp.Resp.body = R"({
        "response": {
            "public_key" : "\"v=DKIM1; k=rsa; t=s; p=...",
            "private_key": "",
            "domain": "yandex.ru",
            "selector": "mail_selector",
            "enabled": true
        },
        "status": "ok"
    })";
    InitHttpMock(httpResp);

    auto [ec, key] = Run(domain);

    ASSERT_EQ(HttpMock->CallCount, 1ul);
    ASSERT_TRUE(ec == EC_PARSE_ERROR);
}

TEST_F(TTestFourasClient, EmptySelector) {
    std::string domain = "yandex.ru";
    auto httpResp = HTTP_200;
    httpResp.Resp.body = R"({
        "response": {
            "public_key" : "\"v=DKIM1; k=rsa; t=s; p=...",
            "private_key": "super_secret_key",
            "domain": "yandex.ru",
            "selector": "",
            "enabled": true
        },
        "status": "ok"
    })";
    InitHttpMock(httpResp);

    auto [ec, key] = Run(domain);

    ASSERT_EQ(HttpMock->CallCount, 1ul);
    ASSERT_TRUE(ec == EC_PARSE_ERROR);
}

TEST_F(TTestFourasClient, HttpNotFound) {
    std::string domain = "yandex.ru";
    auto httpResp = HTTP_404;
    httpResp.Resp.body = R"({
        "response": {
            "code": "not_found",
            "message": "Not Found",
            "params": {}
        },
        "status": "error"
    })";
    InitHttpMock(httpResp);

    auto [ec, key] = Run(domain);

    ASSERT_EQ(HttpMock->CallCount, 1ul);
    ASSERT_TRUE(ec == EC_NOT_FOUND);
}

TEST_F(TTestFourasClient, HttpError) {
    std::string domain = "yandex.ru";
    auto httpResp = HTTP_401;
    httpResp.Resp.body = "Need TVM ticket";
    InitHttpMock(httpResp);

    auto [ec, key] = Run(domain);

    ASSERT_EQ(HttpMock->CallCount, 1ul);
    ASSERT_TRUE(ec == EC_UNAUTHORIZED);
}
