#include <mail/nwsmtp/src/header_parser.h>
#include <mail/nwsmtp/src/header_storage.h>
#include <mail/nwsmtp/src/types.h>
#include <mail/nwsmtp/src/utils.h>

#include <gtest/gtest.h>

#include <cctype>
#include <sstream>
#include <vector>

using namespace NNwSmtp;

template <typename TSet>
bool CompareSets(const TSet& lhs, const TSet& rhs) {
    if (lhs.size() != rhs.size()) {
        return false;
    }

    return std::is_permutation(lhs.begin(), lhs.end(), rhs.begin());
}

template <typename TRange1, typename TRange2 = TRange1>
bool CompareRanges(const TRange1& lhs, const TRange2& rhs) {
    if (lhs.size() != rhs.size()) {
        return false;
    }

    return std::equal(lhs.begin(), lhs.end(), rhs.begin());
}

class TTestHeaderStorage : public testing::Test {
protected:
    void ParseHeaders(const std::string& headers) {
        std::stringstream ss(headers);
        std::string line;

        while (std::getline(ss >> std::ws, line)) {
            line.erase(std::find_if(line.rbegin(), line.rend(), [](int c) {
                return !std::isspace(c);
            }).base(), line.end());

            NUtil::AppendToSegment(Message, line + "\r\n");
        }

        NUtil::AppendToSegment(Message, "\r\nBody");

        TBufferRange r{Message.cbegin(), Message.cend()};
        parse_header(r, [this](const auto& name, const auto&, const auto& value) {
            Storage.Add(name, value);
        });
    }

protected:
    NNwSmtp::TBuffer Message;
    THeaderStorageImpl Storage;
};

TEST_F(TTestHeaderStorage, Count) {
    const std::string headerStr = R"(
        Field1: field1 value
        Field2: test_data
        FIELD1: field1 second value
        Field1: field1 third value
        Field3: unknown
        field2: test_data2
    )";

    ParseHeaders(headerStr);

    ASSERT_EQ(Storage.CountAll(), 6ul);
    ASSERT_EQ(Storage.Count("field1"), 3ul);
    ASSERT_EQ(Storage.Count("field2"), 2ul);
    ASSERT_EQ(Storage.Count("FIELD3"), 1ul);
    ASSERT_EQ(Storage.Count("unknwn"), 0ul);
}

TEST_F(TTestHeaderStorage, GetAllHeaders) {
    const std::string headerStr = R"(
        Field1: field1 value
        Field2: test_data
        FIELD1: field1 second value
        Field1: field1 third value
        Field3: unknown
        field2: test_data2
    )";

    ParseHeaders(headerStr);

    ASSERT_TRUE(CompareSets(Storage.GetUniqueHeaderNames(), {"field1", "field2", "field3"}));

    std::vector<std::string> names;
    std::vector<std::string> values;
    std::vector<unsigned int> orderNums;

    auto range = Storage.GetAllHeaders();

    for (const auto& header : range) {
        names.emplace_back(header.Name.begin(), header.Name.end());
        values.emplace_back(header.Value.begin(), header.Value.end());
        orderNums.push_back(header.OrderNum);
    }

    ASSERT_TRUE(CompareRanges(
        names,
        {"Field1", "Field2", "FIELD1", "Field1", "Field3", "field2"}
    ));

    ASSERT_TRUE(CompareRanges(
        values,
        {"field1 value", "test_data", "field1 second value", "field1 third value", "unknown", "test_data2"}
    ));

    ASSERT_TRUE(CompareRanges(orderNums, {0, 1, 2, 3, 4, 5}));
}

TEST_F(TTestHeaderStorage, GetHeaders) {
    const std::string headerStr = R"(
        Field1: field1 value
        Field2: test_data
        FIELD1: field1 second value
        Field1: field1 third value
        Field3: unknown
        field2: test_data2
    )";

    ParseHeaders(headerStr);

    std::vector<std::string> names;
    std::vector<std::string> values;
    std::vector<unsigned int> orderNums;

    auto range = Storage.GetHeaders("FiElD1");

    for (const auto& header : range) {
        names.emplace_back(header.Name.begin(), header.Name.end());
        values.emplace_back(header.Value.begin(), header.Value.end());
        orderNums.push_back(header.OrderNum);
    }

    ASSERT_TRUE(CompareRanges(names, {"Field1", "FIELD1", "Field1"}));
    ASSERT_TRUE(CompareRanges(values, {"field1 value", "field1 second value", "field1 third value"}));
    ASSERT_TRUE(CompareRanges(orderNums, {0, 2, 3}));

    auto errorRange = Storage.GetHeaders("unknown");
    ASSERT_TRUE(errorRange.begin() == errorRange.end());
}

TEST_F(TTestHeaderStorage, SizeForEmptyHeaders) {
    ASSERT_EQ(Storage.Size(), 0ul);
}

TEST_F(TTestHeaderStorage, SizeForSingleHeader) {
    ParseHeaders("Field1: field1 value\r\n");
    ASSERT_EQ(Storage.Size(), 1ul);
}

TEST_F(TTestHeaderStorage, SizeForTwoHeaders) {
    ParseHeaders("Field1: field1\r\nField2: field2\r\n");
    ASSERT_EQ(Storage.Size(), 2ul);
}

TEST_F(TTestHeaderStorage, SizeForTwoSameHeaders) {
    ParseHeaders("Field1: field1\r\nField1: field2\r\n");
    ASSERT_EQ(Storage.Size(), 2ul);
}
