#include <nwsmtp/hosts_resolver.h>

#include <gtest/gtest.h>

#include <boost/asio/ip/address.hpp>

#include <algorithm>
#include <string>
#include <vector>

class TTestHostsResolver : public testing::Test {
public:
    TTestHostsResolver()
        : Resolver({})
    {}

    static auto MakeAddress(const std::string& address) {
        return boost::asio::ip::make_address(address);
    }

protected:
    THostsResolver Resolver;
    const std::string ipv4HostAddress = "127.0.0.1";
    const std::string ipv6Host = "rcsrv-resolve-test6.mail.yandex.net";
    const std::string ipv6HostAddress = "::1";
    const std::string unknownHost = "rcsrv-resolve-test-unknown.mail.yandex.net";
};

TEST_F(TTestHostsResolver, SingleIpv6Host) {
    std::vector<std::string> hosts{ipv6Host};
    auto res = Resolver.ResolveHosts(hosts);

    ASSERT_EQ(res.size(), 1ul);
    ASSERT_TRUE(res.front() == MakeAddress(ipv6HostAddress));
}

TEST_F(TTestHostsResolver, SingleIpv6) {
    std::vector<std::string> hosts{ipv6HostAddress};
    auto res = Resolver.ResolveHosts(hosts);

    ASSERT_EQ(res.size(), 1ul);
    ASSERT_TRUE(res.front() == MakeAddress(ipv6HostAddress));
}

TEST_F(TTestHostsResolver, SingleIpv4) {
    std::vector<std::string> hosts{ipv4HostAddress};
    auto res = Resolver.ResolveHosts(hosts);

    ASSERT_EQ(res.size(), 1ul);
    ASSERT_TRUE(res.front() == MakeAddress(ipv4HostAddress));
}

TEST_F(TTestHostsResolver, SingleUnknownHost) {
    std::vector<std::string> hosts{unknownHost};
    auto res = Resolver.ResolveHosts(hosts);

    ASSERT_EQ(res.size(), 0ul);
}

TEST_F(TTestHostsResolver, Mixed) {
    std::vector<std::string> hosts{unknownHost, ipv4HostAddress, ipv6Host};
    std::vector<boost::asio::ip::address> expectedAddresses{
        MakeAddress(ipv4HostAddress),
        MakeAddress(ipv6HostAddress)};

    auto res = Resolver.ResolveHosts(hosts);
    ASSERT_EQ(res.size(), 2ul);
    ASSERT_TRUE(std::is_permutation(res.begin(), res.end(), expectedAddresses.begin()));
}
