#include <mail/nwsmtp/src/ml/client_impl.h>

#include <gtest/gtest.h>

using namespace NNwSmtp::NML;


TEST(ListTest, NormalizeString) {
    ASSERT_EQ(normalize("Hello"), "hello");
    ASSERT_EQ(normalize("Hello "), "hello");
    ASSERT_EQ(normalize("  Hello "), "hello");
    ASSERT_EQ(normalize("  Hello"), "hello");
    ASSERT_EQ(normalize(""), "");
    ASSERT_EQ(normalize(" "), "");
}

TEST(ParseMLData, Empty) {
    auto resp = parse_ml_data("{}");
    ASSERT_FALSE(resp.Internal);
    ASSERT_FALSE(resp.Open);
    ASSERT_FALSE(resp.Readonly);
    ASSERT_TRUE(resp.Subscribers.empty());
    ASSERT_TRUE(resp.WhoCanWrite.empty());
}

TEST(ParseMLData, NonDefaultFlags) {
    auto resp = parse_ml_data(R"({
        "is_internal": true,
        "is_open": true,
        "readonly": true,
        "emails" : [],
        "login": "foo",
        "status": "ok",
        "type": "user"
    })");
    ASSERT_TRUE(resp.Internal);  // only for yateam
    ASSERT_TRUE(resp.Open);  // anybody could write to ml
    ASSERT_TRUE(resp.Readonly);  // nobody could write to ml
    ASSERT_TRUE(resp.Subscribers.empty());
    ASSERT_TRUE(resp.WhoCanWrite.empty());
}

TEST(ParseMLData, SingleEmail) {
    auto resp = parse_ml_data(R"({
        "emails": [
            "foo@yandex-team.ru"
        ],
        "login": "foo",
        "status": "ok",
        "type": "user"
    })");
    ASSERT_FALSE(resp.Subscribers.empty());
    ASSERT_EQ(resp.Subscribers[0], "foo@yandex-team.ru");
}

TEST(ParseMLData, TwoEmails) {
    auto resp = parse_ml_data(R"({
        "emails": [
            "foo@mail.yandex-team.ru",
            "foo@ld.yandex-team.ru"
        ],
        "login": "foo",
        "status": "ok",
        "type": "user"
    })");
    ASSERT_EQ(2ul, resp.Subscribers.size());
    ASSERT_EQ(resp.Subscribers[0], "foo@mail.yandex-team.ru");
    ASSERT_EQ(resp.Subscribers[1], "foo@ld.yandex-team.ru");
}

TEST(ParseMLData, SubscribersInbox) {
    auto resp = parse_ml_data(R"({
        "subscribers": {
            "inbox": [
                "foo@mail.yandex-team.ru",
                "bar@mail.yandex-team.ru"
            ]
        },
        "emails": [
        ],
        "login": "foo",
        "status": "ok",
        "type": "maillist"
    })");
    ASSERT_FALSE(resp.Internal);
    ASSERT_FALSE(resp.Open);
    ASSERT_FALSE(resp.Readonly);

    ASSERT_EQ(resp.Subscribers.size(), 2ul);
    ASSERT_EQ(resp.Subscribers[0], "foo@mail.yandex-team.ru");
    ASSERT_EQ(resp.Subscribers[1], "bar@mail.yandex-team.ru");
}

TEST(ParseMLData, WhoCanWrite) {
    auto resp = parse_ml_data(R"({
        "emails": [
            "foo@mail.yandex-team.ru",
            "bar@mail.yandex-team.ru"
        ],
        "who_can_write": [
            "bar@mail.yandex-team.ru"
        ],
        "login": "foo",
        "status": "ok",
        "type": "user"
    })");
    ASSERT_EQ(resp.Subscribers.size(), 2ul);
    ASSERT_EQ(resp.Subscribers[0], "foo@mail.yandex-team.ru");
    ASSERT_EQ(resp.Subscribers[1], "bar@mail.yandex-team.ru");

    ASSERT_FALSE(resp.WhoCanWrite.empty());
    ASSERT_EQ(*resp.WhoCanWrite.begin(), "bar@mail.yandex-team.ru");
}
