#include <mail/nwsmtp/src/ml/client_impl.h>
#include <mail/nwsmtp/src/options.h>

#include "base_http_test.h"

#include <library/cpp/resource/resource.h>

#include <boost/asio.hpp>

#include <gtest/gtest.h>

using namespace NNwSmtp::NML;

TEST(MakeRequest, SimpleRequest) {
    auto req = makeRequest(
        "http://ml.y-t.ru/delivery/info?email=",
        "foo@example.com");
    ASSERT_TRUE(req.method == yhttp::request::method_t::GET);
    ASSERT_EQ(req.url, "http://ml.y-t.ru/delivery/info?email=foo@example.com");
}

TEST(MakeOptions, Timeouts) {
    using namespace std::chrono_literals;

    NNwSmtp::Options::LLOpts opts;
    opts.timeout = 12s;
    opts.connectTimeout = 11s;

    auto httpOpts = makeOptions(opts);
    ASSERT_TRUE(httpOpts.timeouts.total == 12s);
    ASSERT_TRUE(httpOpts.timeouts.connect == 11s);
}

struct TListClient : public NTesting::TBaseHttpTest<NTesting::THttpClientMock> {

    std::pair<boost::system::error_code, TResponse> Run(
        NNwSmtp::Options::LLOpts opts,
        TRequest request
    ){
        TClientPtr client = std::make_shared<TClient>(opts, HttpMock, Io);

        boost::system::error_code resErrc;
        TResponse resResp;

        client->Run(
            boost::make_shared<NNwSmtp::TContext>(Ctx->uniq_id(), std::string{}, std::string{}, std::string{}), request,
            [this, &resErrc, &resResp](auto ec, auto resp) {
                ++CallCount;
                resErrc = ec;
                resResp = resp;
        });
        Io.run();
        return {resErrc, resResp};
    }

    size_t CallCount = 0;
    boost::asio::io_context Io;
};

TEST_F(TListClient, DeliveryInfoResponseOk) {
    auto httpResp =  HTTP_200;
    httpResp.Resp.body = NResource::Find("delivery_info.json");
    InitHttpMock(httpResp);

    auto [errc, resp] = Run({}, {});

    ASSERT_EQ(CallCount, 1ul);
    ASSERT_FALSE(errc);
    ASSERT_EQ(resp.Subscribers.size(), 1ul);
    ASSERT_EQ(resp.Subscribers[0], "foo@example.com");
}

TEST_F(TListClient, RetryOnHttp500) {
    InitHttpMock(HTTP_500);

    NNwSmtp::Options::LLOpts opts;
    opts.attempts = 5;

    auto [errc, resp] = Run(opts, {});

    ASSERT_EQ(CallCount, 1ul);
    ASSERT_TRUE(errc == list_client_ec::INVALID_CODE);
    ASSERT_TRUE(resp.Subscribers.empty());
    ASSERT_EQ(HttpMock->CallCount, opts.attempts);
}

TEST_F(TListClient, SuccessAfterRetryOnHttp500) {
    THttpResponse httpOk = HTTP_200;
    httpOk.Resp.body = NResource::Find("delivery_info.json");
    InitHttpMock({HTTP_500, httpOk});

    NNwSmtp::Options::LLOpts opts;
    opts.attempts = 2;

    auto [errc, resp] = Run(opts, {});

    ASSERT_EQ(CallCount, 1ul);
    ASSERT_FALSE(errc);
    ASSERT_EQ(resp.Subscribers.size(), 1ul);
    ASSERT_EQ(resp.Subscribers[0], "foo@example.com");
    ASSERT_EQ(HttpMock->CallCount, opts.attempts);
}
