#include "spf_mock.h"
#include "async_check_sender_mock.h"

#include <mail/nwsmtp/src/context.h>
#include <mail/nwsmtp/src/settings_authorization/settings_authorization.h>
#include <mail/nwsmtp/src/mailfrom/command_impl.h>

#include <mail/nwsmtp/ut/util/check_bb_mocks.h>
#include <mail/nwsmtp/ut/mailfrom/settings_authorization_mock.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <memory>

namespace {

using namespace ::testing;
using namespace NNwSmtp;

using TBBChecksMock = NBlackBox::NTest::TBBChecksMock;
using TSettingsAuthorizationMock = NTest::TSettingsAuthorizationMock;
using TSpfCheckMock = NTest::TSpfCheckMock;
using TAsyncCheckSenderClientMock = NTest::TAsyncCheckSenderClientMock;

struct TTestMailFrom : Test {
    TTestMailFrom()
        : Ctx(boost::make_shared<TContext>("conn", "env", "cluster", "host"))
    {
        NNwSmtp::glog = std::make_shared<TLog>();
    }

    void RunMailFrom(NMailFromCommand::TConfig config, NMailFromCommand::TRequest req, NMailFromCommand::TCallback cb) {
        auto cmd = NMailFromCommand::TMailFromCommand(config, SpfMock, BBChecksMock, SettingsAuthorizationMock, CheckSenderMock, IoContext);
        cmd.Run(Ctx, std::move(req), std::move(cb));
        IoContext.run();
    }

    std::shared_ptr<StrictMock<TBBChecksMock>> BBChecksMock = std::make_shared<StrictMock<TBBChecksMock>>();
    std::shared_ptr<StrictMock<TSettingsAuthorizationMock>> SettingsAuthorizationMock = std::make_shared<StrictMock<TSettingsAuthorizationMock>>();
    std::shared_ptr<StrictMock<TAsyncCheckSenderClientMock>> CheckSenderMock = std::make_shared<StrictMock<TAsyncCheckSenderClientMock>>();
    std::shared_ptr<StrictMock<TSpfCheckMock>> SpfMock = std::make_shared<StrictMock<TSpfCheckMock>>();

    boost::asio::io_context IoContext;
    TContextPtr Ctx;
};

TEST_F(TTestMailFrom, for_authenticated_and_need_check_auth_when_user_is_banned_should_return_error) {
    NMailFromCommand::TConfig cfg {};
    NMailFromCommand::TRequest req {
        .BBSessionSuid = 123,
        .IsAuthenticated = true,
        .NeedCheckAuth = true,
    };

    EXPECT_CALL(*SettingsAuthorizationMock, Run(_, _, _)).WillOnce(
        InvokeArgument<2>(NBlackBox::EError::UserBlocked, NBlackBox::TResponse{})
    );

    RunMailFrom(std::move(cfg), std::move(req), [](TErrorCode ec, NMailFromCommand::TResponse) {
        EXPECT_EQ(ec, NMailFromCommand::EError::AuthReject);
    });
}

TEST_F(TTestMailFrom, for_authenticated_and_mailfrom_addr_doesnt_belong_to_user_should_return_error) {
    NMailFromCommand::TConfig cfg {
        .BlackBoxCheckSender = true
    };
    NMailFromCommand::TRequest req {
        .BBSessionSuid = 123,
        .IsAuthenticated = true,
    };

    EXPECT_CALL(*BBChecksMock, CheckMailFrom(_, _, _, _)).WillOnce(
        InvokeArgument<3>(NBlackBox::EError::Ok, NBlackBox::TResponse{.Suid = 234})
    );

    RunMailFrom(std::move(cfg), std::move(req), [](TErrorCode ec, NMailFromCommand::TResponse) {
        EXPECT_EQ(ec, NMailFromCommand::EError::RejectAddrNotOwnedByAuthUser);
    });
}

TEST_F(TTestMailFrom, for_authenticated_and_need_check_auth_and_everything_is_fine_should_return_ok) {
    NMailFromCommand::TConfig cfg {
        .BlackBoxCheckSender = true
    };
    NMailFromCommand::TRequest req {
        .MailAddr = "test@yandex.ru",
        .BBSessionSuid = 123,
        .IsAuthenticated = true,
        .NeedCheckAuth = true,
    };
    NBlackBox::TResponse blackBoxResp {
        .Login = "test@yandex.ru",
        .Suid = 123,
        .Uid = "234",
    };

    EXPECT_CALL(*SettingsAuthorizationMock, Run(_, _, _)).WillOnce(
        InvokeArgument<2>(NBlackBox::EError::Ok, blackBoxResp)
    );

    EXPECT_CALL(*BBChecksMock, CheckMailFrom(_, _, _, _)).WillOnce(
        InvokeArgument<3>(NBlackBox::EError::Ok, blackBoxResp)
    );

    EXPECT_CALL(*CheckSenderMock, Run(_, _, _)).WillOnce(
        InvokeArgument<2>(NRateSrv::EError::Accept)
    );

    RunMailFrom(std::move(cfg), std::move(req), [=](TErrorCode ec, NMailFromCommand::TResponse resp) {
        ASSERT_FALSE(ec);

        ASSERT_TRUE(resp.CheckAuthResponse);
        ASSERT_EQ(resp.CheckAuthResponse->BBSessionLogin, blackBoxResp.Login);
        ASSERT_EQ(resp.CheckAuthResponse->BBSessionSuid, blackBoxResp.Suid);

        ASSERT_TRUE(resp.CheckBBResponse);
        ASSERT_EQ(resp.CheckBBResponse->BBResult.Login, blackBoxResp.Login);
        ASSERT_EQ(resp.CheckBBResponse->BBResult.Suid, blackBoxResp.Suid);
        ASSERT_EQ(resp.CheckBBResponse->BBResult.Uid, blackBoxResp.Uid);

        ASSERT_TRUE(resp.SmtpAnswer.starts_with("250"));
    });
}

TEST_F(TTestMailFrom, for_authenticated_and_need_check_auth_and_everything_is_fine_except_ratesrv_should_return_rejected_by_ratesrv) {
    NMailFromCommand::TConfig cfg {
        .BlackBoxCheckSender = true
    };
    NMailFromCommand::TRequest req {
        .MailAddr = "test@yandex.ru",
        .BBSessionSuid = 123,
        .IsAuthenticated = true,
        .NeedCheckAuth = true,
    };
    NBlackBox::TResponse blackBoxResp {
        .Login = "test@yandex.ru",
        .Suid = 123,
        .Uid = "234",
    };

    EXPECT_CALL(*SettingsAuthorizationMock, Run(_, _, _)).WillOnce(
        InvokeArgument<2>(NBlackBox::EError::Ok, blackBoxResp)
    );

    EXPECT_CALL(*BBChecksMock, CheckMailFrom(_, _, _, _)).WillOnce(
        InvokeArgument<3>(NBlackBox::EError::Ok, blackBoxResp)
    );

    EXPECT_CALL(*CheckSenderMock, Run(_, _, _)).WillOnce(
        InvokeArgument<2>(NRateSrv::EError::Reject)
    );

    RunMailFrom(std::move(cfg), std::move(req), [=](TErrorCode ec, NMailFromCommand::TResponse) {
        ASSERT_EQ(ec, NMailFromCommand::EError::RejectRateSrv);
    });
}

TEST_F(TTestMailFrom, for_need_check_spf_when_spf_ok_should_return_ok) {
    NMailFromCommand::TConfig cfg {
        .BlackBoxCheckSender = false,
        .CheckSPF = true
    };
    NMailFromCommand::TRequest req {
        .MailAddr = "test@yandex.ru",
        .IsAuthenticated = false,
        .NeedCheckAuth = false,
    };

    EXPECT_CALL(*CheckSenderMock, Run(_, _, _)).WillOnce(
        InvokeArgument<2>(NRateSrv::EError::Accept)
    );
    EXPECT_CALL(*SpfMock, Check(_, _, _, _, _)).WillOnce(InvokeArgument<4>(NMailFromCommand::EError::Ok,
        NSPF::TResponse {
            .Result = "good",
            .Expl = "passed",
        }
    ));

    RunMailFrom(std::move(cfg), std::move(req), [=](TErrorCode ec, NMailFromCommand::TResponse resp) {
        ASSERT_FALSE(ec);

        ASSERT_TRUE(resp.SPFResponse);
        ASSERT_EQ(resp.SPFResponse->Explain, "passed");
        ASSERT_EQ(resp.SPFResponse->Result, "good");
    });
}

}
