#include <mail/nwsmtp/src/nsls/client_impl.h>
#include <mail/nwsmtp/src/nsls/error_code.h>
#include <mail/nwsmtp/src/nsls/types.h>
#include <mail/nwsmtp/src/nsls/types_reflection.h>

#include <mail/nwsmtp/ut/util/http_client_mock.h>
#include <mail/nwsmtp/ut/test_with_spawn.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <boost/asio.hpp>

#include <optional>
#include <memory>

namespace {

using namespace testing;
using namespace NNwSmtp;
using namespace NNwSmtp::NNsls;
using namespace NNwSmtp::NTests;

using Request = yhttp::request;
using Response = yhttp::response;

TRequest BuildStoreRequest() {
    TRequest request {};
    request.envelope.session_id = "session_id";
    request.envelope.envelope_id = "envelope_id";
    request.envelope.mail_from.email = "pepa@gig.com";

    request.envelope.remote_ip = "::1";
    request.envelope.remote_host = "localhost";
    request.envelope.helo = "pig_host";

    request.message.spam = false;
    request.message.stid = "stid";
    request.message.timemark = 0;

    request.recipients.push_back(TRecipient {
        "george@pig.com",
        "1",
        ELocal::yes,
        TNotification {false, false, false},
        true,
        std::nullopt,
        std::nullopt
    });

    request.requestId = "request_id";

    THint hint = {{"first", {"1", "2"}}, {"second", {"3"}}};
    request.message.hints.emplace_back(std::move(hint));

    return request;
}

struct TTestNslsClient: TTestWithSpawn {
    const std::shared_ptr<StrictMock<TClusterClientMock>> ClusterClient = GetStrictMockedClusterClient();
    const std::shared_ptr<TNslsClient> NslsClient = std::make_shared<TNslsClient>(ClusterClient, Io);

    const TRequest request = BuildStoreRequest();

    TContextPtr Context = boost::make_shared<TContext>("","","","");

    const Request StoreRequest = Request::POST(
        "/store?request_id=request_id",
        R"({"envelope":{"remote_ip":"::1","remote_host":"localhost","mail_from":{"email":"pepa@gig.com","envid":""})"
        R"(,"session_id":"session_id","envelope_id":"envelope_id","helo":"pig_host"},)"
        R"("message":{"stid":"stid","front":"","timemark":0,"spam":false,"hints":[{"first":["1","2"],"second":["3"]}]},)"
        R"("recipients":[{"email":"george@pig.com","uid":"1","is_local":"yes",)"
        R"("notify":{"success":false,"failure":false,"delay":false},"is_mailish":true}]})"
    );

    template <typename THandler>
    TResponse Store(
        THandler handler
    ) {
        boost::asio::async_completion<
            THandler,
            void(TErrorCode, TResponse)
        > init(handler);
        NslsClient->Store(Context, request, init.completion_handler);
        return init.result.get();
    }
};

TEST_F(TTestNslsClient, for_request_ended_with_error_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        EXPECT_CALL(*ClusterClient, async_run(_, StoreRequest, _, _))
            .WillOnce(InvokeArgument<3>(ymod_httpclient::http_error::code::ssl_error, Response {}));

        TErrorCode ec;
        auto result = Store(yield[ec]);
        ASSERT_TRUE(ec);
        EXPECT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
    });
}

TEST_F(TTestNslsClient, for_request_ended_with_406_code_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        EXPECT_CALL(*ClusterClient, async_run(_, StoreRequest, _, _))
            .WillOnce(InvokeArgument<3>(TErrorCode {}, Response {406, {}, {}, {}}));

        TErrorCode ec;
        auto result = Store(yield[ec]);
        ASSERT_TRUE(ec);
        EXPECT_EQ(ec, EError::EC_PERM_ERROR);
    });
}

TEST_F(TTestNslsClient, for_request_ended_with_500_code_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        EXPECT_CALL(*ClusterClient, async_run(_, StoreRequest, _, _))
            .WillOnce(InvokeArgument<3>(TErrorCode {}, Response {500, {}, {}, {}}));

        TErrorCode ec;
        auto result = Store(yield[ec]);
        ASSERT_TRUE(ec);
        EXPECT_EQ(ec, EError::EC_BAD_STATUS);
    });
}

TEST_F(TTestNslsClient, for_failed_parsing_response_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        EXPECT_CALL(*ClusterClient, async_run(_, StoreRequest, _, _))
            .WillOnce(InvokeArgument<3>(TErrorCode {}, Response {200, {}, "", {}}));

        TErrorCode ec;
        auto result = Store(yield[ec]);
        ASSERT_TRUE(ec);
        EXPECT_EQ(ec, EError::EC_PARSE_ERROR);
    });
}

TEST_F(TTestNslsClient, for_response_should_return_stored_response) {
    WithSpawn([this](boost::asio::yield_context yield) {

        EXPECT_CALL(*ClusterClient, async_run(_, StoreRequest, _, _))
            .WillOnce(InvokeArgument<3>(TErrorCode {}, Response {200, {}, R"({"recipients": []})", {}}));

        TErrorCode ec;
        auto result = Store(yield[ec]);
        ASSERT_FALSE(ec);
        EXPECT_EQ(result.Status, "error");

    });
}

} // namespace anonymous
