#include "utils.h"

#include <mail/nwsmtp/src/header_storage.h>
#include <mail/nwsmtp/src/utils.h>

#include <gtest/gtest.h>

using NNwSmtp::ParseMessage;
using NNwSmtp::NUtil::MakeSegment;

TEST(ParseMessage, SingleHeaderAndBodyParsedOk) {
    auto msg = MakeSegment("Header: Value\r\n\r\nBody");
    auto [headers, body] = ParseMessage(msg);
    ASSERT_EQ(headers.Size(), 1ul);
    ASSERT_EQ(std::string(body.begin(), body.end()), "Body");
}

TEST(ParseMessage, TwoHeadersAndBodyWithDifferentDelimitersParsedOk) {
    for (const auto& delimiter : {"\n", "\r\n"}) {
        auto msg = MakeSegment(
            "Header1: Value1", delimiter,
            "Header2: Value2", delimiter,
            delimiter,
            "Body");
        auto [headers, body] = ParseMessage(msg);
        ASSERT_EQ(headers.Size(), 2ul);
        ASSERT_EQ(std::string(body.begin(), body.end()), "Body");
    }
}

TEST(ParseMessage, FoldingHeaderParsedOk) {
    auto msg = MakeSegment(
        "Header: Very\r\n\tLong Header\r\n",
        "\r\nBody");
    auto [headers, body] = ParseMessage(msg);
    ASSERT_EQ(headers.Size(), 1ul);
    ASSERT_EQ(std::string(body.begin(), body.end()), "Body");
}

TEST(ParseMessage, NoBodyParsedOk) {
    auto msg = MakeSegment("Header: Value\r\n\r\n");
    auto [headers, body] = ParseMessage(msg);
    ASSERT_EQ(headers.Size(), 1ul);
    ASSERT_FALSE(body);
}

TEST(ParseMessage, NoHeadersParsedFail) {
    auto msg = MakeSegment("\r\n\r\nBody");
    auto [headers, body] = ParseMessage(msg);
    ASSERT_TRUE(headers.Empty());
    ASSERT_FALSE(body);
}

TEST(ParseMessage, BodyWithSingleCrlfParsedFail) {
    auto msg = MakeSegment("\r\nBody");
    auto [headers, body] = ParseMessage(msg);
    ASSERT_TRUE(headers.Empty());
    ASSERT_FALSE(body);
}

TEST(ParseMessage, NoDelimitersParsedFail) {
    auto msg = MakeSegment("Body");
    auto [headers, body] = ParseMessage(msg);
    ASSERT_TRUE(headers.Empty());
    ASSERT_FALSE(body);
}

TEST(ParseMessage, SingleHeaderAndDifferentDelimitersParsedOk) {
    for (const auto& delimiter : {"\n", "\r\n"}) {
        auto msg = MakeSegment("Header: Value", delimiter);
        auto [headers, body] = ParseMessage(msg);
        ASSERT_EQ(headers.Size(), 1ul);
        ASSERT_FALSE(body);
    }
}

TEST(ParseMessage, SingleHeaderWithoutDelimitersParsedFail) {
    auto msg = MakeSegment("Header: Value");
    auto [headers, body] = ParseMessage(msg);
    ASSERT_TRUE(headers.Empty());
    ASSERT_FALSE(body);
}

TEST(ParseMessage, BrokenHeaderSkipped) {
    auto msg = MakeSegment("HeaderValue\r\n", "Foo: Value\r\n\r\nBody");
    auto [headers, body] = ParseMessage(msg);
    ASSERT_EQ(headers.Size(), 1ul);
    ASSERT_EQ(std::string(body.begin(), body.end()), "Body");
}

TEST(ParseMessage, SingleDelimiterBetweenHeaderAndBodyParsedFail) {
    for (const auto& delimiter : {"\n", "\r\n"}) {
        auto msg = MakeSegment("Header: Value", delimiter, "Body");
        auto [headers, body] = ParseMessage(msg);
        ASSERT_TRUE(headers.Empty());
        ASSERT_FALSE(body);
    }
}

TEST(ParseMessage, BodyContainsCrlfsParsedOk) {
    auto msg = MakeSegment(
        "Header: Value\r\n",
        "\r\nBody1",
        "\r\n\r\nBody2");
    auto [headers, body] = ParseMessage(msg);
    ASSERT_EQ(headers.Size(), 1ul);
    ASSERT_EQ(std::string(body.begin(), body.end()), "Body1\r\n\r\nBody2");
}

TEST(ParseMessage, JustDelimitersParsedFail) {
    auto msg = MakeSegment("\r\n\r\n");
    auto [headers, body] = ParseMessage(msg);
    ASSERT_TRUE(headers.Empty());
    ASSERT_FALSE(body);
}

TEST(ParseMessage, EmptyMessageParsedFail) {
    auto msg = MakeSegment("");
    auto [headers, body] = ParseMessage(msg);
    ASSERT_TRUE(headers.Empty());
    ASSERT_FALSE(body);
}
