#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <mail/nwsmtp/src/rcpt_to/utils.h>

namespace {

using namespace NNwSmtp;
using namespace NNwSmtp::NRcptTo;
using namespace testing;

TEST(SetRcptDsnNotifies, for_empty_should_set_none) {
    check_rcpt_t rcpt;
    SetRcptDsnNotifies(rcpt, "");
    EXPECT_EQ(rcpt.RcptContext.NotifyMode, dsn::Options::NONE);
}

TEST(SetRcptDsnNotifies, for_never_should_set_never) {
    check_rcpt_t rcpt;
    SetRcptDsnNotifies(rcpt, "Never");
    EXPECT_EQ(rcpt.RcptContext.NotifyMode, dsn::Options::NEVER);
}

TEST(SetRcptDsnNotifies, for_failure_should_set_failure) {
    check_rcpt_t rcpt;
    SetRcptDsnNotifies(rcpt, "Failure");
    EXPECT_TRUE(rcpt.RcptContext.NotifyMode & dsn::Options::FAILURE);
}

TEST(SetRcptDsnNotifies, for_success_should_set_success) {
    check_rcpt_t rcpt;
    SetRcptDsnNotifies(rcpt, "Success");
    EXPECT_TRUE(rcpt.RcptContext.NotifyMode & dsn::Options::SUCCESS);
}

TEST(SetRcptDsnNotifies, for_delay_should_set_delay) {
    check_rcpt_t rcpt;
    SetRcptDsnNotifies(rcpt, "Delay");
    EXPECT_TRUE(rcpt.RcptContext.NotifyMode & dsn::Options::DELAY);
}

TEST(SetRcptDsnNotifies, for_all_should_set_all) {
    check_rcpt_t rcpt;
    SetRcptDsnNotifies(rcpt, "Failure,Success,Delay");
    EXPECT_TRUE(rcpt.RcptContext.NotifyMode & (dsn::Options::FAILURE | dsn::Options::SUCCESS | dsn::Options::DELAY));
}

TEST(SetRcptDsnNotifies, for_never_and_all_should_set_never) {
    check_rcpt_t rcpt;
    SetRcptDsnNotifies(rcpt, "Failure,Success,Delay,NEVER");
    EXPECT_EQ(rcpt.RcptContext.NotifyMode, dsn::Options::NEVER);
}

TEST(NeedCalc, for_alias_should_return_false) {
    TConfig config;
    check_rcpt_t rcpt;
    rcpt.RcptContext.IsAlias = true;
    EXPECT_FALSE(NeedCallRouter(config, rcpt));
}

TEST(NeedCalc, for_primary_routing_none_should_return_false) {
    TConfig config;
    check_rcpt_t rcpt;
    config.Delivery.routing.primary = RoutingSettings::NONE;
    EXPECT_FALSE(NeedCallRouter(config, rcpt));
}

TEST(NeedCalc, for_primary_routing_is_not_none_and_not_alias_should_return_true) {
    TConfig config;
    check_rcpt_t rcpt;
    config.Delivery.routing.primary = RoutingSettings::DNS;
    EXPECT_TRUE(NeedCallRouter(config, rcpt));
}

TEST(NeedCheckBB, for_bb_check_and_not_alias_and_routing_dns_and_domain_local_should_return_true) {
    TConfig config;
    check_rcpt_t rcpt;
    rcpt.RcptContext.IsAlias = false;
    config.Blackbox.checkRcpt = true;
    rcpt.RcptContext.DomainType = DomainType::LOCAL;
    config.Delivery.routing.primary = RoutingSettings::Type::DNS;

    EXPECT_TRUE(NeedCheckBBForRcpt(config, rcpt));
}

TEST(NeedCheckBB, for_bb_check_and_alias_and_routing_dns_and_domain_local_should_return_false) {
    TConfig config;
    check_rcpt_t rcpt;
    rcpt.RcptContext.IsAlias = true;
    config.Blackbox.checkRcpt = true;
    rcpt.RcptContext.DomainType = DomainType::LOCAL;
    config.Delivery.routing.primary = RoutingSettings::Type::DNS;

    EXPECT_FALSE(NeedCheckBBForRcpt(config, rcpt));
}

TEST(NeedCheckBB, for_not_bb_check_and_not_alias_and_routing_dns_and_domain_local_should_return_false) {
    TConfig config;
    check_rcpt_t rcpt;
    rcpt.RcptContext.IsAlias = false;
    config.Blackbox.checkRcpt = false;
    rcpt.RcptContext.DomainType = DomainType::LOCAL;
    config.Delivery.routing.primary = RoutingSettings::Type::DNS;

    EXPECT_FALSE(NeedCheckBBForRcpt(config, rcpt));
}

TEST(NeedCheckBB, for_bb_check_and_not_alias_and_routing_none_and_domain_unknown_should_return_true) {
    TConfig config;
    check_rcpt_t rcpt;
    rcpt.RcptContext.IsAlias = false;
    config.Blackbox.checkRcpt = true;
    rcpt.RcptContext.DomainType = DomainType::UNKNOWN;
    config.Delivery.routing.primary = RoutingSettings::Type::NONE;

    EXPECT_TRUE(NeedCheckBBForRcpt(config, rcpt));
}

TEST(NeedCheckBB, for_bb_check_and_alias_and_routing_none_and_domain_unknown_should_return_false) {
    TConfig config;
    check_rcpt_t rcpt;
    rcpt.RcptContext.IsAlias = true;
    config.Blackbox.checkRcpt = true;
    rcpt.RcptContext.DomainType = DomainType::UNKNOWN;
    config.Delivery.routing.primary = RoutingSettings::Type::NONE;

    EXPECT_FALSE(NeedCheckBBForRcpt(config, rcpt));
}

TEST(NeedCheckBB, for_not_bb_check_and_alias_and_routing_none_and_domain_unknown_should_return_false) {
    TConfig config;
    check_rcpt_t rcpt;
    rcpt.RcptContext.IsAlias = false;
    config.Blackbox.checkRcpt = false;
    rcpt.RcptContext.DomainType = DomainType::UNKNOWN;
    config.Delivery.routing.primary = RoutingSettings::Type::NONE;

    EXPECT_FALSE(NeedCheckBBForRcpt(config, rcpt));
}

}
