#include <mail/nwsmtp/src/rcpts.h>

#include <gtest/gtest.h>

using NNwSmtp::TRcpts;

TEST(TRcpts, EmptyNoRcpts) {
    TRcpts rcpts;
    ASSERT_TRUE(rcpts.Empty());
}

TEST(TRcpts, EmptyOneRcpt) {
    TRcpts rcpts;
    rcpts.Add({"first@ya.ru", "1"});
    ASSERT_FALSE(rcpts.Empty());
}

TEST(TRcpts, SizeNoRcpts) {
    TRcpts rcpts;
    ASSERT_EQ(rcpts.GetSize(), 0ul);
}

TEST(TRcpts, SizeOneRcpt) {
    TRcpts rcpts;
    rcpts.Add({"first@ya.ru", "1"});
    ASSERT_EQ(rcpts.GetSize(), 1ul);
}

TEST(TRcpts, AddOne) {
    TRcpts rcpts;
    rcpts.Add({"first@ya.ru", "1"});
    ASSERT_FALSE(rcpts.Empty());
    ASSERT_EQ(rcpts.GetSize(), 1ul);
}

TEST(TRcpts, AddTwoNoDuplicates) {
    TRcpts rcpts;
    rcpts.Add({"first@ya.ru", "1"});
    rcpts.Add({"second@ya.ru", "2"});
    ASSERT_EQ(rcpts.GetSize(), 2ul);
}

TEST(TRcpts, AddTwoOneDuplicate) {
    TRcpts rcpts;
    rcpts.Add({"first@ya.ru", "1"});
    rcpts.Add({"fiRst@ya.ru", "1"});
    ASSERT_EQ(rcpts.GetSize(), 1ul);
}

TEST(TRcpts, GetByEmailOne) {
    TRcpts rcpts;
    rcpts.Add({"first@ya.ru", "1"});
    auto* rcpt = rcpts.GetByEmail("first@ya.ru");
    ASSERT_NE(rcpt, nullptr);
    ASSERT_EQ(rcpt->m_name, "first@ya.ru");
    ASSERT_EQ(rcpt->m_uid, "1");

    rcpt = rcpts.GetByEmail("second@ya.ru");
    ASSERT_EQ(rcpt, nullptr);
}

TEST(TRcpts, GetByEmailEmptyEmail) {
    TRcpts rcpts;
    rcpts.Add({"first@ya.ru", "1"});
    auto* rcpt = rcpts.GetByEmail("");
    ASSERT_EQ(rcpt, nullptr);
}

TEST(TRcpts, GetByEmailTwo) {
    TRcpts rcpts;
    rcpts.Add({"first@ya.ru", "1"});
    rcpts.Add({"second@ya.ru", "2"});

    auto* rcpt = rcpts.GetByEmail("first@ya.ru");
    ASSERT_NE(rcpt, nullptr);
    ASSERT_EQ(rcpt->m_name, "first@ya.ru");
    ASSERT_EQ(rcpt->m_uid, "1");

    rcpt = rcpts.GetByEmail("second@ya.ru");
    ASSERT_NE(rcpt, nullptr);
    ASSERT_EQ(rcpt->m_name, "second@ya.ru");
    ASSERT_EQ(rcpt->m_uid, "2");
}

TEST(TRcpts, GetByUidOne) {
    TRcpts rcpts;
    rcpts.Add({"first@ya.ru", "1"});
    auto* rcpt = rcpts.GetByUid("1");
    ASSERT_NE(rcpt, nullptr);
    ASSERT_EQ(rcpt->m_name, "first@ya.ru");
    ASSERT_EQ(rcpt->m_uid, "1");

    rcpt = rcpts.GetByUid("2");
    ASSERT_EQ(rcpt, nullptr);
}

TEST(TRcpts, GetByUidEmptyUid) {
    TRcpts rcpts;
    rcpts.Add({"first@ya.ru", "1"});
    auto* rcpt = rcpts.GetByUid("1");
    ASSERT_NE(rcpt, nullptr);
    ASSERT_EQ(rcpt->m_name, "first@ya.ru");
    ASSERT_EQ(rcpt->m_uid, "1");
}

TEST(TRcpts, GetByUidTwo) {
    TRcpts rcpts;
    rcpts.Add({"first@ya.ru", "1"});
    rcpts.Add({"second@ya.ru", "2"});

    auto* rcpt = rcpts.GetByUid("1");
    ASSERT_NE(rcpt, nullptr);
    ASSERT_EQ(rcpt->m_name, "first@ya.ru");
    ASSERT_EQ(rcpt->m_uid, "1");

    rcpt = rcpts.GetByUid("2");
    ASSERT_NE(rcpt, nullptr);
    ASSERT_EQ(rcpt->m_name, "second@ya.ru");
    ASSERT_EQ(rcpt->m_uid, "2");
}

TEST(TRcpts, GetByUidSameUidsDifferentEmails) {
    TRcpts rcpts;
    rcpts.Add({"first@yandex.ru", "1"});
    rcpts.Add({"first@ld.yandex.ru", "1"});

    auto* rcpt = rcpts.GetByUid("1");
    ASSERT_NE(rcpt, nullptr);
    ASSERT_EQ(rcpt->m_name, "first@yandex.ru");
    ASSERT_EQ(rcpt->m_uid, "1");
}

TEST(TRcpts, RemoveOne) {
    TRcpts rcpts;
    rcpts.Add({"first@yandex.ru", "1"});
    rcpts.Remove("first@yandex.ru");
    ASSERT_EQ(rcpts.GetSize(), 0ul);
}

TEST(TRcpts, RemoveTwo) {
    TRcpts rcpts;
    rcpts.Add({"first@yandex.ru", "1"});
    rcpts.Add({"second@yandex.ru", "2"});

    ASSERT_EQ(rcpts.GetSize(), 2ul);
    rcpts.Remove("first@yandex.ru");
    ASSERT_EQ(rcpts.GetSize(), 1ul);
    rcpts.Remove("second@yandex.ru");
    ASSERT_EQ(rcpts.GetSize(), 0ul);
}

TEST(TRcpts, RemoveFromEmpty) {
    TRcpts rcpts;
    ASSERT_EQ(rcpts.GetSize(), 0ul);
    rcpts.Remove("first@yandex.ru");
    ASSERT_EQ(rcpts.GetSize(), 0ul);
}

TEST(TRcpts, RemoveByEmpty) {
    TRcpts rcpts;
    rcpts.Add({"first@yandex.ru", "1"});
    ASSERT_EQ(rcpts.GetSize(), 1ul);
    rcpts.Remove("");
    ASSERT_EQ(rcpts.GetSize(), 1ul);
}

TEST(TRcpts, ContainsOne) {
    TRcpts rcpts;
    rcpts.Add({"first@yandex.ru", "1"});
    ASSERT_TRUE(rcpts.Contains("first@yandex.ru"));
    ASSERT_FALSE(rcpts.Contains("second@yandex.ru"));
}

TEST(TRcpts, ContainsTwo) {
    TRcpts rcpts;
    rcpts.Add({"first@yandex.ru", "1"});
    rcpts.Add({"second@yandex.ru", "2"});
    ASSERT_TRUE(rcpts.Contains("first@yandex.ru"));
    ASSERT_TRUE(rcpts.Contains("second@yandex.ru"));
}

TEST(TRcpts, ContainsNoRcpts) {
    TRcpts rcpts;
    ASSERT_FALSE(rcpts.Contains("first@yandex.ru"));
}

TEST(TRcpts, ContainsByEmptyNoRcpts) {
    TRcpts rcpts;
    ASSERT_FALSE(rcpts.Contains(""));
}

TEST(TRcpts, ContainsByEmptyOneRcpt) {
    TRcpts rcpts;
    rcpts.Add({"first@yandex.ru", "1"});
    ASSERT_FALSE(rcpts.Contains(""));
}

TEST(TRcpts, IterateOverOne) {
    TRcpts rcpts;
    rcpts.Add({"first@yandex.ru", "1"});

    std::vector<std::string> emails;
    for (const auto& rcpt : rcpts) {
        emails.emplace_back(rcpt.m_name);
    }
    ASSERT_EQ(emails, std::vector<std::string>{"first@yandex.ru"});
}

TEST(TRcpts, IterateOverTwo) {
    TRcpts rcpts;
    rcpts.Add({"first@yandex.ru", "1"});
    rcpts.Add({"second@yandex.ru", "2"});

    std::vector<std::string> emails;
    for (const auto& rcpt : rcpts) {
        emails.emplace_back(rcpt.m_name);
    }
    ASSERT_EQ(emails, (std::vector<std::string>{"first@yandex.ru", "second@yandex.ru"}));
}

TEST(TRcpts, IterateOverEmpty) {
    TRcpts rcpts;
    std::vector<std::string> emails;
    for (const auto& rcpt : rcpts) {
        emails.emplace_back(rcpt.m_name);
    }
    ASSERT_EQ(emails, std::vector<std::string>{});
}
