#include <mail/nwsmtp/src/resolver/impl.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>


using namespace NNwSmtp;
using namespace NNwSmtp::NDns;
using namespace testing;

struct MockIteratorPtr {
    bool operator==(const MockIteratorPtr& rhs) const {
        return Value == rhs.Value;
    }
    std::string operator*() const {
        return Value;
    }

public:
    std::string Value;
};

struct MockResolver {
    using iterator_ptr = MockIteratorPtr;
    using TPtrCallback = std::function<void(boost::system::error_code, iterator_ptr)>;

    MOCK_METHOD(void, async_resolve_ptr, (const std::string& query, TPtrCallback), ());
};

TEST(TResolver, IpSuccefullyResolved) {
    StrictMock<MockResolver> tmp;
    MockIteratorPtr iterator{"localhost"};
    EXPECT_CALL(tmp, async_resolve_ptr(_, _))
        .WillOnce(InvokeArgument<1>(boost::system::error_code{}, iterator));

    TImpl(tmp).AsyncResolveIp(
        boost::make_shared<TContext>(std::string{}, std::string{}, std::string{}, std::string{}), "127.0.0.1",
        [](auto ec, auto host) {
            ASSERT_FALSE(ec);
            ASSERT_EQ(host, "localhost");
        });
}

TEST(TResolver, DnsEntryNotFound) {
    StrictMock<MockResolver> tmp;
    EXPECT_CALL(tmp, async_resolve_ptr(_, _))
        .WillOnce(InvokeArgument<1>(boost::system::error_code{}, MockIteratorPtr()));

    TImpl(tmp).AsyncResolveIp(
        boost::make_shared<TContext>(std::string{}, std::string{}, std::string{}, std::string{}), "127.0.0.1",
        [](auto ec, auto host) {
            ASSERT_FALSE(ec);
            ASSERT_EQ(host, "");
    });
}

TEST(TResolver, OnErrorReturnsError) {
    StrictMock<MockResolver> tmp;
    EXPECT_CALL(tmp, async_resolve_ptr(_, _))
        .WillOnce(InvokeArgument<1>(boost::asio::error::operation_aborted, MockIteratorPtr()));

    TImpl(tmp).AsyncResolveIp(
        boost::make_shared<TContext>(std::string{}, std::string{}, std::string{}, std::string{}), "127.0.0.1",
        [](auto ec, auto) {
            ASSERT_EQ(ec, boost::asio::error::operation_aborted);
        });
}

TEST(TResolver, FqdnEndsWithPeriod) {
    StrictMock<MockResolver> tmp;
    MockIteratorPtr iterator{"sun.tuc.noao.edu."};
    EXPECT_CALL(tmp, async_resolve_ptr(_, _))
        .WillOnce(InvokeArgument<1>(boost::system::error_code{}, iterator));

    TImpl(tmp).AsyncResolveIp(
        boost::make_shared<TContext>(std::string{}, std::string{}, std::string{}, std::string{}), "123.123.123.123",
        [](auto ec, auto host) {
            ASSERT_FALSE(ec);
            ASSERT_EQ(host, "sun.tuc.noao.edu");
        });
}

TEST(TResolver, WrongEndpointThrows) {
    StrictMock<MockResolver> tmp;
    TImpl resolver(tmp);
    ASSERT_THROW(resolver.AsyncResolveIp(
        boost::make_shared<TContext>(std::string{}, std::string{}, std::string{}, std::string{}), "wrong_endpoint",
        [](auto, auto){}), std::exception);
}
