#include <mail/nwsmtp/src/web/send_mail/args.h>
#include <mail/nwsmtp/src/web/send_mail/make_args.h>

#include <mail/sova/include/nwsmtp/decoder/mimedecoder.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/asio.hpp>
#include <boost/range/algorithm.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/adaptor/transformed.hpp>

#include <memory>
#include <sstream>

namespace {

using namespace testing;
using namespace NNwSmtp;
using namespace NNwSmtp::NWeb;
using namespace NNwSmtp::NWeb::NSendMail;

auto GetArgs() {
    auto args = TArgs("::1", {"george@pig.ru"},
        {"mummy@pig.ru", "daddy@pig.ru"}, {"grandpa@pig.ru", "grandma@pig.ru"}, {"lid1", "lid2"},
        {"label1", "label2"},  {"label3", "label4"}, {{"george@pig.ru", {"label5", "label6"}},
        {"grandpa@pig.ru", {"label7", "label8"}}}, "1", "1", "pighost");
    args.Service = "test";
    args.FromEmail = "peppa@pig.ru";
    args.Uid = "42";
    args.SpamCheck = "1";
    args.AvirCheck = "1";
    args.RawIp = boost::asio::ip::make_address("::1");
    return std::make_shared<TArgs>(args);
}

std::vector<std::pair<std::string, std::string>> GetHintValue(const std::string& hint) {
    std::string::size_type pos = hint.find(':');
    if (pos == std::string::npos) {
        return {};
    }

    CMimeDecoder encoder;
    std::stringstream decodedValue;
    std::stringstream header;
    std::vector<std::string> headersValue;

    header << boost::trim_copy(hint.substr(pos + 1));
    encoder.decode("base64", header, decodedValue);
    std::string stringHeader = decodedValue.str();
    boost::split(headersValue, stringHeader, boost::is_any_of("\n"), boost::token_compress_on);

    auto result = headersValue
        | boost::adaptors::filtered([&] (const auto& v) {
            return !v.empty() && v.find('=') != std::string::npos;
        })
        | boost::adaptors::transformed([&] (const auto& v) {
            std::string::size_type pos = v.find('=');
            return make_pair(v.substr(0, pos), v.substr(pos + 1));

        })
        | boost::adaptors::filtered([&] (const auto& v) {
            return v.first != "received_date";
        });

    return {result.begin(), result.end()};
}

TEST(MakeArgsTest, for_send_mail_args_should_return_x_yandex_common_hint) {
    auto hint = MakeXYandexCommonHint(GetArgs());

    EXPECT_THAT(
        GetHintValue(hint.Str()),
        ElementsAre(
            std::pair{"ipfrom", "::1"},
            std::pair{"label", "label3"},
            std::pair{"label", "label4"},
            std::pair{"source", "test"}
        )
    );
}

TEST(MakeArgsTest, for_send_mail_args_should_return_x_yandex_sender_hint) {
    auto hint = MakeXYandexSenderHint(GetArgs());

    EXPECT_THAT(
        GetHintValue(hint.Str()),
        ElementsAre(
            std::pair{"bcc", "grandma@pig.ru"},
            std::pair{"bcc", "grandpa@pig.ru"},
            std::pair{"email", "peppa@pig.ru"},
            std::pair{"filters", "0"},
            std::pair{"host", "pighost"},
            std::pair{"label", "label1"},
            std::pair{"label", "label2"},
            std::pair{"lid", "lid1"},
            std::pair{"lid", "lid2"},
            std::pair{"notify", "1"},
            std::pair{"save_to_sent", "1"},
            std::pair{"skip_loop_prevention", "1"}
        )
    );
}

TEST(MakeArgsTest, for_send_mail_args_should_return_x_yandex_personal_hint) {
    auto hint = MakeXYandexPersonalHint("peppa@pig.ru", {"label1", "label2"});

    EXPECT_THAT(
        GetHintValue(hint.Str()),
        ElementsAre(
            std::pair{"email", "peppa@pig.ru"},
            std::pair{"label", "label1"},
            std::pair{"label", "label2"}
        )
    );
}

TEST(MakeArgsTest, for_send_system_mail_args_should_return_send_system_mail_x_yandex_hint) {
    auto hint = MakeSendSystemMailXYandexHint(GetArgs());

    EXPECT_THAT(
        GetHintValue(hint.Str()),
        ElementsAre(
            std::pair{"label", "label1"},
            std::pair{"label", "label2"},
            std::pair{"lid", "lid1"},
            std::pair{"lid", "lid2"}
        )
    );
}

TEST(MakeArgsTest,
        for_send_mail_args_with_zero_spam_check_should_return_headers_with_spam_header) {
    auto args = GetArgs();
    args->SpamCheck = "0";
    EXPECT_EQ(MakeAdditionalHeaders(args), "X-Yandex-Spam: 1\r\n");
}

TEST(MakeArgsTest,
        for_send_mail_args_with_none_zero_spam_check_should_return_headers_without_spam_header) {
    auto args = GetArgs();
    args->SpamCheck = "1";
    EXPECT_EQ(MakeAdditionalHeaders(args), "");
}

TEST(MakeArgsTest,
        for_send_mail_args_with_zero_avir_check_should_return_headers_with_avir_header) {
    auto args = GetArgs();
    args->AvirCheck = "0";
    EXPECT_EQ(MakeAdditionalHeaders(args), "X-Yandex-Avir: 1\r\n");
}

TEST(MakeArgsTest,
        for_send_mail_args_with_none_zero_avir_check_should_return_headers_without_avir_header) {
    auto args = GetArgs();
    args->AvirCheck = "1";
    EXPECT_EQ(MakeAdditionalHeaders(args), "");
}

TEST(MakeArgsTest, for_send_mail_args_should_return_recipients) {
    auto args = GetArgs();
    MakeRecipients(args);
    EXPECT_THAT(
        args->Recipients,
        ElementsAre(
            "george@pig.ru",
            "mummy@pig.ru",
            "daddy@pig.ru",
            "grandpa@pig.ru",
            "grandma@pig.ru"
        )
    );
}

MATCHER_P(EqHostArgs, other, "Equality matcher for host arguments") {
    return arg->Host == other->Host
        && arg->Ip == other->Ip
        && arg->RawIp == other->RawIp
        && arg->RemoteIp == other->RemoteIp;
}

TEST(MakeArgsTest, for_not_empty_host_name_and_ip_params_should_return_host_params) {
    auto result = GetArgs();
    MakeHostArgs(result, "127.0.0.1");
    auto args = GetArgs();
    args->RemoteIp = "127.0.0.1";
    EXPECT_THAT(result, EqHostArgs(args));
}

TEST(MakeArgsTest, for_bad_ip_params_should_return_host_params_with_remote_ip_value) {
    auto result = GetArgs();
    result->Ip = "pig";
    MakeHostArgs(result, "127.0.0.1");
    auto args = GetArgs();
    args->Host = "127.0.0.1";
    args->Ip = "127.0.0.1";
    args->RawIp = boost::asio::ip::make_address("127.0.0.1");
    args->RemoteIp = "127.0.0.1";
    EXPECT_THAT(result, EqHostArgs(args));
}

TEST(MakeArgsTest, for_empty_ip_params_should_return_host_params_with_remote_ip_value) {
    auto result = GetArgs();
    result->Ip = "";
    MakeHostArgs(result, "127.0.0.1");
    auto args = GetArgs();
    args->Host = "127.0.0.1";
    args->Ip = "127.0.0.1";
    args->RawIp = boost::asio::ip::make_address("127.0.0.1");
    args->RemoteIp = "127.0.0.1";
    EXPECT_THAT(result, EqHostArgs(args));
}

} // namespace anonymous
