#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <mail/nwsmtp/src/big_ml/types.h>
#include <mail/nwsmtp/src/options.h>
#include <mail/nwsmtp/src/log.h>
#include <mail/nwsmtp/src/web/send_mail/args.h>
#include <mail/nwsmtp/src/web/send_mail/error_code.h>
#include <mail/nwsmtp/src/web/send_mail/send_mail.h>
#include <mail/nwsmtp/src/types.h>
#include <mail/nwsmtp/ut/send_mail/send_mail_mocks.h>

#include <memory>
#include <vector>

namespace NNwSmtp::NBigML {

static bool operator ==(const TRequest& lhs, const TRequest& rhs) {
    return lhs.FromEmail == rhs.FromEmail
        && lhs.ToEmail == rhs.ToEmail;
}

}

namespace NNwSmtp::NML {

static bool operator ==(const TRequest& lhs, const TRequest& rhs) {
    return lhs.Email == rhs.Email;
}

}

namespace {

using namespace testing;
using namespace NNwSmtp;
using namespace NNwSmtp::NWeb::NSendMail;
using namespace NNwSmtp::NWeb::NSendMail::NTest;

TArgsPtr MakeArgs(std::string fromEmail, std::vector<std::string> recipients) {
    auto args = std::make_shared<TArgs>();
    args->FromEmail = std::move(fromEmail);
    args->Recipients = std::move(recipients);
    return args;
}

auto MakeConfig() {
    Options::LLOpts corpListP {};
    corpListP.use = false;
    return std::make_shared<TConfig>(true, true, true, true,
        RoutingSettings::DNS, true, std::move(corpListP), false,
        Options::DeliveryToSenderControl {true, true}, 3, "clusterName",
        "hostName", 100
    );
}

struct TTestSendMail: Test {

    TTestSendMail() {
        glog = std::make_shared<TLog>();
    }

    std::shared_ptr<StrictMock<TRouterClientMock>> RouterClientMock
        = std::make_shared<StrictMock<TRouterClientMock>>();

    std::shared_ptr<StrictMock<TBBChecksMock>> BlackBoxChecksMock
        = std::make_shared<StrictMock<TBBChecksMock>>();

    std::shared_ptr<StrictMock<TBigMLMock>> BigMLMock
        = std::make_shared<StrictMock<TBigMLMock>>();

    std::shared_ptr<StrictMock<TMLMock>> MLMock
        = std::make_shared<StrictMock<TMLMock>>();

    std::shared_ptr<StrictMock<TDeliveryMock>> DeliveryMock
        = std::make_shared<StrictMock<TDeliveryMock>>();

    TConfigPtr Config = MakeConfig();

    void CallSendMail(
        THandler handler,
        TArgsPtr args
    ) {
        auto sendMailHandle = std::make_shared<TSendMail>(
            handler,
            RouterClientMock,
            BlackBoxChecksMock,
            BigMLMock,
            MLMock,
            DeliveryMock,
            Config,
            args
        );

        yplatform::spawn(sendMailHandle);
    }
};

TEST_F(TTestSendMail, for_from_email_that_is_longer_than_256_characters_should_return_error) {
    const InSequence s;
    std::string fromEmail = R"(LQr7jxJ9NiyUFnYiCNQxwbxbQ3epFm1K2jMllSz13uRhtcZ3i7eOuhD0FB07OLL3R63ozAYh)"
        R"(MmDlTtT1cAIlo0uLGVaeFelZesWvgy3dLMh5DAFiMp0ThDOfJbMKesEoPmydKGi7GhIwyFQuRHrS7pjzKVbJSOCfOy)"
        R"(BIrggbZUO5m74pJKUN0iIYn3qbBYH7Y65Kbc0a98tSugxbNNobrmifvZ9v4y7JOYPECVvd31lZN8v31Kmkiyrd6y5xcF6B@ya.ru)";
    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, EError::PathToLong);
        },
        MakeArgs(fromEmail, {})
    );
}

TEST_F(TTestSendMail, for_not_valid_from_email_should_return_error) {
    const InSequence s;
    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, EError::BadAddressSyntax);
        },
        MakeArgs("mummy_pig.com", {})
    );
}

TEST_F(TTestSendMail, for_not_checked_from_email_should_no_return_error) {
    const InSequence s;
    Config->DeliveryToSenderControl.use = false;
    auto args = MakeArgs("mummy@pig.com", {});
    args->CheckMailFrom = false;
    CallSendMail(
        [](auto ec) {
            ASSERT_FALSE(ec);
        },
        args
    );
}

TEST_F(TTestSendMail, for_failed_black_box_from_email_check_should_return_error) {
    const InSequence s;
    EXPECT_CALL(*BlackBoxChecksMock, CheckMailFrom(_, "mummy@pig.com", _, _))
        .WillOnce(
            InvokeArgument<3>(
                ymod_httpclient::http_error::code::ssl_error,
                NBlackBox::TResponse {}
            )
        );
    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
        },
        MakeArgs("mummy@pig.com", {})
    );
}

TEST_F(TTestSendMail, for_empty_recipients_list_should_no_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    Config->DeliveryToSenderControl.use = false;
    CallSendMail(
        [](auto ec) {
            ASSERT_FALSE(ec);
        },
        MakeArgs("mummy@pig.com", {})
    );
}

TEST_F(TTestSendMail, for_not_valid_recipients_should_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, EError::BadRecipient);
        },
        MakeArgs("mummy@pig.com", {"peppa_pig.com"})
    );
}

TEST_F(TTestSendMail, for_size_of_recipients_list_greater_than_limit_should_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, EError::ToManyRecipients);
        },
        MakeArgs("mummy@pig.com", {"peppa1@pig.com", "peppa2@pig.com", "peppa3@pig.com"})
    );
}

TEST_F(TTestSendMail, for_failed_router_request_should_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    EXPECT_CALL(*RouterClientMock, asyncRoute(_, "pig.com", _))
        .WillOnce(
            InvokeArgument<2>(
                ymod_httpclient::http_error::code::ssl_error,
                NNwSmtp::Router::Response {}
            )
        );
    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
        },
        MakeArgs("mummy@pig.com", {"peppa@pig.com"})
    );
}

TEST_F(TTestSendMail, for_failed_black_box_recipient_check_should_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    EXPECT_CALL(*RouterClientMock, asyncRoute(_, "pig.com", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NNwSmtp::Router::Response {
                    DomainType::LOCAL,
                    "mx.pig.com"
                }
            )
        );
    EXPECT_CALL(*BlackBoxChecksMock, CheckRecipient(_, "peppa@pig.com", _, _, _, _))
        .WillOnce(
            InvokeArgument<5>(
                ymod_httpclient::http_error::code::ssl_error,
                NBlackBox::TResponse {}
            )
        );
    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
        },
        MakeArgs("mummy@pig.com", {"peppa@pig.com"})
    );
}

TEST_F(TTestSendMail, for_not_valid_second_recipient_should_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    EXPECT_CALL(*RouterClientMock, asyncRoute(_, "pig.com", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NNwSmtp::Router::Response {
                    DomainType::EXTERNAL,
                    "mx.pig.com"
                }
            )
        );
    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, EError::BadRecipient);
        },
        MakeArgs("mummy@pig.com", {"peppa1@pig.com", "george_pig.com"})
    );
}

TEST_F(TTestSendMail, for_failed_big_ml_request_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    Config->BigMLUse = true;
    EXPECT_CALL(*RouterClientMock, asyncRoute(_, "pig.com", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NNwSmtp::Router::Response {
                    DomainType::LOCAL,
                    "mx.pig.com"
                }
            )
        );

    auto reponse = NBlackBox::TResponse {};
    reponse.IsMailList = true;
    EXPECT_CALL(*BlackBoxChecksMock, CheckRecipient(_, "peppa@pig.com", _, _, _, _))
        .WillOnce(InvokeArgument<5>(TErrorCode {}, reponse));

    EXPECT_CALL(*BigMLMock, Run(_, NBigML::TRequest{"mummy@pig.com", "peppa@pig.com"}, _))
        .WillOnce(
            InvokeArgument<2>(
                ymod_httpclient::http_error::code::ssl_error,
                NBigML::TResponse {}
            )
        );

    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
        },
        MakeArgs("mummy@pig.com", {"peppa@pig.com"})
    );
}

TEST_F(TTestSendMail, for_empty_big_ml_response_should_no_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    Config->BigMLUse = true;
    EXPECT_CALL(*RouterClientMock, asyncRoute(_, "pig.com", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NNwSmtp::Router::Response {
                    DomainType::LOCAL,
                    "mx.pig.com"
                }
            )
        );

    auto reponse = NBlackBox::TResponse {};
    reponse.IsMailList = true;
    EXPECT_CALL(*BlackBoxChecksMock, CheckRecipient(_, "peppa@pig.com", _, _, _, _))
        .WillOnce(InvokeArgument<5>(TErrorCode {}, reponse));

    EXPECT_CALL(*BigMLMock, Run(_, NBigML::TRequest{"mummy@pig.com", "peppa@pig.com"}, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NBigML::TResponse {}
            )
        );

    EXPECT_CALL(*DeliveryMock, Run(_, _, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                std::string {}
            )
        );

    CallSendMail(
        [](auto ec) {
            ASSERT_FALSE(ec);
        },
        MakeArgs("mummy@pig.com", {"peppa@pig.com"})
    );
}

TEST_F(TTestSendMail, for_failed_ml_request_should_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    Config->CheckRcpt = false;
    Config->CorpList.use = true;
    EXPECT_CALL(*RouterClientMock, asyncRoute(_, "pig.com", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NNwSmtp::Router::Response {
                    DomainType::LOCAL,
                    "mx.pig.com"
                }
            )
        );

    EXPECT_CALL(*MLMock, Run(_, NML::TRequest{"peppa@pig.com"}, _))
        .WillOnce(
            InvokeArgument<2>(
                ymod_httpclient::http_error::code::ssl_error,
                NML::TResponse {}
            )
        );

    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
        },
        MakeArgs("mummy@pig.com", {"peppa@pig.com"})
    );
}

TEST_F(TTestSendMail,
        for_ml_request_empty_subscribers_should_no_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    Config->CheckRcpt = false;
    Config->CorpList.use = true;
    EXPECT_CALL(*RouterClientMock, asyncRoute(_, "pig.com", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NNwSmtp::Router::Response {
                    DomainType::LOCAL,
                    "mx.pig.com"
                }
            )
        );

    EXPECT_CALL(*MLMock, Run(_, NML::TRequest{"peppa@pig.com"}, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NML::TResponse {}
            )
        );

    EXPECT_CALL(*DeliveryMock, Run(_, _, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                std::string {}
            )
        );

    CallSendMail(
        [](auto ec) {
            ASSERT_FALSE(ec);
        },
        MakeArgs("mummy@pig.com", {"peppa@pig.com"})
    );
}

TEST_F(TTestSendMail, for_read_only_mail_list_should_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    Config->CheckRcpt = false;
    Config->CorpList.use = true;
    EXPECT_CALL(*RouterClientMock, asyncRoute(_, "pig.com", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NNwSmtp::Router::Response {
                    DomainType::LOCAL,
                    "mx.pig.com"
                }
            )
        );

    EXPECT_CALL(*MLMock, Run(_, NML::TRequest{"peppa@pig.com"}, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NML::TResponse {true, true, true, {"peppa@pig.com"}, {}}
            )
        );

    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, EError::ReadOnlyMailList);
        },
        MakeArgs("mummy@pig.com", {"peppa@pig.com"})
    );
}

TEST_F(TTestSendMail, for_internal_mail_list_should_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    Config->CheckRcpt = false;
    Config->CorpList.use = true;
    EXPECT_CALL(*RouterClientMock, asyncRoute(_, "pig.com", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NNwSmtp::Router::Response {
                    DomainType::LOCAL,
                    "mx.pig.com"
                }
            )
        );

    EXPECT_CALL(*MLMock, Run(_, NML::TRequest{"peppa@pig.com"}, _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NML::TResponse {true, false, true, {"peppa@pig.com"}, {}}
            )
        );

    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, EError::MailListWriteProhibited);
        },
        MakeArgs("mummy@pig.com", {"peppa@pig.com"})
    );
}

TEST_F(TTestSendMail, for_failed_delivery_should_return_error) {
    const InSequence s;
    Config->CheckSender = false;
    EXPECT_CALL(*RouterClientMock, asyncRoute(_, "pig.com", _))
        .WillOnce(
            InvokeArgument<2>(
                TErrorCode {},
                NNwSmtp::Router::Response {
                    DomainType::EXTERNAL,
                    "mx.pig.com"
                }
            )
        );

    EXPECT_CALL(*DeliveryMock, Run(_, _, _))
        .WillOnce(
            InvokeArgument<2>(
                NAsyncDlv::EError::DeliveryRejected,
                std::string {}
            )
        );

    CallSendMail(
        [](auto ec) {
            ASSERT_TRUE(ec);
            ASSERT_EQ(ec, NAsyncDlv::EError::DeliveryRejected);
        },
        MakeArgs("mummy@pig.com", {"peppa@pig.com"})
    );
}

} // namespace anonymous
