#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <mail/nwsmtp/src/delivery/async/error_code.h>
#include <mail/nwsmtp/src/options.h>
#include <mail/nwsmtp/src/rcpts.h>
#include <mail/nwsmtp/src/rcpt_context.h>
#include <mail/nwsmtp/src/web/send_mail/utils.h>

#include <memory>

namespace {

using namespace testing;
using namespace NNwSmtp;
using namespace NNwSmtp::NWeb::NSendMail;

auto MakeConfig() {
    return std::make_shared<TConfig>(true, true, true, true,
        RoutingSettings::DNS, true, Options::LLOpts {}, false,
        Options::DeliveryToSenderControl {true, true}, 3, "clusterName",
        "hostName", 100
    );
}

auto MakeRcptContext() {
    return std::make_shared<TRcptContext>();
}

TEST(TestSendMailUtils, for_email_not_containing_percent_should_return_this_email) {
    EXPECT_EQ(TransformPercentRecipient("to@ya.ru", MakeConfig()), "to@ya.ru");
}

TEST(TestSendMailUtils, for_email_containing_percent_in_first_position_should_return_this_email) {
    EXPECT_EQ(TransformPercentRecipient("\%to@ya.ru", MakeConfig()), "\%to@ya.ru");
}

TEST(TestSendMailUtils, for_email_containing_percent_should_return_modified_email) {
    EXPECT_EQ(TransformPercentRecipient("to\%to.ru@ya.ru", MakeConfig()), "to@to.ru");
}

TEST(TestCheckRecipient, for_email_without_dog_should_return_false) {
    EXPECT_FALSE(CheckAt("to"));
}

TEST(TestCheckRecipient, for_email_with_dog_in_last_position_should_return_false) {
    EXPECT_FALSE(CheckAt("to@"));
}

TEST(TestCheckRecipient, for_email_with_dog_in_first_position_should_return_false) {
    EXPECT_FALSE(CheckAt("@to"));
}

TEST(TestCheckRecipient,
        for_email_that_contains_illegal_characters_and_allow_strict_ascii_recipient_should_return_false) {
    EXPECT_FALSE(CheckStrictAscii("(to)@ya", MakeConfig()));
}

TEST(TestCheckRecipient,
        for_email_that_contains_illegal_characters_and_not_allow_strict_ascii_recipient_should_return_true) {
    const auto config = MakeConfig();
    config->StrictAsciiRecipient = false;
    EXPECT_TRUE(CheckStrictAscii("(to)@ya", config));
}

TEST(TestCheckRecipient, for_email_with_percent_character_in_first_position_should_return_false) {
    EXPECT_FALSE(CheckPercent("\%to@ya", MakeConfig()));
}

TEST(TestCheckRecipient, for_email_with_percent_character_and_not_allow_percent_hack_should_return_false) {
    const auto config = MakeConfig();
    config->AllowPercentHack = false;
    EXPECT_FALSE(CheckPercent("to%@ya", config));
}

TEST(TestIsRouting, for_alias_email_should_return_false) {
    const auto vc = MakeRcptContext();
    vc->IsAlias = true;
    EXPECT_FALSE(IsRouting(MakeConfig(), vc));
}

TEST(TestIsRouting, for_none_primary_routing_should_return_false) {
    const auto config = MakeConfig();
    config->Primary = NNwSmtp::RoutingSettings::Type::NONE;
    EXPECT_FALSE(IsRouting(config, MakeRcptContext()));
}

TEST(TestIsRouting, for_not_none_primary_routing_and_not_alias_email_should_return_true) {
    const auto vc = MakeRcptContext();
    vc->IsAlias = false;
    const auto config = MakeConfig();
    config->Primary = NNwSmtp::RoutingSettings::Type::DNS;
    EXPECT_TRUE(IsRouting(config, vc));
}

TEST(TestIsBlackBoxRecipientCheck,
        for_check_recipient_and_none_primary_routing_and_not_alias_email_should_return_true) {
    const auto config = MakeConfig();
    config->CheckRcpt = true;
    config->Primary = NNwSmtp::RoutingSettings::Type::NONE;
    const auto vc = MakeRcptContext();
    vc->IsAlias = false;
    EXPECT_TRUE(IsBlackBoxRecipientCheck(config, vc));
}

TEST(TestIsBlackBoxRecipientCheck,
        for_check_recipient_and_dns_primary_routing_and_not_alias_email_should_return_true) {
    const auto config = MakeConfig();
    config->CheckRcpt = true;
    config->Primary = NNwSmtp::RoutingSettings::Type::DNS;
    const auto vc = MakeRcptContext();
    vc->IsAlias = false;
    EXPECT_FALSE(IsBlackBoxRecipientCheck(config, vc));
}

TEST(TestIsBlackBoxRecipientCheck,
        for_check_recipient_and_domain_type_local_and_not_alias_email_should_return_true) {
    const auto config = MakeConfig();
    config->CheckRcpt = true;
    const auto vc = MakeRcptContext();
    vc->IsAlias = false;
    vc->DomainType = NNwSmtp::DomainType::LOCAL;
    EXPECT_TRUE(IsBlackBoxRecipientCheck(config, vc));
}

TEST(TestIsBlackBoxRecipientCheck,
        for_check_recipient_and_domain_type_not_local_and_not_alias_email_should_return_false) {
    const auto config = MakeConfig();
    config->CheckRcpt = true;
    const auto vc = MakeRcptContext();
    vc->IsAlias = false;
    vc->DomainType = NNwSmtp::DomainType::EXTERNAL;
    EXPECT_FALSE(IsBlackBoxRecipientCheck(config, vc));
}

TEST(TestIsBlackBoxRecipientCheck,
        for_not_check_recipient_should_return_false) {
    const auto config = MakeConfig();
    config->CheckRcpt = false;
    EXPECT_FALSE(IsBlackBoxRecipientCheck(config, MakeRcptContext()));
}

TEST(TestIsBlackBoxRecipientCheck, for_alias_email_should_return_false) {
    const auto vc = MakeRcptContext();
    vc->IsAlias = true;
    EXPECT_FALSE(IsBlackBoxRecipientCheck(MakeConfig(), vc));
}

TEST(TestIsBlackBoxMailFromCheck,
        for_not_check_mail_from_config_and_check_mail_from_argument_should_return_false) {
    const auto config = MakeConfig();
    config->CheckSender = false;
    EXPECT_FALSE(IsBlackBoxMailFromCheck(config, true));
}

TEST(TestIsBlackBoxMailFromCheck,
        for_check_mail_from_config_and_not_check_mail_from_argument_should_return_false) {
    const auto config = MakeConfig();
    config->CheckSender = true;
    EXPECT_FALSE(IsBlackBoxMailFromCheck(config, false));
}

TEST(TestIsBlackBoxMailFromCheck,
        for_check_mail_from_config_and_check_mail_from_argument_should_return_true) {
    const auto config = MakeConfig();
    config->CheckSender = true;
    EXPECT_TRUE(IsBlackBoxMailFromCheck(config, true));
}

TEST(TestValidateAddress, for_empty_address_should_return_true) {
    EXPECT_TRUE(ValidateAddress("", MakeConfig()));
}

TEST(TestValidateAddress, for_email_that_contains_illegal_characters_should_return_false) {
    EXPECT_FALSE(ValidateAddress("(ya)@ya.ru", MakeConfig()));
}

TEST(TestValidateAddress, for_invalid_email_syntax_end_true_check_sender_syntax_should_return_false) {
    const auto config = MakeConfig();
    config->CheckSenderSyntax = true;
    EXPECT_FALSE(ValidateAddress("ya.ru", config));
}

TEST(TestValidateAddress, for_invalid_email_syntax_end_false_check_sender_syntax_should_return_true) {
    const auto config = MakeConfig();
    config->CheckSenderSyntax = false;
    EXPECT_TRUE(ValidateAddress("ya.ru", config));
}

TEST(TestValidateAddress, for_valid_email_syntax_should_return_true) {
    EXPECT_TRUE(ValidateAddress("ya@ya.ru", MakeConfig()));
}

TEST(TestGetDomain, for_valid_email_should_return_domain) {
    EXPECT_EQ(GetDomain("ya@ya.ru"), "ya.ru");
}

TEST(TestGetDomain, for_invalid_email_should_return_empty_string) {
    EXPECT_EQ(GetDomain("yaya.ru"), "");
}

struct TTestGetBanReason: TestWithParam<std::tuple<TErrorCode, std::optional<std::string>>> {
};

TEST_P(TTestGetBanReason, test_get_ban_reason) {
    EXPECT_EQ(
        GetBanReason(std::get<0>(GetParam())),
        std::get<1>(GetParam())
    );
}

INSTANTIATE_TEST_SUITE_P(
    test_get_ban_reason,
    TTestGetBanReason,
    Values(
        std::make_tuple(
            NAsyncDlv::EError::RfcFailRejected,
            "RfcFail"
        ),
        std::make_tuple(
            NAsyncDlv::EError::UrlRblRejected,
            "UrlRbl"
        ),
        std::make_tuple(
            NAsyncDlv::EError::BadKarmaRejected,
            "BadKarma"
        ),
        std::make_tuple(
            NAsyncDlv::EError::MailLimitsRejected,
            "MailLimits"
        ),
        std::make_tuple(
            NAsyncDlv::EError::PddAdminKarmaRejected,
            "PddAdminKarma"
        ),
        std::make_tuple(
            NAsyncDlv::EError::BouncesRejected,
            "Bounces"
        ),
        std::make_tuple(
            NAsyncDlv::EError::SpamComplRejected,
            "SpamCompl"
        ),
        std::make_tuple(
            NAsyncDlv::EError::MaliciousRejected,
            std::nullopt
        )
    )
);

} // namespace anonymous
