#include "init_log.h"

#include <mail/nwsmtp/src/settings/settings_client.h>
#include <mail/nwsmtp/src/utils.h>

#include <mail/nwsmtp/ut/util/http_client_mock.h>
#include <mail/nwsmtp/ut/util/tvm2_module_mock.h>

#include <utility>

namespace {

using namespace testing;

using NNwSmtp::NSettings::make_error_code;
using NNwSmtp::NSettings::TAuthSettings;
using NNwSmtp::NSettings::TCallback;
using NNwSmtp::NSettings::TSettingsClient;
using NNwSmtp::NSettings::TUid;
using NNwSmtp::NTests::GetStrictMockedClusterClient;
using NNwSmtp::NTests::GetStrictMockedTvm2Module;
using NNwSmtp::NTests::TClusterClientMock;
using NNwSmtp::NTests::TGetClusterClientMockWrapper;
using NNwSmtp::NTests::TGetTvm2ModuleMockWrapper;
using NNwSmtp::NTests::TTvm2ModuleMock;
using NNwSmtp::NUtil::TGetTvm2Module;
using NNwSmtp::TContext;
using NNwSmtp::TContextPtr;

using Options = ymod_httpclient::cluster_call::options;
using Request = yhttp::request;
using Response = yhttp::response;

const std::string INCORRECT_RESPONSE{
    R"({"settings":{"profile":{"single_settings":{"enable_imap":"on","enable_pop":"on"}},)"
    R"("parametersINCORRECT":{"single_settings":{"enable_imap_auth_plain":"on"}}}})"};
const std::string CORRECT_RESPONSE_WITH_EMPTY_FIELDS{
    R"({"settings":{"profile":{"single_settings":{"enable_imap":"","enable_pop":""}},)"
    R"("parameters":{"single_settings":{"enable_imap_auth_plain":""}}}})"};
const std::string CORRECT_RESPONSE_WITH_MISSING_FIELDS{
    R"({"settings":{"profile":{"single_settings":{"enable_imap":"","enable_pop":""}},)"
    R"("parameters":{"single_settings":{}}}})"};
const std::string CORRECT_RESPONSE_WITH_NONEMPTY_FIELDS{
    R"({"settings":{"profile":{"single_settings":{"enable_imap":"on","enable_pop":"on"}},)"
    R"("parameters":{"single_settings":{"enable_imap_auth_plain":"on"}}}})"};

struct TTestSettingsClient : Test {
    void SetUp() override {
        NTesting::InitGlobalLog();
    }

    void TestGetAuthSettings(TCallback callback)
    {
        const TSettingsClient settingsClient;
        settingsClient.GetAuthSettings(Context, Uid, IoContext, std::move(callback), GetTvm2Module,
            GetClusterClient);
        IoContext.run();
    }

    const std::shared_ptr<StrictMock<TTvm2ModuleMock>> Tvm2Module{GetStrictMockedTvm2Module()};
    TGetTvm2ModuleMockWrapper GetTvm2Module;
    const std::shared_ptr<StrictMock<TClusterClientMock>> ClusterClient{GetStrictMockedClusterClient()};
    TGetClusterClientMockWrapper GetClusterClient;
    const std::string ConnectionId{"ConnectionId"};
    const std::string EnvelopeId{"EnvelopeId"};
    const TContextPtr Context{boost::make_shared<TContext>(ConnectionId, EnvelopeId, std::string{},
        std::string{})};
    const TUid Uid{"42"};
    const std::string ServiceName{"settings"};
    const Request RequestToSettingsWithoutHeaders{Request::GET("/get?uid=" + Uid +
        "&settings_list=enable_pop%0denable_imap%0denable_imap_auth_plain&service=smtp")};
    const std::string ServiceTicket{"ServiceTicket"};
    const Request RequestToSettingsWithHeaders{Request::GET("/get?uid=" + Uid +
        "&settings_list=enable_pop%0denable_imap%0denable_imap_auth_plain&service=smtp",
        "X-Ya-Service-Ticket: " + ServiceTicket + "\r\n")};
    const int StatusOk{200};
    const int StatusNonRetryable{404};
    const int StatusRetryable{500};
    boost::asio::io_context IoContext;
};

TEST_F(TTestSettingsClient, for_tvm2_module_error_must_return_ec_tvm_service_ticket_error) {
    const InSequence sequence;
    EXPECT_CALL(*GetClusterClient.impl, call()).WillOnce(Return(ClusterClient));
    EXPECT_CALL(*GetTvm2Module.impl, call()).WillOnce(Return(Tvm2Module));
    EXPECT_CALL(*Tvm2Module, get_service_ticket(ServiceName, _)).WillOnce(Return(
        ymod_tvm::error::make_error_code(ymod_tvm::error::unknown_error)));
    TestGetAuthSettings([&](auto errorCode, TAuthSettings authSettings = {}) {
        ASSERT_TRUE(errorCode);
        EXPECT_EQ(make_error_code(NNwSmtp::NSettings::EC_TVM_SERVICE_TICKET_ERROR), errorCode);
        EXPECT_EQ((TAuthSettings{}), authSettings);
    });
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(Tvm2Module.get()));
}

TEST_F(TTestSettingsClient, for_cluster_client_error_must_return_error) {
    const InSequence sequence;
    EXPECT_CALL(*GetClusterClient.impl, call()).WillOnce(Return(ClusterClient));
    EXPECT_CALL(*GetTvm2Module.impl, call()).WillOnce(Return(Tvm2Module));
    EXPECT_CALL(*Tvm2Module, get_service_ticket(ServiceName, _)).WillOnce(DoAll(
        SetArgReferee<1>(ServiceTicket), Return(boost::system::error_code{})));
    const auto expectedErrorCode{ymod_httpclient::http_error::make_error_code(yhttp::errc::connect_error)};
    EXPECT_CALL(*ClusterClient, async_run(_, RequestToSettingsWithHeaders, _, _)).
        WillOnce(WithArg<3>([&](auto&& yieldCtx){boost::asio::post(IoContext, [&, yieldCtx]{
        yieldCtx(expectedErrorCode, Response{StatusOk, {}, CORRECT_RESPONSE_WITH_EMPTY_FIELDS, {}});});}));
    TestGetAuthSettings([&](auto errorCode, TAuthSettings authSettings = {}) {
        ASSERT_TRUE(errorCode);
        EXPECT_EQ(expectedErrorCode, errorCode);
        EXPECT_EQ((TAuthSettings{}), authSettings);
    });
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(Tvm2Module.get()));
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(ClusterClient.get()));
}

TEST_F(TTestSettingsClient, for_cluster_client_non_retryable_status_must_return_ec_non_retryable_status) {
    const InSequence sequence;
    EXPECT_CALL(*GetClusterClient.impl, call()).WillOnce(Return(ClusterClient));
    EXPECT_CALL(*GetTvm2Module.impl, call()).WillOnce(Return(Tvm2Module));
    EXPECT_CALL(*Tvm2Module, get_service_ticket(ServiceName, _)).WillOnce(DoAll(
        SetArgReferee<1>(ServiceTicket), Return(boost::system::error_code{})));
    EXPECT_CALL(*ClusterClient, async_run(_, RequestToSettingsWithHeaders, _, _)).
        WillOnce(WithArg<3>([&](auto&& yieldCtx){boost::asio::post(IoContext, [&, yieldCtx]{
        yieldCtx(ymod_httpclient::http_error::make_error_code(yhttp::errc::success),
        Response{StatusNonRetryable, {}, CORRECT_RESPONSE_WITH_EMPTY_FIELDS, {}});});}));
    TestGetAuthSettings([&](auto errorCode, TAuthSettings authSettings = {}) {
        ASSERT_TRUE(errorCode);
        EXPECT_EQ(make_error_code(NNwSmtp::NSettings::EC_NON_RETRYABLE_STATUS), errorCode);
        EXPECT_EQ((TAuthSettings{}), authSettings);
    });
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(Tvm2Module.get()));
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(ClusterClient.get()));
}

TEST_F(TTestSettingsClient, for_cluster_client_retries_exceeded_must_return_ec_retries_exceeded) {
    const InSequence sequence;
    EXPECT_CALL(*GetClusterClient.impl, call()).WillOnce(Return(ClusterClient));
    EXPECT_CALL(*GetTvm2Module.impl, call()).WillOnce(Return(Tvm2Module));
    EXPECT_CALL(*Tvm2Module, get_service_ticket(ServiceName, _)).WillOnce(DoAll(
        SetArgReferee<1>(ServiceTicket), Return(boost::system::error_code{})));
    EXPECT_CALL(*ClusterClient, async_run(_, RequestToSettingsWithHeaders, _,  _)).
        WillOnce(WithArg<3>([&](auto&& yieldCtx){boost::asio::post(IoContext, [&, yieldCtx]{
        yieldCtx(ymod_httpclient::http_error::make_error_code(yhttp::errc::success),
        Response{StatusRetryable, {}, CORRECT_RESPONSE_WITH_EMPTY_FIELDS, {}});});}));
    TestGetAuthSettings([&](auto errorCode, TAuthSettings authSettings = {}) {
        ASSERT_TRUE(errorCode);
        EXPECT_EQ(make_error_code(NNwSmtp::NSettings::EC_RETRIES_EXCEEDED), errorCode);
        EXPECT_EQ((TAuthSettings{}), authSettings);
    });
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(Tvm2Module.get()));
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(ClusterClient.get()));
}

TEST_F(TTestSettingsClient, for_incorrect_response_must_return_ec_bad_response) {
    const InSequence sequence;
    EXPECT_CALL(*GetClusterClient.impl, call()).WillOnce(Return(ClusterClient));
    EXPECT_CALL(*GetTvm2Module.impl, call()).WillOnce(Return(Tvm2Module));
    EXPECT_CALL(*Tvm2Module, get_service_ticket(ServiceName, _)).WillOnce(DoAll(
        SetArgReferee<1>(ServiceTicket), Return(boost::system::error_code{})));
    EXPECT_CALL(*ClusterClient, async_run(_, RequestToSettingsWithHeaders, _, _)).
        WillOnce(WithArg<3>([&](auto&& yieldCtx){boost::asio::post(IoContext, [&, yieldCtx]{
        yieldCtx(ymod_httpclient::http_error::make_error_code(yhttp::errc::success),
        Response{StatusOk, {}, INCORRECT_RESPONSE, {}});});}));
    TestGetAuthSettings([&](auto errorCode, TAuthSettings authSettings = {}) {
        ASSERT_TRUE(errorCode);
        EXPECT_EQ(make_error_code(NNwSmtp::NSettings::EC_BAD_RESPONSE), errorCode);
        EXPECT_EQ((TAuthSettings{}), authSettings);
    });
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(Tvm2Module.get()));
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(ClusterClient.get()));
}

class TTestSettingsClientWithPairParam
    : public TTestSettingsClient
    , public WithParamInterface<std::pair<std::string, TAuthSettings>>
{
};

TEST_P(TTestSettingsClientWithPairParam,
    for_correct_responses_of_different_contents_must_return_corresponding_values)
{
    const InSequence sequence;
    EXPECT_CALL(*GetClusterClient.impl, call()).WillOnce(Return(ClusterClient));
    EXPECT_CALL(*GetTvm2Module.impl, call()).WillOnce(Return(Tvm2Module));
    EXPECT_CALL(*Tvm2Module, get_service_ticket(ServiceName, _)).WillOnce(DoAll(
        SetArgReferee<1>(ServiceTicket), Return(boost::system::error_code{})));
    const auto param{GetParam()};
    EXPECT_CALL(*ClusterClient, async_run(_, RequestToSettingsWithHeaders, _, _)).
        WillOnce(WithArg<3>([&](auto&& yieldCtx){boost::asio::post(IoContext, [&, yieldCtx]{
        yieldCtx(ymod_httpclient::http_error::make_error_code(yhttp::errc::success),
        Response{StatusOk, {}, param.first, {}});});}));
    TestGetAuthSettings([&](auto errorCode, TAuthSettings authSettings = {}) {
        ASSERT_FALSE(errorCode);
        EXPECT_EQ(param.second, authSettings);
    });
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(Tvm2Module.get()));
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(ClusterClient.get()));
}

INSTANTIATE_TEST_SUITE_P(UseCorrectResponsesOfDifferentContents, TTestSettingsClientWithPairParam, Values (
    std::make_pair(CORRECT_RESPONSE_WITH_EMPTY_FIELDS, TAuthSettings{}),
    std::make_pair(CORRECT_RESPONSE_WITH_MISSING_FIELDS, TAuthSettings{}),
    std::make_pair(CORRECT_RESPONSE_WITH_NONEMPTY_FIELDS, TAuthSettings{.EnablePop = true,
        .EnableImap = true, .EnableImapAuthPlain = true})
));

}
