#include "init_log.h"

#include <mail/nwsmtp/src/blackbox/bb_checks_impl.cpp>
#include <mail/nwsmtp/src/settings_authorization/settings_authorization_impl.h>

#include <mail/nwsmtp/ut/util/check_bb_mocks.h>
#include <mail/nwsmtp/ut/util/settings_client_mock.h>

namespace NNwSmtp {

static inline bool operator==(const TAuthData& left, const TAuthData& right) {
    return std::tie(left.Login, left.Password, left.Token, left.Ip, left.Port, left.Method) ==
        std::tie(right.Login, right.Password, right.Token, right.Ip, right.Port, right.Method);
}

static inline bool operator==(const TContext& left, const TContext& right) {
    return
        std::tie(left.GetConnectionId(), left.GetEnvelopeId(), left.GetClusterName(), left.GetHostName()) ==
        std::tie(right.GetConnectionId(), right.GetEnvelopeId(), right.GetClusterName(), right.GetHostName());
}

}

namespace NNwSmtp::NBlackBox {

static inline bool operator==(const TResponse& left, const TResponse& right) {
    return left.Login == right.Login &&
        left.Suid == right.Suid &&
        left.Uid == right.Uid &&
        left.DefaultEmail == right.DefaultEmail &&
        left.Hosted == right.Hosted &&
        left.CatchAll == right.CatchAll &&
        left.Domain == right.Domain &&
        left.Country == right.Country &&
        left.OrgId == right.OrgId &&
        left.PhoneConfirmed == right.PhoneConfirmed &&
        left.IsCorpList == right.IsCorpList &&
        left.IsMailList == right.IsMailList &&
        left.IsBlocked == right.IsBlocked &&
        left.IsAssessor == right.IsAssessor &&
        left.HasPDDsid == right.HasPDDsid &&
        left.AppPasswordEnabled == right.AppPasswordEnabled &&
        left.RegistrationDate == right.RegistrationDate &&
        left.BornDate == right.BornDate &&
        left.Mdb == right.Mdb &&
        left.Karma == right.Karma &&
        left.KarmaStatus == right.KarmaStatus &&
        left.KarmaBanTime == right.KarmaBanTime &&
        left.Scopes == right.Scopes &&
        left.AuthSuccess == right.AuthSuccess &&
        left.ErrorStr == right.ErrorStr;
}

}

namespace {

using namespace testing;
namespace NBlackBox = NNwSmtp::NBlackBox;

using NBlackBox::EError;
using NNwSmtp::TErrorCode;
namespace NAuth = NNwSmtp::NAuth;
using NNwSmtp::AuthAllowedBySettings;
using NNwSmtp::check;
using NNwSmtp::NSettings::make_error_code;
using NNwSmtp::NSettings::TAuthSettings;
using NNwSmtp::NSettings::TUid;
using NNwSmtp::Options;
using NNwSmtp::TContext;
using NNwSmtp::TContextPtr;
using NTesting::TSettingsClientMock;
using NNwSmtp::NBlackBox::EError;

using namespace NNwSmtp::NBlackBox::NTest;

using TSettingsAuthorization = NNwSmtp::TSettingsAuthorization<TSettingsClientMock>;
using TCallback = TSettingsAuthorization::TCallback;

struct TTestSettingsAuthorization : Test {
    void SetUp() override {
        NTesting::InitGlobalLog();
    }

    void MakeSettingsAuthorization(Options::AuthSettingsOpts authSettingsOpts, TCallback callback)
    {
        SettingsAuthorization = std::make_shared<TSettingsAuthorization>(IoContext, Context, authSettingsOpts,
            BBChecksMock, SettingsClientMock, std::move(callback));
    }

    void TestSettingsAuthorization(std::int32_t expectedCallbackCallCount = 1)
    {
        SettingsAuthorization->Start(AuthData);
        IoContext.run();
        EXPECT_EQ(expectedCallbackCallCount, CallbackCallCount);
    }

    boost::asio::io_context IoContext;
    const std::string ConnectionId{"ConnectionId"};
    const TContextPtr Context{boost::make_shared<TContext>(ConnectionId, std::string{}, std::string{}, std::string{})};
    const TUid Uid{"1009"};
    const std::shared_ptr<StrictMock<TBBChecksMock>> BBChecksMock{
        std::make_shared<StrictMock<TBBChecksMock>>()};
    const std::shared_ptr<StrictMock<TSettingsClientMock>> SettingsClientMock{
        std::make_shared<StrictMock<TSettingsClientMock>>()};
    std::shared_ptr<TSettingsAuthorization> SettingsAuthorization;
    NNwSmtp::TAuthData AuthData{{"login"}, {"password"}, {"token"}, {"ip"}, {"port"}, {"method"}};
    std::int32_t CallbackCallCount{0};
};

TEST_F(TTestSettingsAuthorization, AuthAllowedForPopEnabled) {
    TAuthSettings authSettings;
    authSettings.EnablePop = true;
    EXPECT_TRUE(AuthAllowedBySettings(Context, authSettings, {}, {}));
}

TEST_F(TTestSettingsAuthorization, AuthDisallowedForPopAndImapDisabled) {
    EXPECT_FALSE(AuthAllowedBySettings(Context, {}, {}, {}));
}

TEST_F(TTestSettingsAuthorization,
    AuthAllowedForAuthPlainSettingCheckingEnabledImapAuthPlainDisabledAuthPlainRequestedAndAppPasswordEnabled)
{
    TAuthSettings authSettings;
    authSettings.EnableImap = true;
    const auto appPasswordEnabled{true};
    EXPECT_TRUE(AuthAllowedBySettings(Context, authSettings, NAuth::EAuthMethod::PLAIN,
        appPasswordEnabled));
    EXPECT_TRUE(AuthAllowedBySettings(Context, authSettings, NAuth::EAuthMethod::LOGIN,
        appPasswordEnabled));
}

TEST_F(TTestSettingsAuthorization,
    AuthAllowedForAuthPlainSettingCheckingEnabledImapAuthPlainDisabledAndXoauth2Requested)
{
    TAuthSettings authSettings;
    authSettings.EnableImap = true;
    EXPECT_TRUE(AuthAllowedBySettings(Context, authSettings, NAuth::EAuthMethod::XOAUTH2, {}));
}

TEST_F(TTestSettingsAuthorization,
    AuthAllowedForAuthPlainSettingCheckingEnabledImapAuthPlainEnabledAndAuthPlainRequested)
{
    TAuthSettings authSettings;
    authSettings.EnableImap = true;
    authSettings.EnableImapAuthPlain = true;
    EXPECT_TRUE(AuthAllowedBySettings(Context, authSettings, NAuth::EAuthMethod::PLAIN, {}));
    EXPECT_TRUE(AuthAllowedBySettings(Context, authSettings, NAuth::EAuthMethod::LOGIN, {}));
}

struct TTestSettingsAuthorizationWithChkStatusParam
    : public TTestSettingsAuthorization
    , public WithParamInterface<TErrorCode>
{
};

TEST_P(TTestSettingsAuthorizationWithChkStatusParam, PassForNonChkAccept) {
    const Options::AuthSettingsOpts authSettingsOpts;
    const NBlackBox::TResponse expectedResponse;
    EXPECT_CALL(*BBChecksMock, CheckAuth(Pointee(*Context), AuthData, _)).WillOnce(WithArg<2>([&](auto&& callback) {
        boost::asio::post(IoContext, [&, localCallback = std::move(callback)] {
            localCallback(GetParam(), expectedResponse);
        });
    }));
    MakeSettingsAuthorization(authSettingsOpts, [&](auto ec, auto response){
        ++CallbackCallCount;
        EXPECT_EQ(GetParam(), ec);
        EXPECT_EQ(expectedResponse, response);
    });

    TestSettingsAuthorization();
}

INSTANTIATE_TEST_SUITE_P(ErrorCode, TTestSettingsAuthorizationWithChkStatusParam,
    Values(EError::Ok,
           EError::BbTempUserError,
           EError::UserNotFound,
           EError::UserBlocked,
           EError::TempBanUser,
           EError::BadKarma,
           EError::EmptySender,
           EError::EmptyAuthData,
           EError::ForbiddenForAssessors,
           EError::Mdbreg,
           EError::NoSuid,
           EError::NoPddeula,
           EError::NotFoundSmtpScope,
           EError::NoAccessRights));

TEST_F(TTestSettingsAuthorization, PassForSettingsError) {
    Options::AuthSettingsOpts authSettingsOpts;
    authSettingsOpts.use = true;
    const auto expectedErrorCode = EError::Ok;
    const NBlackBox::TResponse expectedResponse = {.Uid = Uid};
    const InSequence sequence;
    EXPECT_CALL(*BBChecksMock, CheckAuth(Pointee(*Context), AuthData, _)).WillOnce(WithArg<2>([&](auto&& callback) {
        boost::asio::post(IoContext, [&, localCallback = std::move(callback)] {
            localCallback(expectedErrorCode, expectedResponse);
        });
    }));

    EXPECT_CALL(*SettingsClientMock, GetAuthSettings(Pointee(*Context), Uid, _, _)).
        WillOnce(WithArg<3>([&](auto&& callback)
    {
        boost::asio::post(IoContext, [localCallback = std::move(callback)] {
            localCallback(make_error_code(NNwSmtp::NSettings::EC_BAD_RESPONSE), {});
        });
    }));

    MakeSettingsAuthorization(authSettingsOpts, [&](auto ec, auto response){
        ++CallbackCallCount;
        EXPECT_EQ(expectedErrorCode, ec);
        EXPECT_EQ(expectedResponse, response);
    });

    TestSettingsAuthorization();
}

TEST_F(TTestSettingsAuthorization, RejectWhenAuthAllowedBySettingsReturnsFalse) {
    Options::AuthSettingsOpts authSettingsOpts;
    authSettingsOpts.use = true;
    AuthData.Method = "PLAIN";
    const auto expectedErrorCode = EError::NoAccessRights;
    const NBlackBox::TResponse expectedResponse = {
        .Uid = Uid,
        .ErrorStr = "This user does not have access rights to this service"
    };
    const InSequence sequence;
    EXPECT_CALL(*BBChecksMock, CheckAuth(Pointee(*Context), AuthData, _)).WillOnce(WithArg<2>([&](auto&& callback) {
        boost::asio::post(IoContext, [&, localCallback = std::move(callback)] {
            localCallback(EError::Ok, expectedResponse);
        });
    }));

    EXPECT_CALL(*SettingsClientMock, GetAuthSettings(Pointee(*Context), Uid, _, _)).
        WillOnce(WithArg<3>([&](auto&& callback)
    {
        boost::asio::post(IoContext, [localCallback = std::move(callback)] {
            const auto enableImap = true;
            localCallback({}, {{}, enableImap, {}});
        });
    }));

    MakeSettingsAuthorization(authSettingsOpts, [&](auto ec, auto response){
        ++CallbackCallCount;
        EXPECT_EQ(expectedErrorCode, ec);
        EXPECT_EQ(expectedResponse, response);
    });

    TestSettingsAuthorization();
}

TEST_F(TTestSettingsAuthorization, PassWhenAuthAllowedBySettingsReturnsTrue) {
    Options::AuthSettingsOpts authSettingsOpts;
    authSettingsOpts.use = true;
    AuthData.Method = "PLAIN";
    const auto expectedErrorCode = EError::Ok;
    const NBlackBox::TResponse expectedResponse = {.Uid = Uid};
    const InSequence sequence;
    EXPECT_CALL(*BBChecksMock, CheckAuth(Pointee(*Context), AuthData, _)).WillOnce(WithArg<2>([&](auto&& callback) {
        boost::asio::post(IoContext, [&, localCallback = std::move(callback)] {
            localCallback(expectedErrorCode, expectedResponse);
        });
    }));

    EXPECT_CALL(*SettingsClientMock, GetAuthSettings(Pointee(*Context), Uid, _, _)).
        WillOnce(WithArg<3>([&](auto&& callback)
    {
        boost::asio::post(IoContext, [localCallback = std::move(callback)] {
            const auto enablePop = true;
            localCallback({}, {enablePop, {}, {}});
        });
    }));

    MakeSettingsAuthorization(authSettingsOpts, [&](auto ec, auto response){
        ++CallbackCallCount;
        EXPECT_EQ(expectedErrorCode, ec);
        EXPECT_EQ(expectedResponse, response);
    });

    TestSettingsAuthorization();
}

}
