#include <mail/nwsmtp/src/smtp_client/smtp_targeting.h>

#include <memory>
#include <string>
#include <tuple>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <ymod_smtpclient/errors.h>
#include <ymod_smtpclient/smtp_point.h>
#include <ymod_smtpclient/response.h>

#include <mail/ymod_smtpclient/tests/mocks/smtp_session_mock.h>

namespace {

using namespace testing;
using namespace NNwSmtp::SmtpClient::NSmtpTargeting;

class SmtpTargetingTest : public Test {
public:
    using SmtpSessionMock = StrictMock<SmtpSessionMock>;

    std::shared_ptr<SmtpSessionMock> smtpSession = std::make_shared<SmtpSessionMock>();
};

TEST_F(SmtpTargetingTest, OnConnectFailureReturnErrorCode) {
    const auto connectErrorCode = ymod_smtpclient::error::Unknown;

    EXPECT_CALL(*smtpSession, asyncConnect(_, _, _, _)).WillOnce(InvokeArgument<3>(connectErrorCode));

    AsyncTargeting(smtpSession, ymod_smtpclient::SmtpPoint{}, [=](auto ec, auto host) {
        EXPECT_EQ(connectErrorCode, ec);
        EXPECT_TRUE(host.empty());
    });
}

TEST_F(SmtpTargetingTest, OnGreetingFailureReturnErrorCode) {
    const auto greetingErrorCode = ymod_smtpclient::error::Unknown;
    const ymod_smtpclient::MultiLineResponse response;

    EXPECT_CALL(*smtpSession, asyncConnect(_, _, _, _)).WillOnce(InvokeArgument<3>(ymod_smtpclient::error::Success));
    EXPECT_CALL(*smtpSession, asyncGreeting(_)).WillOnce(InvokeArgument<0>(greetingErrorCode, response));

    AsyncTargeting(smtpSession, ymod_smtpclient::SmtpPoint{}, [=](auto ec, auto host) {
        EXPECT_EQ(greetingErrorCode, ec);
        EXPECT_TRUE(host.empty());
    });
}

TEST_F(SmtpTargetingTest, OnMissingGreetingLineReturnErrorCode) {
    const auto expectedCode = ymod_smtpclient::error::ParseResponseError;
    const ymod_smtpclient::MultiLineResponse response;

    EXPECT_CALL(*smtpSession, asyncConnect(_, _, _, _)).WillOnce(InvokeArgument<3>(ymod_smtpclient::error::Success));
    EXPECT_CALL(*smtpSession, asyncGreeting(_)).WillOnce(InvokeArgument<0>(ymod_smtpclient::error::Success, response));
    EXPECT_CALL(*smtpSession, asyncQuit(_));

    AsyncTargeting(smtpSession, ymod_smtpclient::SmtpPoint{}, [=](auto ec, auto host) {
        EXPECT_EQ(expectedCode, ec);
        EXPECT_TRUE(host.empty());
    });
}

TEST_F(SmtpTargetingTest, OnEmptyGreetingLineReturnErrorCode) {
    const auto expectedCode = ymod_smtpclient::error::ParseResponseError;
    const ymod_smtpclient::MultiLineResponse response {0, boost::none, {""}};

    EXPECT_CALL(*smtpSession, asyncConnect(_, _, _, _)).WillOnce(InvokeArgument<3>(ymod_smtpclient::error::Success));
    EXPECT_CALL(*smtpSession, asyncGreeting(_)).WillOnce(InvokeArgument<0>(ymod_smtpclient::error::Success, response));
    EXPECT_CALL(*smtpSession, asyncQuit(_));

    AsyncTargeting(smtpSession, ymod_smtpclient::SmtpPoint{}, [=](auto ec, auto host) {
        EXPECT_EQ(expectedCode, ec);
        EXPECT_TRUE(host.empty());
    });
}

TEST_F(SmtpTargetingTest, OnNonEmptyGreetingLineDataExtractHostnameAndReturnSuccess) {
    const auto expectedCode = ymod_smtpclient::error::Success;
    const auto expectedHost = "some-host";
    const ymod_smtpclient::MultiLineResponse response {0, boost::none, {"some-host other data"}};

    EXPECT_CALL(*smtpSession, asyncConnect(_, _, _, _)).WillOnce(InvokeArgument<3>(ymod_smtpclient::error::Success));
    EXPECT_CALL(*smtpSession, asyncGreeting(_)).WillOnce(InvokeArgument<0>(ymod_smtpclient::error::Success, response));
    EXPECT_CALL(*smtpSession, asyncQuit(_));

    AsyncTargeting(smtpSession, ymod_smtpclient::SmtpPoint{}, [=](auto ec, auto host) {
        EXPECT_EQ(expectedCode, ec);
        EXPECT_EQ(expectedHost, host);
    });
}

} // namespace
