#include <mail/nwsmtp/src/so/client_impl.h>
#include <mail/nwsmtp/src/so/errors.h>
#include <mail/nwsmtp/src/so/request.h>
#include <mail/nwsmtp/src/so/to_protobuf.h>
#include <mail/nwsmtp/src/so/types.h>
#include <mail/nwsmtp/src/so/types_reflection.h>
#include <mail/nwsmtp/src/utils.h>

#include <mail/nwsmtp/ut/util/http_client_mock.h>
#include <mail/nwsmtp/ut/test_with_spawn.h>

#include <mail/so/api/so_api.pb.h>

#include <library/cpp/resource/resource.h>

#include <yplatform/encoding/base64.h>

#include <contrib/libs/protobuf/src/google/protobuf/util/json_util.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <boost/asio.hpp>

#include <optional>
#include <memory>

namespace NNwSmtp::NSO {

static bool operator ==(const TPersonalResolution& lhs, const TPersonalResolution& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

static bool operator ==(const TActivityInfo& lhs, const TActivityInfo& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

static bool operator ==(const TOutParameters& lhs, const TOutParameters& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

static bool operator ==(const TResponse& lhs, const TResponse& rhs) {
    return boost::fusion::operator==(lhs, rhs);
}

} // namespace NNwSmtp::NSO

namespace {

using namespace testing;
using namespace NNwSmtp;
using namespace NNwSmtp::NSO;
using namespace NNwSmtp::NTests;

using THttpRequest = yhttp::request;
using THttpResponse = yhttp::response;

TRequestPtr MakeRequest() {
    return std::make_shared<TRequest>(
        TEnvelope {
            .ConnectInfo = {
                .Host = "host",
                .Ip = std::string("\xc0\x00\x02\xeb", 4),
                .Domain = "domain",
                .SessionId = "session_id"
            },
            .MailFrom = std::make_optional(
                TEmailInfo {
                    .Address = {.Email = "email@email.ru"},
                    .Uid = "228"
                }
            ),
            .Rcpts = {
                {
                    .Address = {.Email = "rspt1@email.ru"},
                    .Uid = std::make_optional<std::string>("11"),
                    .Suid = std::make_optional<std::string>("12"),
                    .Country = std::make_optional<std::string>("country1"),
                    .Karma = std::make_optional<std::string>("13"),
                    .KarmaStatus = std::make_optional<std::string>("14"),
                    .IsMaillist = true
                },
                {
                    .Address = {.Email = "rspt2@email.ru"},
                    .Uid = std::make_optional<std::string>("21"),
                    .Suid = std::make_optional<std::string>("22"),
                    .Country = std::make_optional<std::string>("country2"),
                    .Karma = std::make_optional<std::string>("23"),
                    .KarmaStatus = std::make_optional<std::string>("24"),
                }
            },
            .Timestamp = "12345678",
        },
        "body",
        true
    );
}

TResponse MakeResponse() {
    return {
        .Resolution = EResolution::SO_RESOLUTION_ACCEPT,
        .DenyGraylist = true,
        .OutParameters = TOutParameters{},
        .ActivityInfos = std::vector<TActivityInfo>{
            {.Uid = "996694137"}
        },
        .SoClasses = {"people", "trust_6", "correspond"},
        .PersonalResolutions = std::vector<NSO::TPersonalResolution> {
            {
                .Uid = "228",
                .Resolution = EResolution::SO_RESOLUTION_SPAM,
                .SoClasses = {"people", "trust_6", "correspond", "pf_spam"}
            }
        },
    };
}

struct TTestSOHttpClient: TTestWithSpawn {
    const std::shared_ptr<StrictMock<TClusterClientMock>> ClusterClient = GetStrictMockedClusterClient();
    const std::shared_ptr<TSOClient> SOClient = std::make_shared<TSOClient>(ClusterClient,  NSO::ESOType::SOOut, Io);

    const TRequestPtr Request = MakeRequest();

    TContextPtr Context = boost::make_shared<TContext>("","","","");

    const THttpRequest CheckRequest = THttpRequest::POST(
        "/v3/antispam?format=protobuf&output-format=protobuf-json&session_id=session_id",
        SoRequestToBinary(
            R"({"smtp_envelope":{"connect_info":{"remote_host":"host","remote_ip":"wAAC6w==","remote_domain":"domain")"
            R"(,"session_id":"session_id"},"email_type":"EMAIL_TYPE_REGULAR","mail_from":)"
            R"({"address":{"email":"email@email.ru"},"uid":"228","is_maillist":false},)"
            R"("recipients":[{"address":{"email":"rspt1@email.ru"},"uid":"11","suid":"12","country":"country1",)"
            R"("karma":"13","karma_status":"14","is_maillist":true},{"address":{"email":"rspt2@email.ru"},)"
            R"("uid":"21","suid":"22","country":"country2","karma":"23","karma_status":"24","is_maillist":false}],)"
            R"("email_timestamp":"12345678"},"raw_email":"Ym9keQ==","dry_run":true,"get_delivery_log":false})")
    );

    std::string CheckResponse =
        R"({"resolution": "SO_RESOLUTION_ACCEPT", "out_parameters":{}, "deny_graylist": true, "so_classes":)"
        R"(["people", "trust_6", "correspond"], "activity_infos":[{"uid":"996694137"}], "personal_resolutions":)"
        R"([{"uid": "228", "resolution": "SO_RESOLUTION_SPAM", "so_classes":)"
        R"(["people", "trust_6", "correspond", "pf_spam"]}]})";

    static std::string SoRequestToBinary(const std::string& str) {
        mail::so::api::v1::SoRequest request;
        google::protobuf::util::Status status = google::protobuf::util::JsonStringToMessage(TProtoStringType{str}, &request);
        EXPECT_EQ(status.ok(), true);
        return request.SerializeAsStringOrThrow();
    }

    template <typename THandler>
    TResponse Check(
        THandler handler
    ) {
        boost::asio::async_completion<
            THandler,
            void(TErrorCode, TResponse)
        > init(handler);
        SOClient->Check(Context, Request, init.completion_handler);
        return init.result.get();
    }
};

TEST_F(TTestSOHttpClient, for_request_ended_with_error_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        EXPECT_CALL(*ClusterClient, async_run(_, CheckRequest, _, _))
            .WillOnce(InvokeArgument<3>(ymod_httpclient::http_error::code::ssl_error, THttpResponse{}));

        TErrorCode ec;
        auto result = Check(yield[ec]);
        ASSERT_TRUE(ec);
        EXPECT_EQ(ec, ymod_httpclient::http_error::code::ssl_error);
    });
}

TEST_F(TTestSOHttpClient, for_request_ended_with_500_code_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        EXPECT_CALL(*ClusterClient, async_run(_, CheckRequest, _, _))
            .WillOnce(InvokeArgument<3>(TErrorCode {}, THttpResponse{500, {}, {}, {}}));

        TErrorCode ec;
        auto result = Check(yield[ec]);
        ASSERT_TRUE(ec);
        EXPECT_EQ(ec, EError::BadStatus);
    });
}

TEST_F(TTestSOHttpClient, for_failed_parsing_response_should_return_error) {
    WithSpawn([this](boost::asio::yield_context yield) {
        EXPECT_CALL(*ClusterClient, async_run(_, CheckRequest, _, _))
            .WillOnce(InvokeArgument<3>(TErrorCode {}, THttpResponse{200, {}, "", {}}));

        TErrorCode ec;
        auto result = Check(yield[ec]);
        ASSERT_TRUE(ec);
        EXPECT_EQ(ec, EError::ParseError);
    });
}

TEST_F(TTestSOHttpClient, for_response_should_return_so_response) {
    WithSpawn([this](boost::asio::yield_context yield) {
        EXPECT_CALL(*ClusterClient, async_run(_, CheckRequest, _, _))
            .WillOnce(InvokeArgument<3>(TErrorCode {}, THttpResponse {200, {}, CheckResponse, {}}));

        TErrorCode ec;
        auto result = Check(yield[ec]);
        ASSERT_FALSE(ec);
        EXPECT_EQ(result, MakeResponse());

    });
}

TEST_F(TTestSOHttpClient, validate_binary_serialization) {
    EXPECT_EQ(ToProtobuf(*MakeRequest()), NResource::Find("simple_so_request.bin"));
}

} // namespace anonymous
