#include "spf_check_mocks.h"
#include "test_with_spawn.h"

#include <mail/nwsmtp/src/spf/aspf.h>
#include <mail/nwsmtp/src/spf/config.h>
#include <mail/nwsmtp/src/spf/spf_check_impl.h>
#include <mail/nwsmtp/src/types.h>

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <functional>
#include <memory>

namespace {

using namespace testing;
using namespace NNwSmtp;
using namespace NNwSmtp::NSPF;
using namespace NNwSmtp::NSPF::NTest;

auto MakeConfig() {
    return std::make_shared<TConfig>(
        resolver_options {},
        NNwSmtp::Options::SPFOpts {true, 0}
    );
}

struct TTestSPFChecks: TTestWithSpawn {
    TTestSPFChecks() {
        SPFClientMockHolder.Mock = &SPFClientMock;
    }

    StrictMock<TSPFClientMock> SPFClientMock;
    TSPFCheckClient<TSPFClientMockImpl> SPFCheckClient
        = TSPFCheckClient<TSPFClientMockImpl> {MakeConfig(), Io};

    template <typename THandler>
    TResponse CallSPFCheck(
        std::string from,
        std::string domain,
        THandler handler
    ) {
        boost::asio::async_completion<
            THandler,
            void(TErrorCode, TResponse)
        > init(handler);

        SPFCheckClient.Check(
            from,
            domain,
            boost::asio::ip::make_address("127.0.0.1"),
            [](const std::string&) {},
            init.completion_handler
        );
        return init.result.get();
    }
};

TEST_F(TTestSPFChecks, for_timeout_spf_check_should_stop_client_end_return_empty_result) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;

        TErrorCode ec;
        auto params = spf_parameters {
            boost::asio::ip::make_address("127.0.0.1"),
            "pig.com",
            "peppa@pig.com",
            [](const std::string&) {}
        };

        EXPECT_CALL(SPFClientMock, start(_, params, _, _, _));
        EXPECT_CALL(SPFClientMock, stop());
        auto response = CallSPFCheck("peppa@pig.com", "pig.com", yield[ec]);

        ASSERT_FALSE(ec);
        ASSERT_FALSE(response.Result);
        ASSERT_FALSE(response.Expl);
    });
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(&SPFClientMock));
}

TEST_F(TTestSPFChecks, for_success_spf_check_should_return_result) {
    WithSpawn([this](boost::asio::yield_context yield) {
        const InSequence s;

        TErrorCode ec;
        auto params = spf_parameters {
            boost::asio::ip::make_address("127.0.0.1"),
            "pig.com",
            "peppa@pig.com",
            [](const std::string&) {}
        };

        EXPECT_CALL(SPFClientMock, start(_, params, _, _, _))
            .WillOnce(InvokeArgument<3>(
                std::optional<std::string> {"result_one"},
                std::optional<std::string> {"result_two"})
            );
        auto response = CallSPFCheck("peppa@pig.com", "pig.com", yield[ec]);

        ASSERT_FALSE(ec);
        ASSERT_TRUE(response.Result);
        ASSERT_TRUE(response.Expl);
        ASSERT_EQ(*response.Result, "result_one");
        ASSERT_EQ(*response.Expl, "result_two");
    });
    EXPECT_TRUE(Mock::VerifyAndClearExpectations(&SPFClientMock));
}

} // namespace anonymous
