#include <mail/nwsmtp/src/spf/aspf.h>
#include <mail/nwsmtp/src/utils.h>

#include <nwsmtp/resolver.h>

#include <mail/libyadns/include/yadns/resolver_options.h>

#include <gtest/gtest.h>

#include <optional>


struct TSpfCheck : public testing::Test {
    using TResult = std::pair<std::optional<std::string>, std::optional<std::string>>;

    TResult Run(std::string ip, std::string mailFromDomain, std::string heloDomain) {
        spf_parameters params;

        params.ip = boost::asio::ip::address::from_string(ip);
        params.domain = heloDomain;
        params.from = mailFromDomain;

        boost::asio::io_service ioService;
        TResult result;
        boost::make_shared<spf_check>()->start(ioService, params, {},
            [&result](auto resolution, auto explanation) {
                result = {resolution, explanation};
            });
        ioService.run();
        return result;
    }
};

const std::string ALLOWED_IPV4_ADDRESS = "213.180.223.192";
const std::string ALLOWED_IPV6_ADDRESS = "2a02:6b8:0:1a2d:428d:5cff:fe34:f9f0";


TEST_F(TSpfCheck, LocalhostPass) {
    auto res = Run("127.0.0.1", "yandex.ru", "yandex.ru");
    ASSERT_TRUE(res.first && res.second);
    ASSERT_EQ(*res.first, "pass");
    ASSERT_TRUE(Contains(*res.second, ": localhost is always allowed, rule=[]"));
}

TEST_F(TSpfCheck, AllowedIpv4Pass) {
    auto res = Run(ALLOWED_IPV4_ADDRESS, "yandex.ru", "yandex.ru");
    ASSERT_TRUE(res.first && res.second);
    ASSERT_EQ(*res.first, "pass");
    ASSERT_TRUE(Contains(
        *res.second,
        ": domain of yandex.ru designates 213.180.223.192 as permitted sender, rule=[ip4:213.180.223.192/26]"));
}

TEST_F(TSpfCheck, NoHeloDomainInvalid) {
    auto res = Run(ALLOWED_IPV4_ADDRESS, "yandex.ru", "");
    ASSERT_TRUE(res.first);
    ASSERT_EQ(*res.first, "(invalid)");
}

TEST_F(TSpfCheck, NoMailFromAndHeloDomains) {
    auto res = Run(ALLOWED_IPV4_ADDRESS, "", "");
    ASSERT_FALSE(res.first);
    ASSERT_FALSE(res.second);
}

TEST_F(TSpfCheck, AllowedIpv4NoMailFromDomainPass) {
    auto res = Run(ALLOWED_IPV4_ADDRESS, "", "yandex.ru");
    ASSERT_TRUE(res.first);
    ASSERT_EQ(*res.first, "pass");
}

TEST_F(TSpfCheck, AllowedIpv6Pass) {
    auto res = Run(ALLOWED_IPV6_ADDRESS, "yandex.ru", "yandex.ru");
    ASSERT_TRUE(res.first && res.second);
    ASSERT_EQ(*res.first, "pass");
    ASSERT_TRUE(Contains(
        *res.second,
        ": domain of yandex.ru designates 2a02:6b8:0:1a2d:428d:5cff:fe34:f9f0 as permitted sender, rule=[ip6:2a02:6b8:0:1a2d::/64]"));
}

TEST_F(TSpfCheck, NotAllowedSoftFail) {
    auto res = Run("192.168.0.1", "yandex.ru", "yandex.ru");
    ASSERT_TRUE(res.first && res.second);
    ASSERT_EQ(*res.first, "softfail");
    ASSERT_TRUE(Contains(
        *res.second,
        ": transitioning domain of yandex.ru does not designate 192.168.0.1 as permitted sender, rule=[~all]"));
}

TEST_F(TSpfCheck, NoSpfRecordsAvailable) {
    auto res = Run(ALLOWED_IPV4_ADDRESS, "mail.yandex.net", "mail.yandex.net");
    ASSERT_TRUE(res.first);
    ASSERT_EQ(*res.first, "(invalid)");
}
