#pragma once

#include <ymod_httpclient/call.h>
#include <ymod_httpclient/cluster_client.h>

#include <gmock/gmock.h>
#include <gtest/gtest.h>

namespace ymod_httpclient {

static inline bool operator==(const request& left, const request& right) {
    return (left.method == right.method) && (left.url == right.url) && (left.headers == right.headers) &&
        ((left.body && right.body && *left.body == *right.body) || (!left.body && !right.body));
}

static inline bool operator==(const timeouts& left, const timeouts& right) {
    return (left.connect == right.connect) && (left.total == right.total);
}

static inline bool operator==(const options& left, const options& right) {
    return (left.log_post_body == right.log_post_body) && (left.log_headers == right.log_headers) &&
        (left.reuse_connection == right.reuse_connection) && (left.timeouts == right.timeouts);
}

static inline std::ostream& operator<<(std::ostream& stream, const request& value) {
    return stream << "yhttp::request {" << static_cast<int>(value.method) << ", " << value.url << ", " <<
        value.headers << ", " << (value.body ? *value.body : std::string{}) << "}";
}

static inline std::ostream& operator<<(std::ostream& stream, const timeouts& value) {
    return stream << "ymod_httpclient::timeouts {" << value.connect.count() << ", " <<
        value.total.count() << "}";
}

static inline std::ostream& operator<<(std::ostream& stream, const options& value) {
    using namespace std::string_literals;
    return stream << "yhttp::options {" << (value.log_post_body ? std::to_string(*value.log_post_body) :
        "none"s) << ", " << (value.log_headers ? std::to_string(*value.log_headers) : "none"s) << ", " <<
        (value.reuse_connection ? std::to_string(*value.reuse_connection) : "none"s) << ", " <<
        value.timeouts << "}";
}

} // namespace ymod_httpclient

namespace NNwSmtp::NTests {

using namespace testing;

struct TGetHttpClientMock {
    MOCK_METHOD(std::shared_ptr<yhttp::simple_call>, call, (), (const));
};

struct TGetHttpClientMockWrapper {
    std::shared_ptr<StrictMock<TGetHttpClientMock>> impl{std::make_shared<StrictMock<TGetHttpClientMock>>()};
    std::shared_ptr<yhttp::simple_call> operator()() const {
        return impl->call();
    }
};

struct THttpClientMock : yhttp::simple_call {
    using TTaskContextPtr = yhttp::simple_call::task_context_ptr;
    using TCallbackType = yhttp::simple_call::callback_type;
    MOCK_METHOD(yhttp::response, run, (TTaskContextPtr, yhttp::request), (override));
    MOCK_METHOD(yhttp::response, run, (TTaskContextPtr, yhttp::request, const yhttp::options&), (override));
    MOCK_METHOD(void, async_run, (TTaskContextPtr, yhttp::request, TCallbackType), (override));
    MOCK_METHOD(void, async_run, (TTaskContextPtr, yhttp::request, const yhttp::options&, TCallbackType), (override));
};

struct TGetClusterClientMock {
    MOCK_METHOD(std::shared_ptr<ymod_httpclient::cluster_call>, call, (), (const));
};

struct TGetClusterClientMockWrapper {
    std::shared_ptr<StrictMock<TGetClusterClientMock>> impl{
        std::make_shared<StrictMock<TGetClusterClientMock>>()};
    std::shared_ptr<ymod_httpclient::cluster_call> operator()() const {
        return impl->call();
    }
};

struct TClusterClientMock : ymod_httpclient::cluster_call {
    using TTaskContextPtr = yhttp::simple_call::task_context_ptr;
    using TCallbackType = yhttp::simple_call::callback_type;

    MOCK_METHOD(void, async_run, (TTaskContextPtr, yhttp::request, TCallbackType), (override));
    MOCK_METHOD(void, async_run, (TTaskContextPtr, yhttp::request,
        const ymod_httpclient::cluster_call::options&, TCallbackType), (override));
};

inline std::shared_ptr<StrictMock<TClusterClientMock>> GetStrictMockedClusterClient() {
    return std::make_shared<StrictMock<TClusterClientMock>>();
}

} // namespace NNwSmtp::NTests
