#include "handle_mocks.h"
#include "response_mocks.h"

#include <mail/ymod_tvm/src/tvm2/impl.h>
#include <mail/yplatform/include/yplatform/application/config/yaml_to_ptree.h>

#include <mail/nwsmtp/src/web/handle_base.h>
#include <mail/nwsmtp/src/web/tvm_guard.h>
#include <mail/nwsmtp/src/web/types.h>
#include <mail/nwsmtp/src/web/utils/exception.h>
#include <mail/nwsmtp/src/web/utils/get_header.h>

#include <memory>
#include <optional>

namespace {

using namespace testing;
using namespace tvm_guard;
using namespace NTesting;
using namespace NNwSmtp::NWeb;

using TTmvModuleImpl = ymod_tvm::tvm2::impl;

struct TTestBaseHandle : Test {
    TTestBaseHandle() {
        const std::string cfg =
            "bb_env: blackbox\n"
            "default: reject\n"
            "strong_uid_check: false\n"
            "rules:\n"
            "-   name: handler\n"
            "    paths: ['/allowed']\n"
            "    default_action: accept\n"
            "    accept_by_service: []";
        yplatform::ptree node;
        utils::config::yaml_to_ptree::convert_str(cfg, node);
        Guard = MakeTvmGuard(node, std::static_pointer_cast<TTvmModule>(Module));
        HandleMockHolder.Mock = &HandleMock;
    }

    TRequest Request = boost::make_shared<ymod_webserver::request>();

    boost::shared_ptr<StrictMock<TResponseMock>> Response = boost::make_shared<StrictMock<TResponseMock>>();

    std::shared_ptr<TTmvModuleImpl> Module;
    TTvmGuardPtr Guard;
    StrictMock<THandleMock> HandleMock;
};

TEST_F(TTestBaseHandle,
        for_request_with_not_allowed_method_should_be_added_not_allowed_code_to_response) {
    Request->method = EHttpMethod::mth_head;
    const InSequence s;

    EXPECT_CALL(*Response, set_code(EHttpCode::method_not_allowed, _));
    EXPECT_CALL(*Response, set_content_type("application/json"));
    EXPECT_CALL(*Response, result_body(_));

    auto handle = TBaseHandle<THandle>{
        Guard,
        "/allowed",
        EHttpMethod::mth_post
    };

    handle(Response, Request);
}

TEST_F(TTestBaseHandle,
        for_request_that_was_rejected_should_be_added_unauthorized_code_to_response) {
    Request->method = EHttpMethod::mth_post;
    const InSequence s;

    EXPECT_CALL(*Response, set_code(EHttpCode::unauthorized, _));
    EXPECT_CALL(*Response, set_content_type("application/json"));
    EXPECT_CALL(*Response, result_body(_));

    auto handle = TBaseHandle<THandle>{
        Guard,
        "/handle",
        EHttpMethod::mth_post
    };

    handle(Response, Request);
}

TEST_F(TTestBaseHandle, for_successful_request_should_set_ok_code) {
    Request->method = EHttpMethod::mth_post;

    EXPECT_CALL(HandleMock, Handle());

    auto handle = TBaseHandle<THandle>{
        Guard,
        "/allowed",
        EHttpMethod::mth_post
    };

    handle(Response, Request);
}

TEST_F(TTestBaseHandle, for_request_that_throw_exception_should_added_internal_server_error_code_to_response) {
    Request->method = EHttpMethod::mth_post;
    const InSequence s;

    EXPECT_CALL(HandleMock, Handle()).WillOnce(Throw(std::exception()));
    EXPECT_CALL(*Response, set_code(EHttpCode::internal_server_error, _));
    EXPECT_CALL(*Response, set_content_type("application/json"));
    EXPECT_CALL(*Response, result_body(_));

    auto handle = TBaseHandle<THandle>{
        Guard,
        "/allowed",
        EHttpMethod::mth_post
    };

    handle(Response, Request);
}

TEST_F(TTestBaseHandle,
        for_request_that_throw_exception_on_parse_body_should_added_bad_request_error_code_to_response) {
    Request->method = EHttpMethod::mth_post;
    const InSequence s;

    EXPECT_CALL(HandleMock, Handle()).WillOnce(Throw(TArgumentParseException{""}));
    EXPECT_CALL(*Response, set_code(EHttpCode::bad_request, _));
    EXPECT_CALL(*Response, set_content_type("application/json"));
    EXPECT_CALL(*Response, result_body(_));

    auto handle = TBaseHandle<THandle>{
        Guard,
        "/allowed",
        EHttpMethod::mth_post
    };

    handle(Response, Request);
}

TEST(TTestGetHeader, for_get_existing_header_should_return_optional_whith_its_content) {
    TRequest request = boost::make_shared<ymod_webserver::request>();
    request->headers = {{"header", "42"}};
    EXPECT_EQ(
        GetOptionalHeader(
            request,
            "Header"
        ),
        std::make_optional("42")
    );
}

TEST(TTestGetHeader, for_get_not_existing_header_should_return_nullopt) {
    TRequest request = boost::make_shared<ymod_webserver::request>();
    request->headers = {{"header", "42"}};
    EXPECT_EQ(
        GetOptionalHeader(
            request,
            "not_header"
        ),
        std::nullopt
    );
}

} // namespace anonymous
